/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmMakefileTargetGenerator.h"

#include <algorithm>
#include <cassert>
#include <cstdio>
#include <iterator>
#include <sstream>
#include <unordered_map>
#include <unordered_set>
#include <utility>

#include <cm/memory>
#include <cmext/algorithm>

#include "cmComputeLinkInformation.h"
#include "cmCustomCommand.h"
#include "cmCustomCommandGenerator.h"
#include "cmGeneratedFileStream.h"
#include "cmGeneratorExpression.h"
#include "cmGeneratorTarget.h"
#include "cmGlobalUnixMakefileGenerator3.h"
#include "cmLinkLineComputer.h" // IWYU pragma: keep
#include "cmLocalCommonGenerator.h"
#include "cmLocalUnixMakefileGenerator3.h"
#include "cmMakefile.h"
#include "cmMakefileExecutableTargetGenerator.h"
#include "cmMakefileLibraryTargetGenerator.h"
#include "cmMakefileUtilityTargetGenerator.h"
#include "cmMessageType.h"
#include "cmOutputConverter.h"
#include "cmPolicies.h"
#include "cmProperty.h"
#include "cmRange.h"
#include "cmRulePlaceholderExpander.h"
#include "cmSourceFile.h"
#include "cmSourceFileLocationKind.h"
#include "cmState.h"
#include "cmStateDirectory.h"
#include "cmStateSnapshot.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmake.h"

cmMakefileTargetGenerator::cmMakefileTargetGenerator(cmGeneratorTarget* target)
  : cmCommonTargetGenerator(target)
{
  CustomCommandDriver = OnBuild;
  LocalGenerator =
    static_cast<cmLocalUnixMakefileGenerator3*>(target->GetLocalGenerator());
  GlobalGenerator = static_cast<cmGlobalUnixMakefileGenerator3*>(
    LocalGenerator->GetGlobalGenerator());
  cmake* cm = GlobalGenerator->GetCMakeInstance();
  NoRuleMessages = false;
  if (cmProp ruleStatus = cm->GetState()->GetGlobalProperty("RULE_MESSAGES")) {
    NoRuleMessages = cmIsOff(*ruleStatus);
  }
  switch (GeneratorTarget->GetPolicyStatusCMP0113()) {
    case cmPolicies::WARN:
    case cmPolicies::OLD:
      CMP0113New = false;
      break;
    case cmPolicies::NEW:
    case cmPolicies::REQUIRED_IF_USED:
    case cmPolicies::REQUIRED_ALWAYS:
      CMP0113New = true;
      break;
  }
  MacOSXContentGenerator = cm::make_unique<MacOSXContentGeneratorType>(this);
}

cmMakefileTargetGenerator::~cmMakefileTargetGenerator() = default;

std::unique_ptr<cmMakefileTargetGenerator> cmMakefileTargetGenerator::New(
  cmGeneratorTarget* tgt)
{
  std::unique_ptr<cmMakefileTargetGenerator> result;

  switch (tgt->GetType()) {
    case cmStateEnums::EXECUTABLE:
      result = cm::make_unique<cmMakefileExecutableTargetGenerator>(tgt);
      break;
    case cmStateEnums::STATIC_LIBRARY:
    case cmStateEnums::SHARED_LIBRARY:
    case cmStateEnums::MODULE_LIBRARY:
    case cmStateEnums::OBJECT_LIBRARY:
      result = cm::make_unique<cmMakefileLibraryTargetGenerator>(tgt);
      break;
    case cmStateEnums::INTERFACE_LIBRARY:
    case cmStateEnums::UTILITY:
      result = cm::make_unique<cmMakefileUtilityTargetGenerator>(tgt);
      break;
    default:
      return result;
      // break; /* unreachable */
  }
  return result;
}

std::string cmMakefileTargetGenerator::GetConfigName()
{
  auto const& configNames = LocalGenerator->GetConfigNames();
  assert(configNames.size() == 1);
  return configNames.front();
}

void cmMakefileTargetGenerator::GetDeviceLinkFlags(
  std::string& linkFlags, const std::string& linkLanguage)
{
  cmGeneratorTarget::DeviceLinkSetter setter(*GetGeneratorTarget());

  std::vector<std::string> linkOpts;
  GeneratorTarget->GetLinkOptions(linkOpts, GetConfigName(), linkLanguage);
  // LINK_OPTIONS are escaped.
  LocalGenerator->AppendCompileOptions(linkFlags, linkOpts);
}

void cmMakefileTargetGenerator::GetTargetLinkFlags(
  std::string& flags, const std::string& linkLanguage)
{
  LocalGenerator->AppendFlags(flags,
                              GeneratorTarget->GetSafeProperty("LINK_FLAGS"));

  std::string linkFlagsConfig =
    cmStrCat("LINK_FLAGS_", cmSystemTools::UpperCase(GetConfigName()));
  LocalGenerator->AppendFlags(
    flags, GeneratorTarget->GetSafeProperty(linkFlagsConfig));

  std::vector<std::string> opts;
  GeneratorTarget->GetLinkOptions(opts, GetConfigName(), linkLanguage);
  // LINK_OPTIONS are escaped.
  LocalGenerator->AppendCompileOptions(flags, opts);

  LocalGenerator->AppendPositionIndependentLinkerFlags(
    flags, GeneratorTarget, GetConfigName(), linkLanguage);
}

void cmMakefileTargetGenerator::CreateRuleFile()
{
  // Create a directory for this target.
  TargetBuildDirectory = LocalGenerator->GetTargetDirectory(GeneratorTarget);
  TargetBuildDirectoryFull =
    LocalGenerator->ConvertToFullPath(TargetBuildDirectory);
  cmSystemTools::MakeDirectory(TargetBuildDirectoryFull);

  // Construct the rule file name.
  BuildFileName = cmStrCat(TargetBuildDirectory, "/build.make");
  BuildFileNameFull = cmStrCat(TargetBuildDirectoryFull, "/build.make");

  // Construct the rule file name.
  ProgressFileNameFull = cmStrCat(TargetBuildDirectoryFull, "/progress.make");

  // reset the progress count
  NumberOfProgressActions = 0;

  // Open the rule file.  This should be copy-if-different because the
  // rules may depend on this file itself.
  BuildFileStream = cm::make_unique<cmGeneratedFileStream>(
    BuildFileNameFull, false, GlobalGenerator->GetMakefileEncoding());
  if (!BuildFileStream) {
    return;
  }
  BuildFileStream->SetCopyIfDifferent(true);
  LocalGenerator->WriteDisclaimer(*BuildFileStream);
  if (GlobalGenerator->AllowDeleteOnError()) {
    std::vector<std::string> no_depends;
    std::vector<std::string> no_commands;
    LocalGenerator->WriteMakeRule(
      *BuildFileStream, "Delete rule output on recipe failure.",
      ".DELETE_ON_ERROR", no_depends, no_commands, false);
  }
  LocalGenerator->WriteSpecialTargetsTop(*BuildFileStream);
}

void cmMakefileTargetGenerator::WriteTargetBuildRules()
{
  // -- Write the custom commands for this target

  // Evaluates generator expressions and expands prop_value
  auto evaluatedFiles =
    [this](const std::string& prop_value) -> std::vector<std::string> {
    std::vector<std::string> files;
    cmExpandList(cmGeneratorExpression::Evaluate(prop_value, LocalGenerator,
                                                 GetConfigName(),
                                                 GeneratorTarget),
                 files);
    return files;
  };

  // Look for additional files registered for cleaning in this directory.
  if (cmProp prop_value =
        Makefile->GetProperty("ADDITIONAL_MAKE_CLEAN_FILES")) {
    std::vector<std::string> const files = evaluatedFiles(*prop_value);
    CleanFiles.insert(files.begin(), files.end());
  }

  // Look for additional files registered for cleaning in this target.
  if (cmProp prop_value =
        GeneratorTarget->GetProperty("ADDITIONAL_CLEAN_FILES")) {
    std::vector<std::string> const files = evaluatedFiles(*prop_value);
    // For relative path support
    std::string const& binaryDir = LocalGenerator->GetCurrentBinaryDirectory();
    for (std::string const& cfl : files) {
      CleanFiles.insert(cmSystemTools::CollapseFullPath(cfl, binaryDir));
    }
  }

  std::string currentBinDir = LocalGenerator->GetCurrentBinaryDirectory();

  // Look for ISPC extra object files generated by this target
  auto ispcAdditionalObjs =
    GeneratorTarget->GetGeneratedISPCObjects(GetConfigName());
  for (std::string const& ispcObj : ispcAdditionalObjs) {
    CleanFiles.insert(
      LocalGenerator->MaybeConvertToRelativePath(currentBinDir, ispcObj));
  }

  // add custom commands to the clean rules?
  bool clean = cmIsOff(Makefile->GetProperty("CLEAN_NO_CUSTOM"));

  // First generate the object rule files.  Save a list of all object
  // files for this target.
  std::vector<cmSourceFile const*> customCommands;
  GeneratorTarget->GetCustomCommands(customCommands, GetConfigName());
  for (cmSourceFile const* sf : customCommands) {
    if (CMP0113New &&
        !LocalGenerator->GetCommandsVisited(GeneratorTarget)
           .insert(sf)
           .second) {
      continue;
    }
    cmCustomCommandGenerator ccg(*sf->GetCustomCommand(), GetConfigName(),
                                 LocalGenerator);
    GenerateCustomRuleFile(ccg);
    if (clean) {
      const std::vector<std::string>& outputs = ccg.GetOutputs();
      for (std::string const& output : outputs) {
        CleanFiles.insert(
          LocalGenerator->MaybeConvertToRelativePath(currentBinDir, output));
      }
      const std::vector<std::string>& byproducts = ccg.GetByproducts();
      for (std::string const& byproduct : byproducts) {
        CleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
          currentBinDir, byproduct));
      }
    }
  }

  // Add byproducts from build events to the clean rules
  if (clean) {
    std::vector<cmCustomCommand> buildEventCommands =
      GeneratorTarget->GetPreBuildCommands();

    cm::append(buildEventCommands, GeneratorTarget->GetPreLinkCommands());
    cm::append(buildEventCommands, GeneratorTarget->GetPostBuildCommands());

    for (const auto& be : buildEventCommands) {
      cmCustomCommandGenerator beg(be, GetConfigName(), LocalGenerator);
      const std::vector<std::string>& byproducts = beg.GetByproducts();
      for (std::string const& byproduct : byproducts) {
        CleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
          currentBinDir, byproduct));
      }
    }
  }
  std::vector<cmSourceFile const*> headerSources;
  GeneratorTarget->GetHeaderSources(headerSources, GetConfigName());
  OSXBundleGenerator->GenerateMacOSXContentStatements(
    headerSources, MacOSXContentGenerator.get(), GetConfigName());
  std::vector<cmSourceFile const*> extraSources;
  GeneratorTarget->GetExtraSources(extraSources, GetConfigName());
  OSXBundleGenerator->GenerateMacOSXContentStatements(
    extraSources, MacOSXContentGenerator.get(), GetConfigName());
  cmProp pchExtension = Makefile->GetDefinition("CMAKE_PCH_EXTENSION");
  std::vector<cmSourceFile const*> externalObjects;
  GeneratorTarget->GetExternalObjects(externalObjects, GetConfigName());
  for (cmSourceFile const* sf : externalObjects) {
    auto const& objectFileName = sf->GetFullPath();
    if (!cmSystemTools::StringEndsWith(objectFileName,
                                       cmToCStr(pchExtension))) {
      ExternalObjects.push_back(objectFileName);
    }
  }

  std::vector<cmSourceFile const*> objectSources;
  GeneratorTarget->GetObjectSources(objectSources, GetConfigName());
  for (cmSourceFile const* sf : objectSources) {
    // Generate this object file's rule file.
    WriteObjectRuleFiles(*sf);
  }
}

void cmMakefileTargetGenerator::WriteCommonCodeRules()
{
  const char* root =
    (Makefile->IsOn("CMAKE_MAKE_INCLUDE_FROM_ROOT") ? "$(CMAKE_BINARY_DIR)/"
                                                    : "");

  // Include the dependencies for the target.
  std::string dependFileNameFull =
    cmStrCat(TargetBuildDirectoryFull, "/depend.make");
  *BuildFileStream << "# Include any dependencies generated for this target.\n"
                   << GlobalGenerator->IncludeDirective << " " << root
                   << cmSystemTools::ConvertToOutputPath(
                        LocalGenerator->MaybeConvertToRelativePath(
                          LocalGenerator->GetBinaryDirectory(),
                          dependFileNameFull))
                   << "\n\n";

  if (!NoRuleMessages) {
    // Include the progress variables for the target.
    *BuildFileStream << "# Include the progress variables for this target.\n"
                     << GlobalGenerator->IncludeDirective << " " << root
                     << cmSystemTools::ConvertToOutputPath(
                          LocalGenerator->MaybeConvertToRelativePath(
                            LocalGenerator->GetBinaryDirectory(),
                            ProgressFileNameFull))
                     << "\n\n";
  }

  // make sure the depend file exists
  if (!cmSystemTools::FileExists(dependFileNameFull)) {
    // Write an empty dependency file.
    cmGeneratedFileStream depFileStream(
      dependFileNameFull, false, GlobalGenerator->GetMakefileEncoding());
    depFileStream << "# Empty dependencies file for "
                  << GeneratorTarget->GetName() << ".\n"
                  << "# This may be replaced when dependencies are built.\n";
  }

  // Open the flags file.  This should be copy-if-different because the
  // rules may depend on this file itself.
  FlagFileNameFull = cmStrCat(TargetBuildDirectoryFull, "/flags.make");
  FlagFileStream = cm::make_unique<cmGeneratedFileStream>(
    FlagFileNameFull, false, GlobalGenerator->GetMakefileEncoding());
  if (!FlagFileStream) {
    return;
  }
  FlagFileStream->SetCopyIfDifferent(true);
  LocalGenerator->WriteDisclaimer(*FlagFileStream);

  // Include the flags for the target.
  *BuildFileStream
    << "# Include the compile flags for this target's objects.\n"
    << GlobalGenerator->IncludeDirective << " " << root
    << cmSystemTools::ConvertToOutputPath(
         LocalGenerator->MaybeConvertToRelativePath(
           LocalGenerator->GetBinaryDirectory(), FlagFileNameFull))
    << "\n\n";
}

void cmMakefileTargetGenerator::WriteTargetLanguageFlags()
{
  // write language flags for target
  std::set<std::string> languages;
  GeneratorTarget->GetLanguages(
    languages, Makefile->GetSafeDefinition("CMAKE_BUILD_TYPE"));
  // put the compiler in the rules.make file so that if it changes
  // things rebuild
  for (std::string const& language : languages) {
    std::string compiler = cmStrCat("CMAKE_", language, "_COMPILER");
    *FlagFileStream << "# compile " << language << " with "
                    << Makefile->GetSafeDefinition(compiler) << "\n";
  }

  bool const escapeOctothorpe = GlobalGenerator->CanEscapeOctothorpe();

  for (std::string const& language : languages) {
    std::string defines = GetDefines(language, GetConfigName());
    std::string includes = GetIncludes(language, GetConfigName());
    if (escapeOctothorpe) {
      // Escape comment characters so they do not terminate assignment.
      cmSystemTools::ReplaceString(defines, "#", "\\#");
      cmSystemTools::ReplaceString(includes, "#", "\\#");
    }
    *FlagFileStream << language << "_DEFINES = " << defines << "\n\n";
    *FlagFileStream << language << "_INCLUDES = " << includes << "\n\n";

    std::vector<std::string> architectures;
    GeneratorTarget->GetAppleArchs(GetConfigName(), architectures);
    architectures.emplace_back();

    for (const std::string& arch : architectures) {
      std::string flags = GetFlags(language, GetConfigName(), arch);
      if (escapeOctothorpe) {
        cmSystemTools::ReplaceString(flags, "#", "\\#");
      }
      *FlagFileStream << language << "_FLAGS" << arch << " = " << flags
                      << "\n\n";
    }
  }
}

void cmMakefileTargetGenerator::MacOSXContentGeneratorType::operator()(
  cmSourceFile const& source, const char* pkgloc, const std::string& config)
{
  // Skip OS X content when not building a Framework or Bundle.
  if (!Generator->GetGeneratorTarget()->IsBundleOnApple()) {
    return;
  }

  std::string macdir =
    Generator->OSXBundleGenerator->InitMacOSXContentDirectory(pkgloc, config);

  // Get the input file location.
  std::string const& input = source.GetFullPath();

  // Get the output file location.
  std::string output =
    cmStrCat(macdir, '/', cmSystemTools::GetFilenameName(input));
  Generator->CleanFiles.insert(
    Generator->LocalGenerator->MaybeConvertToRelativePath(
      Generator->LocalGenerator->GetCurrentBinaryDirectory(), output));
  output = Generator->LocalGenerator->MaybeConvertToRelativePath(
    Generator->LocalGenerator->GetBinaryDirectory(), output);

  // Create a rule to copy the content into the bundle.
  std::vector<std::string> depends;
  std::vector<std::string> commands;
  depends.push_back(input);
  std::string copyEcho = cmStrCat("Copying OS X content ", output);
  Generator->LocalGenerator->AppendEcho(
    commands, copyEcho, cmLocalUnixMakefileGenerator3::EchoBuild);
  std::string copyCommand =
    cmStrCat("$(CMAKE_COMMAND) -E copy ",
             Generator->LocalGenerator->ConvertToOutputFormat(
               input, cmOutputConverter::SHELL),
             ' ',
             Generator->LocalGenerator->ConvertToOutputFormat(
               output, cmOutputConverter::SHELL));
  commands.push_back(std::move(copyCommand));
  Generator->LocalGenerator->WriteMakeRule(
    *Generator->BuildFileStream, nullptr, output, depends, commands, false);
  Generator->ExtraFiles.insert(output);
}

void cmMakefileTargetGenerator::WriteObjectRuleFiles(
  cmSourceFile const& source)
{
  // Identify the language of the source file.
  const std::string& lang = LocalGenerator->GetSourceFileLanguage(source);
  if (lang.empty()) {
    // don't know anything about this file so skip it
    return;
  }

  // Get the full path name of the object file.
  std::string const& objectName = GeneratorTarget->GetObjectName(&source);
  std::string obj = cmStrCat(
    LocalGenerator->GetTargetDirectory(GeneratorTarget), '/', objectName);

  // Avoid generating duplicate rules.
  if (ObjectFiles.find(obj) == ObjectFiles.end()) {
    ObjectFiles.insert(obj);
  } else {
    std::ostringstream err;
    err << "Warning: Source file \"" << source.GetFullPath()
        << "\" is listed multiple times for target \""
        << GeneratorTarget->GetName() << "\".";
    cmSystemTools::Message(err.str(), "Warning");
    return;
  }

  // Create the directory containing the object file.  This may be a
  // subdirectory under the target's directory.
  {
    std::string dir = cmSystemTools::GetFilenamePath(obj);
    cmSystemTools::MakeDirectory(LocalGenerator->ConvertToFullPath(dir));
  }

  // Save this in the target's list of object files.
  Objects.push_back(obj);
  CleanFiles.insert(obj);

  std::vector<std::string> depends;

  // The object file should be checked for dependency integrity.
  std::string objFullPath =
    cmStrCat(LocalGenerator->GetCurrentBinaryDirectory(), '/', obj);
  objFullPath = cmSystemTools::CollapseFullPath(objFullPath);
  std::string srcFullPath =
    cmSystemTools::CollapseFullPath(source.GetFullPath());
  LocalGenerator->AddImplicitDepends(GeneratorTarget, lang, objFullPath,
                                     srcFullPath);

  LocalGenerator->AppendRuleDepend(depends, FlagFileNameFull.c_str());
  LocalGenerator->AppendRuleDepends(depends, FlagFileDepends[lang]);

  // generate the depend scanning rule
  WriteObjectDependRules(source, depends);

  std::string config = GetConfigName();
  std::string configUpper = cmSystemTools::UpperCase(config);

  // Add precompile headers dependencies
  std::vector<std::string> architectures;
  GeneratorTarget->GetAppleArchs(config, architectures);
  if (architectures.empty()) {
    architectures.emplace_back();
  }

  std::string filterArch;
  std::unordered_map<std::string, std::string> pchSources;
  for (const std::string& arch : architectures) {
    const std::string pchSource =
      GeneratorTarget->GetPchSource(config, lang, arch);
    if (pchSource == source.GetFullPath()) {
      filterArch = arch;
    }
    if (!pchSource.empty()) {
      pchSources.insert(std::make_pair(pchSource, arch));
    }
  }

  if (!pchSources.empty() && !source.GetProperty("SKIP_PRECOMPILE_HEADERS")) {
    for (const std::string& arch : architectures) {
      std::string const& pchHeader =
        GeneratorTarget->GetPchHeader(config, lang, arch);
      depends.push_back(pchHeader);
      if (pchSources.find(source.GetFullPath()) == pchSources.end()) {
        depends.push_back(GeneratorTarget->GetPchFile(config, lang, arch));
      }
      LocalGenerator->AddImplicitDepends(GeneratorTarget, lang, objFullPath,
                                         pchHeader);
    }
  }

  if (lang != "ISPC") {
    auto const& headers = GeneratorTarget->GetGeneratedISPCHeaders(config);
    if (!headers.empty()) {
      depends.insert(depends.end(), headers.begin(), headers.end());
    }
  }

  std::string relativeObj =
    cmStrCat(LocalGenerator->GetHomeRelativeOutputPath(), obj);
  // Write the build rule.

  // Build the set of compiler flags.
  std::string flags;

  // explicitly add the explicit language flag before any other flag
  // this way backwards compatibility with user flags is maintained
  if (source.GetProperty("LANGUAGE")) {
    LocalGenerator->AppendFeatureOptions(flags, lang, "EXPLICIT_LANGUAGE");
  }

  // Add language-specific flags.
  std::string langFlags = cmStrCat("$(", lang, "_FLAGS", filterArch, ")");
  LocalGenerator->AppendFlags(flags, langFlags);

  cmGeneratorExpressionInterpreter genexInterpreter(LocalGenerator, config,
                                                    GeneratorTarget, lang);

  // Add Fortran format flags.
  if (lang == "Fortran") {
    AppendFortranFormatFlags(flags, source);
    AppendFortranPreprocessFlags(flags, source);
  }

  std::string ispcHeaderRelative;
  std::string ispcHeaderForShell;
  if (lang == "ISPC") {
    std::string ispcSource =
      cmSystemTools::GetFilenameWithoutLastExtension(objectName);

    std::string directory = GeneratorTarget->GetObjectDirectory(config);
    if (cmProp prop = GeneratorTarget->GetProperty("ISPC_HEADER_DIRECTORY")) {
      directory = cmStrCat(LocalGenerator->GetBinaryDirectory(), '/', *prop);
    }
    ispcHeaderRelative = cmStrCat(directory, '/', ispcSource, ".h");
    ispcHeaderForShell = LocalGenerator->ConvertToOutputFormat(
      ispcHeaderRelative, cmOutputConverter::SHELL);
  }

  // Add flags from source file properties.
  const std::string COMPILE_FLAGS("COMPILE_FLAGS");
  if (cmProp cflags = source.GetProperty(COMPILE_FLAGS)) {
    const std::string& evaluatedFlags =
      genexInterpreter.Evaluate(*cflags, COMPILE_FLAGS);
    LocalGenerator->AppendFlags(flags, evaluatedFlags);
    *FlagFileStream << "# Custom flags: " << relativeObj
                    << "_FLAGS = " << evaluatedFlags << "\n"
                    << "\n";
  }

  const std::string COMPILE_OPTIONS("COMPILE_OPTIONS");
  if (cmProp coptions = source.GetProperty(COMPILE_OPTIONS)) {
    const std::string& evaluatedOptions =
      genexInterpreter.Evaluate(*coptions, COMPILE_OPTIONS);
    LocalGenerator->AppendCompileOptions(flags, evaluatedOptions);
    *FlagFileStream << "# Custom options: " << relativeObj
                    << "_OPTIONS = " << evaluatedOptions << "\n"
                    << "\n";
  }

  // Add precompile headers compile options.
  if (!pchSources.empty() && !source.GetProperty("SKIP_PRECOMPILE_HEADERS")) {
    std::string pchOptions;
    auto pchIt = pchSources.find(source.GetFullPath());
    if (pchIt != pchSources.end()) {
      pchOptions = GeneratorTarget->GetPchCreateCompileOptions(config, lang,
                                                               pchIt->second);
    } else {
      pchOptions = GeneratorTarget->GetPchUseCompileOptions(config, lang);
    }

    const std::string& evaluatedFlags =
      genexInterpreter.Evaluate(pchOptions, COMPILE_OPTIONS);

    LocalGenerator->AppendCompileOptions(flags, evaluatedFlags);
    *FlagFileStream << "# PCH options: " << relativeObj
                    << "_OPTIONS = " << evaluatedFlags << "\n"
                    << "\n";
  }

  // Add include directories from source file properties.
  std::vector<std::string> includes;

  const std::string INCLUDE_DIRECTORIES("INCLUDE_DIRECTORIES");
  if (cmProp cincludes = source.GetProperty(INCLUDE_DIRECTORIES)) {
    const std::string& evaluatedIncludes =
      genexInterpreter.Evaluate(*cincludes, INCLUDE_DIRECTORIES);
    LocalGenerator->AppendIncludeDirectories(includes, evaluatedIncludes,
                                             source);
    *FlagFileStream << "# Custom include directories: " << relativeObj
                    << "_INCLUDE_DIRECTORIES = " << evaluatedIncludes << "\n"
                    << "\n";
  }

  // Add language-specific defines.
  std::set<std::string> defines;

  // Add source-specific preprocessor definitions.
  const std::string COMPILE_DEFINITIONS("COMPILE_DEFINITIONS");
  if (cmProp compile_defs = source.GetProperty(COMPILE_DEFINITIONS)) {
    const std::string& evaluatedDefs =
      genexInterpreter.Evaluate(*compile_defs, COMPILE_DEFINITIONS);
    LocalGenerator->AppendDefines(defines, evaluatedDefs);
    *FlagFileStream << "# Custom defines: " << relativeObj
                    << "_DEFINES = " << evaluatedDefs << "\n"
                    << "\n";
  }
  std::string defPropName = cmStrCat("COMPILE_DEFINITIONS_", configUpper);
  if (cmProp config_compile_defs = source.GetProperty(defPropName)) {
    const std::string& evaluatedDefs =
      genexInterpreter.Evaluate(*config_compile_defs, COMPILE_DEFINITIONS);
    LocalGenerator->AppendDefines(defines, evaluatedDefs);
    *FlagFileStream << "# Custom defines: " << relativeObj << "_DEFINES_"
                    << configUpper << " = " << evaluatedDefs << "\n"
                    << "\n";
  }

  // Get the output paths for source and object files.
  std::string sourceFile = LocalGenerator->ConvertToOutputFormat(
    source.GetFullPath(), cmOutputConverter::SHELL);

  // Construct the build message.
  std::vector<std::string> no_commands;
  std::vector<std::string> commands;

  // add in a progress call if needed
  NumberOfProgressActions++;

  if (!NoRuleMessages) {
    cmLocalUnixMakefileGenerator3::EchoProgress progress;
    MakeEchoProgress(progress);
    std::string buildEcho =
      cmStrCat("Building ", lang, " object ", relativeObj);
    LocalGenerator->AppendEcho(commands, buildEcho,
                               cmLocalUnixMakefileGenerator3::EchoBuild,
                               &progress);
  }

  std::string targetOutPathReal;
  std::string targetOutPathPDB;
  std::string targetOutPathCompilePDB;
  {
    std::string targetFullPathReal;
    std::string targetFullPathPDB;
    std::string targetFullPathCompilePDB =
      ComputeTargetCompilePDB(GetConfigName());
    if (GeneratorTarget->GetType() == cmStateEnums::EXECUTABLE ||
        GeneratorTarget->GetType() == cmStateEnums::STATIC_LIBRARY ||
        GeneratorTarget->GetType() == cmStateEnums::SHARED_LIBRARY ||
        GeneratorTarget->GetType() == cmStateEnums::MODULE_LIBRARY) {
      targetFullPathReal = GeneratorTarget->GetFullPath(
        GetConfigName(), cmStateEnums::RuntimeBinaryArtifact, true);
      targetFullPathPDB =
        cmStrCat(GeneratorTarget->GetPDBDirectory(GetConfigName()), '/',
                 GeneratorTarget->GetPDBName(GetConfigName()));
    }

    targetOutPathReal = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal),
      cmOutputConverter::SHELL);
    targetOutPathPDB = LocalGenerator->ConvertToOutputFormat(
      targetFullPathPDB, cmOutputConverter::SHELL);
    targetOutPathCompilePDB = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathCompilePDB),
      cmOutputConverter::SHELL);

    if (LocalGenerator->IsMinGWMake() &&
        cmHasLiteralSuffix(targetOutPathCompilePDB, "\\")) {
      // mingw32-make incorrectly interprets 'a\ b c' as 'a b' and 'c'
      // (but 'a\ b "c"' as 'a\', 'b', and 'c'!).  Workaround this by
      // avoiding a trailing backslash in the argument.
      targetOutPathCompilePDB.back() = '/';
    }
  }
  cmRulePlaceholderExpander::RuleVariables vars;
  vars.CMTargetName = GeneratorTarget->GetName().c_str();
  vars.CMTargetType =
    cmState::GetTargetTypeName(GeneratorTarget->GetType()).c_str();
  vars.Language = lang.c_str();
  vars.Target = targetOutPathReal.c_str();
  vars.TargetPDB = targetOutPathPDB.c_str();
  vars.TargetCompilePDB = targetOutPathCompilePDB.c_str();
  vars.Source = sourceFile.c_str();
  std::string shellObj =
    LocalGenerator->ConvertToOutputFormat(obj, cmOutputConverter::SHELL);
  vars.Object = shellObj.c_str();
  std::string objectDir = GeneratorTarget->GetSupportDirectory();
  objectDir = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), objectDir),
    cmOutputConverter::SHELL);
  vars.ObjectDir = objectDir.c_str();
  std::string objectFileDir = cmSystemTools::GetFilenamePath(obj);
  objectFileDir = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), objectFileDir),
    cmOutputConverter::SHELL);
  vars.ObjectFileDir = objectFileDir.c_str();
  vars.Flags = flags.c_str();
  vars.ISPCHeader = ispcHeaderForShell.c_str();

  std::string definesString = cmStrCat("$(", lang, "_DEFINES)");

  LocalGenerator->JoinDefines(defines, definesString, lang);

  vars.Defines = definesString.c_str();

  std::string includesString = LocalGenerator->GetIncludeFlags(
    includes, GeneratorTarget, lang, true, false, config);
  LocalGenerator->AppendFlags(includesString, "$(" + lang + "_INCLUDES)");
  vars.Includes = includesString.c_str();

  // At the moment, it is assumed that C, C++, Fortran, and CUDA have both
  // assembly and preprocessor capabilities. The same is true for the
  // ability to export compile commands
  bool lang_has_preprocessor =
    ((lang == "C") || (lang == "CXX") || (lang == "OBJC") ||
     (lang == "OBJCXX") || (lang == "Fortran") || (lang == "CUDA") ||
     lang == "ISPC");
  bool const lang_has_assembly = lang_has_preprocessor;
  bool const lang_can_export_cmds = lang_has_preprocessor;

  std::unique_ptr<cmRulePlaceholderExpander> rulePlaceholderExpander(
    LocalGenerator->CreateRulePlaceholderExpander());

  // Construct the compile rules.
  {
    std::vector<std::string> compileCommands;
    if (lang == "CUDA") {
      std::string cmdVar;
      if (GeneratorTarget->GetPropertyAsBool("CUDA_SEPARABLE_COMPILATION")) {
        cmdVar = "CMAKE_CUDA_COMPILE_SEPARABLE_COMPILATION";
      } else if (GeneratorTarget->GetPropertyAsBool("CUDA_PTX_COMPILATION")) {
        cmdVar = "CMAKE_CUDA_COMPILE_PTX_COMPILATION";
      } else {
        cmdVar = "CMAKE_CUDA_COMPILE_WHOLE_COMPILATION";
      }
      const std::string& compileRule = Makefile->GetRequiredDefinition(cmdVar);
      cmExpandList(compileRule, compileCommands);
    } else {
      const std::string cmdVar = "CMAKE_" + lang + "_COMPILE_OBJECT";
      const std::string& compileRule = Makefile->GetRequiredDefinition(cmdVar);
      cmExpandList(compileRule, compileCommands);
    }

    if (Makefile->IsOn("CMAKE_EXPORT_COMPILE_COMMANDS") &&
        lang_can_export_cmds && compileCommands.size() == 1) {
      std::string compileCommand = compileCommands[0];

      // no launcher for CMAKE_EXPORT_COMPILE_COMMANDS
      rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                   compileCommand, vars);
      std::string workingDirectory =
        LocalGenerator->GetCurrentBinaryDirectory();
      compileCommand.replace(compileCommand.find(langFlags), langFlags.size(),
                             GetFlags(lang, GetConfigName()));
      std::string langDefines = std::string("$(") + lang + "_DEFINES)";
      compileCommand.replace(compileCommand.find(langDefines),
                             langDefines.size(),
                             GetDefines(lang, GetConfigName()));
      std::string langIncludes = std::string("$(") + lang + "_INCLUDES)";
      compileCommand.replace(compileCommand.find(langIncludes),
                             langIncludes.size(),
                             GetIncludes(lang, GetConfigName()));

      cmProp eliminate[] = { Makefile->GetDefinition("CMAKE_START_TEMP_FILE"),
                             Makefile->GetDefinition("CMAKE_END_TEMP_FILE") };
      for (cmProp el : eliminate) {
        if (el) {
          cmSystemTools::ReplaceString(compileCommand, *el, "");
        }
      }

      GlobalGenerator->AddCXXCompileCommand(source.GetFullPath(),
                                            workingDirectory, compileCommand);
    }

    // See if we need to use a compiler launcher like ccache or distcc
    std::string compilerLauncher;
    if (!compileCommands.empty() &&
        (lang == "C" || lang == "CXX" || lang == "Fortran" || lang == "CUDA" ||
         lang == "ISPC" || lang == "OBJC" || lang == "OBJCXX")) {
      std::string const clauncher_prop = lang + "_COMPILER_LAUNCHER";
      cmProp clauncher = GeneratorTarget->GetProperty(clauncher_prop);
      if (cmNonempty(clauncher)) {
        compilerLauncher = *clauncher;
      }
    }

    // Maybe insert an include-what-you-use runner.
    if (!compileCommands.empty() && (lang == "C" || lang == "CXX")) {
      std::string const iwyu_prop = lang + "_INCLUDE_WHAT_YOU_USE";
      cmProp iwyu = GeneratorTarget->GetProperty(iwyu_prop);
      std::string const tidy_prop = lang + "_CLANG_TIDY";
      cmProp tidy = GeneratorTarget->GetProperty(tidy_prop);
      std::string const cpplint_prop = lang + "_CPPLINT";
      cmProp cpplint = GeneratorTarget->GetProperty(cpplint_prop);
      std::string const cppcheck_prop = lang + "_CPPCHECK";
      cmProp cppcheck = GeneratorTarget->GetProperty(cppcheck_prop);
      if (cmNonempty(iwyu) || cmNonempty(tidy) || cmNonempty(cpplint) ||
          cmNonempty(cppcheck)) {
        std::string run_iwyu = "$(CMAKE_COMMAND) -E __run_co_compile";
        if (!compilerLauncher.empty()) {
          // In __run_co_compile case the launcher command is supplied
          // via --launcher=<maybe-list> and consumed
          run_iwyu += " --launcher=";
          run_iwyu += LocalGenerator->EscapeForShell(compilerLauncher);
          compilerLauncher.clear();
        }
        if (cmNonempty(iwyu)) {
          run_iwyu += " --iwyu=";
          run_iwyu += LocalGenerator->EscapeForShell(*iwyu);
        }
        if (cmNonempty(tidy)) {
          run_iwyu += " --tidy=";
          cmProp p = Makefile->GetDefinition("CMAKE_" + lang +
                                             "_CLANG_TIDY_DRIVER_MODE");
          std::string driverMode;
          if (cmNonempty(p)) {
            driverMode = *p;
          } else {
            driverMode = lang == "C" ? "gcc" : "g++";
          }
          run_iwyu += LocalGenerator->EscapeForShell(
            cmStrCat(*tidy, ";--extra-arg-before=--driver-mode=", driverMode));
        }
        if (cmNonempty(cpplint)) {
          run_iwyu += " --cpplint=";
          run_iwyu += LocalGenerator->EscapeForShell(*cpplint);
        }
        if (cmNonempty(cppcheck)) {
          run_iwyu += " --cppcheck=";
          run_iwyu += LocalGenerator->EscapeForShell(*cppcheck);
        }
        if (cmNonempty(tidy) || (cmNonempty(cpplint)) ||
            (cmNonempty(cppcheck))) {
          run_iwyu += " --source=";
          run_iwyu += sourceFile;
        }
        run_iwyu += " -- ";
        compileCommands.front().insert(0, run_iwyu);
      }
    }

    // If compiler launcher was specified and not consumed above, it
    // goes to the beginning of the command line.
    if (!compileCommands.empty() && !compilerLauncher.empty()) {
      std::vector<std::string> args = cmExpandedList(compilerLauncher, true);
      if (!args.empty()) {
        args[0] = LocalGenerator->ConvertToOutputFormat(
          args[0], cmOutputConverter::SHELL);
        for (std::string& i : cmMakeRange(args.begin() + 1, args.end())) {
          i = LocalGenerator->EscapeForShell(i);
        }
      }
      compileCommands.front().insert(0, cmJoin(args, " ") + " ");
    }

    std::string launcher;
    {
      const char* val = LocalGenerator->GetRuleLauncher(GeneratorTarget,
                                                        "RULE_LAUNCH_COMPILE");
      if (cmNonempty(val)) {
        launcher = cmStrCat(val, ' ');
      }
    }

    // Expand placeholders in the commands.
    for (std::string& compileCommand : compileCommands) {
      compileCommand = cmStrCat(launcher, compileCommand);
      rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                   compileCommand, vars);
    }

    // Change the command working directory to the local build tree.
    LocalGenerator->CreateCDCommand(
      compileCommands, LocalGenerator->GetCurrentBinaryDirectory(),
      LocalGenerator->GetBinaryDirectory());
    cm::append(commands, compileCommands);
  }

  // Check for extra outputs created by the compilation.
  std::vector<std::string> outputs(1, relativeObj);
  if (cmProp extra_outputs_str = source.GetProperty("OBJECT_OUTPUTS")) {
    std::string evaluated_outputs = cmGeneratorExpression::Evaluate(
      *extra_outputs_str, LocalGenerator, config);

    if (!evaluated_outputs.empty()) {
      // Register these as extra files to clean.
      cmExpandList(evaluated_outputs, outputs);
    }
  }
  if (!ispcHeaderRelative
         .empty()) { // can't move ispcHeader as vars is using it
    outputs.emplace_back(ispcHeaderRelative);
  }

  if (outputs.size() > 1) {
    CleanFiles.insert(outputs.begin() + 1, outputs.end());
  }

  // Write the rule.
  WriteMakeRule(*BuildFileStream, nullptr, outputs, depends, commands);

  bool do_preprocess_rules = lang_has_preprocessor &&
    LocalGenerator->GetCreatePreprocessedSourceRules();
  bool do_assembly_rules =
    lang_has_assembly && LocalGenerator->GetCreateAssemblySourceRules();
  if (do_preprocess_rules || do_assembly_rules) {
    std::vector<std::string> force_depends;
    force_depends.emplace_back("cmake_force");
    std::string::size_type dot_pos = relativeObj.rfind('.');
    std::string relativeObjBase = relativeObj.substr(0, dot_pos);
    dot_pos = obj.rfind('.');
    std::string objBase = obj.substr(0, dot_pos);

    if (do_preprocess_rules) {
      commands.clear();
      std::string relativeObjI = relativeObjBase + ".i";
      std::string objI = objBase + ".i";

      std::string preprocessEcho =
        cmStrCat("Preprocessing ", lang, " source to ", objI);
      LocalGenerator->AppendEcho(commands, preprocessEcho,
                                 cmLocalUnixMakefileGenerator3::EchoBuild);

      std::string preprocessRuleVar =
        cmStrCat("CMAKE_", lang, "_CREATE_PREPROCESSED_SOURCE");
      if (cmProp preprocessRule = Makefile->GetDefinition(preprocessRuleVar)) {
        std::vector<std::string> preprocessCommands =
          cmExpandedList(*preprocessRule);

        std::string shellObjI = LocalGenerator->ConvertToOutputFormat(
          objI, cmOutputConverter::SHELL);
        vars.PreprocessedSource = shellObjI.c_str();

        // Expand placeholders in the commands.
        for (std::string& preprocessCommand : preprocessCommands) {
          // no launcher for preprocessor commands
          rulePlaceholderExpander->ExpandRuleVariables(
            LocalGenerator, preprocessCommand, vars);
        }

        LocalGenerator->CreateCDCommand(
          preprocessCommands, LocalGenerator->GetCurrentBinaryDirectory(),
          LocalGenerator->GetBinaryDirectory());
        cm::append(commands, preprocessCommands);
      } else {
        std::string cmd =
          cmStrCat("$(CMAKE_COMMAND) -E cmake_unimplemented_variable ",
                   preprocessRuleVar);
        commands.push_back(std::move(cmd));
      }

      LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, relativeObjI,
                                    force_depends, commands, false);
    }

    if (do_assembly_rules) {
      commands.clear();
      std::string relativeObjS = relativeObjBase + ".s";
      std::string objS = objBase + ".s";

      std::string assemblyEcho =
        cmStrCat("Compiling ", lang, " source to assembly ", objS);
      LocalGenerator->AppendEcho(commands, assemblyEcho,
                                 cmLocalUnixMakefileGenerator3::EchoBuild);

      std::string assemblyRuleVar =
        cmStrCat("CMAKE_", lang, "_CREATE_ASSEMBLY_SOURCE");
      if (cmProp assemblyRule = Makefile->GetDefinition(assemblyRuleVar)) {
        std::vector<std::string> assemblyCommands =
          cmExpandedList(*assemblyRule);

        std::string shellObjS = LocalGenerator->ConvertToOutputFormat(
          objS, cmOutputConverter::SHELL);
        vars.AssemblySource = shellObjS.c_str();

        // Expand placeholders in the commands.
        for (std::string& assemblyCommand : assemblyCommands) {
          // no launcher for assembly commands
          rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                       assemblyCommand, vars);
        }

        LocalGenerator->CreateCDCommand(
          assemblyCommands, LocalGenerator->GetCurrentBinaryDirectory(),
          LocalGenerator->GetBinaryDirectory());
        cm::append(commands, assemblyCommands);
      } else {
        std::string cmd =
          cmStrCat("$(CMAKE_COMMAND) -E cmake_unimplemented_variable ",
                   assemblyRuleVar);
        commands.push_back(std::move(cmd));
      }

      LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, relativeObjS,
                                    force_depends, commands, false);
    }
  }
}

void cmMakefileTargetGenerator::WriteTargetCleanRules()
{
  std::vector<std::string> depends;
  std::vector<std::string> commands;

  // Construct the clean target name.
  std::string cleanTarget = cmStrCat(
    LocalGenerator->GetRelativeTargetDirectory(GeneratorTarget), "/clean");

  // Construct the clean command.
  LocalGenerator->AppendCleanCommand(commands, CleanFiles, GeneratorTarget);
  LocalGenerator->CreateCDCommand(commands,
                                  LocalGenerator->GetCurrentBinaryDirectory(),
                                  LocalGenerator->GetBinaryDirectory());

  // Write the rule.
  LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, cleanTarget,
                                depends, commands, true);
}

bool cmMakefileTargetGenerator::WriteMakeRule(
  std::ostream& os, const char* comment,
  const std::vector<std::string>& outputs,
  const std::vector<std::string>& depends,
  const std::vector<std::string>& commands, bool in_help)
{
  bool symbolic = false;
  if (outputs.empty()) {
    return symbolic;
  }

  // Check whether we need to bother checking for a symbolic output.
  bool need_symbolic = GlobalGenerator->GetNeedSymbolicMark();

  // Check whether the first output is marked as symbolic.
  if (need_symbolic) {
    if (cmSourceFile* sf = Makefile->GetSource(outputs[0])) {
      symbolic = sf->GetPropertyAsBool("SYMBOLIC");
    }
  }

  // We always attach the actual commands to the first output.
  LocalGenerator->WriteMakeRule(os, comment, outputs[0], depends, commands,
                                symbolic, in_help);

  // For single outputs, we are done.
  if (outputs.size() == 1) {
    return symbolic;
  }

  // For multiple outputs, make the extra ones depend on the first one.
  std::vector<std::string> const output_depends(1, outputs[0]);
  std::string binDir = LocalGenerator->GetBinaryDirectory();
  for (std::string const& output : cmMakeRange(outputs).advance(1)) {
    // Touch the extra output so "make" knows that it was updated,
    // but only if the output was actually created.
    std::string const out = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(binDir, output),
      cmOutputConverter::SHELL);
    std::vector<std::string> output_commands;

    bool o_symbolic = false;
    if (need_symbolic) {
      if (cmSourceFile* sf = Makefile->GetSource(output)) {
        o_symbolic = sf->GetPropertyAsBool("SYMBOLIC");
      }
    }
    symbolic = symbolic && o_symbolic;

    if (!o_symbolic) {
      output_commands.push_back("@$(CMAKE_COMMAND) -E touch_nocreate " + out);
    }
    LocalGenerator->WriteMakeRule(os, nullptr, output, output_depends,
                                  output_commands, o_symbolic, in_help);

    if (!o_symbolic) {
      // At build time, remove the first output if this one does not exist
      // so that "make" will rerun the real commands that create this one.
      MultipleOutputPairsType::value_type p(output, outputs[0]);
      MultipleOutputPairs.insert(p);
    }
  }
  return symbolic;
}

void cmMakefileTargetGenerator::WriteTargetDependRules()
{
  // must write the targets depend info file
  std::string dir = LocalGenerator->GetTargetDirectory(GeneratorTarget);
  InfoFileNameFull = cmStrCat(dir, "/DependInfo.cmake");
  InfoFileNameFull = LocalGenerator->ConvertToFullPath(InfoFileNameFull);
  InfoFileStream = cm::make_unique<cmGeneratedFileStream>(InfoFileNameFull);
  if (!InfoFileStream) {
    return;
  }
  InfoFileStream->SetCopyIfDifferent(true);
  LocalGenerator->WriteDependLanguageInfo(*InfoFileStream, GeneratorTarget);

  // Store multiple output pairs in the depend info file.
  if (!MultipleOutputPairs.empty()) {
    /* clang-format off */
    *InfoFileStream
      << "\n"
      << "# Pairs of files generated by the same build rule.\n"
      << "set(CMAKE_MULTIPLE_OUTPUT_PAIRS\n";
    /* clang-format on */
    for (auto const& pi : MultipleOutputPairs) {
      *InfoFileStream << "  " << cmOutputConverter::EscapeForCMake(pi.first)
                      << " " << cmOutputConverter::EscapeForCMake(pi.second)
                      << "\n";
    }
    *InfoFileStream << "  )\n\n";
  }

  // Store list of targets linked directly or transitively.
  {
    /* clang-format off */
  *InfoFileStream
    << "\n"
    << "# Targets to which this target links.\n"
    << "set(CMAKE_TARGET_LINKED_INFO_FILES\n";
    /* clang-format on */
    std::vector<std::string> dirs =
      GetLinkedTargetDirectories(GetConfigName());
    for (std::string const& d : dirs) {
      *InfoFileStream << "  \"" << d << "/DependInfo.cmake\"\n";
    }
    *InfoFileStream << "  )\n";
  }

  std::string const& working_dir = LocalGenerator->GetCurrentBinaryDirectory();

  /* clang-format off */
  *InfoFileStream
    << "\n"
    << "# Fortran module output directory.\n"
    << "set(CMAKE_Fortran_TARGET_MODULE_DIR \""
    << GeneratorTarget->GetFortranModuleDirectory(working_dir)
    << "\")\n";
  /* clang-format on */

  // and now write the rule to use it
  std::vector<std::string> depends;
  std::vector<std::string> commands;

  // Construct the name of the dependency generation target.
  std::string depTarget = cmStrCat(
    LocalGenerator->GetRelativeTargetDirectory(GeneratorTarget), "/depend");

  // Add a command to call CMake to scan dependencies.  CMake will
  // touch the corresponding depends file after scanning dependencies.
  std::ostringstream depCmd;
// TODO: Account for source file properties and directory-level
// definitions when scanning for dependencies.
#if !defined(_WIN32) || defined(__CYGWIN__)
  // This platform supports symlinks, so cmSystemTools will translate
  // paths.  Make sure PWD is set to the original name of the home
  // output directory to help cmSystemTools to create the same
  // translation table for the dependency scanning process.
  depCmd << "cd "
         << (LocalGenerator->ConvertToOutputFormat(
              LocalGenerator->GetBinaryDirectory(), cmOutputConverter::SHELL))
         << " && ";
#endif
  // Generate a call this signature:
  //
  //   cmake -E cmake_depends <generator>
  //                          <home-src-dir> <start-src-dir>
  //                          <home-out-dir> <start-out-dir>
  //                          <dep-info> --color=$(COLOR)
  //
  // This gives the dependency scanner enough information to recreate
  // the state of our local generator sufficiently for its needs.
  depCmd << "$(CMAKE_COMMAND) -E cmake_depends \""
         << GlobalGenerator->GetName() << "\" "
         << LocalGenerator->ConvertToOutputFormat(
              LocalGenerator->GetSourceDirectory(), cmOutputConverter::SHELL)
         << " "
         << LocalGenerator->ConvertToOutputFormat(
              LocalGenerator->GetCurrentSourceDirectory(),
              cmOutputConverter::SHELL)
         << " "
         << LocalGenerator->ConvertToOutputFormat(
              LocalGenerator->GetBinaryDirectory(), cmOutputConverter::SHELL)
         << " "
         << LocalGenerator->ConvertToOutputFormat(
              LocalGenerator->GetCurrentBinaryDirectory(),
              cmOutputConverter::SHELL)
         << " "
         << LocalGenerator->ConvertToOutputFormat(
              cmSystemTools::CollapseFullPath(InfoFileNameFull),
              cmOutputConverter::SHELL);
  if (LocalGenerator->GetColorMakefile()) {
    depCmd << " --color=$(COLOR)";
  }
  commands.push_back(depCmd.str());

  // Make sure all custom command outputs in this target are built.
  if (CustomCommandDriver == OnDepends) {
    DriveCustomCommands(depends);
  }

  // Write the rule.
  LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, depTarget, depends,
                                commands, true);
}

void cmMakefileTargetGenerator::DriveCustomCommands(
  std::vector<std::string>& depends)
{
  // Depend on all custom command outputs.
  cm::append(depends, CustomCommandOutputs);
}

void cmMakefileTargetGenerator::WriteObjectDependRules(
  cmSourceFile const& source, std::vector<std::string>& depends)
{
  // Create the list of dependencies known at cmake time.  These are
  // shared between the object file and dependency scanning rule.
  depends.push_back(source.GetFullPath());
  if (cmProp objectDeps = source.GetProperty("OBJECT_DEPENDS")) {
    cmExpandList(*objectDeps, depends);
  }
}

void cmMakefileTargetGenerator::WriteDeviceLinkRule(
  std::vector<std::string>& commands, const std::string& output)
{
  std::string architecturesStr =
    GeneratorTarget->GetSafeProperty("CUDA_ARCHITECTURES");

  if (cmIsOff(architecturesStr)) {
    Makefile->IssueMessage(MessageType::FATAL_ERROR,
                           "CUDA_SEPARABLE_COMPILATION on Clang "
                           "requires CUDA_ARCHITECTURES to be set.");
    return;
  }

  std::vector<std::string> architectures = cmExpandedList(architecturesStr);

  // Ensure there are no duplicates.
  const std::vector<std::string> linkDeps = [&]() -> std::vector<std::string> {
    std::vector<std::string> deps;
    AppendTargetDepends(deps, true);
    GeneratorTarget->GetLinkDepends(deps, GetConfigName(), "CUDA");
    std::copy(Objects.begin(), Objects.end(), std::back_inserter(deps));

    std::unordered_set<std::string> depsSet(deps.begin(), deps.end());
    deps.clear();
    std::copy(depsSet.begin(), depsSet.end(), std::back_inserter(deps));
    return deps;
  }();

  const std::string objectDir = GeneratorTarget->ObjectDirectory;
  const std::string relObjectDir = LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), objectDir);

  // Construct a list of files associated with this executable that
  // may need to be cleaned.
  std::vector<std::string> cleanFiles;
  cleanFiles.push_back(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), output));

  std::string profiles;
  std::vector<std::string> fatbinaryDepends;
  std::string registerFile = cmStrCat(objectDir, "cmake_cuda_register.h");

  // Link device code for each architecture.
  for (const std::string& architectureKind : architectures) {
    // Clang always generates real code, so strip the specifier.
    const std::string architecture =
      architectureKind.substr(0, architectureKind.find('-'));
    const std::string cubin =
      cmStrCat(relObjectDir, "sm_", architecture, ".cubin");

    profiles += cmStrCat(" -im=profile=sm_", architecture, ",file=", cubin);
    fatbinaryDepends.emplace_back(cubin);

    std::string registerFileCmd;

    // The generated register file contains macros that when expanded register
    // the device routines. Because the routines are the same for all
    // architectures the register file will be the same too. Thus generate it
    // only on the first invocation to reduce overhead.
    if (fatbinaryDepends.size() == 1) {
      std::string registerFileRel = LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), registerFile);
      registerFileCmd =
        cmStrCat(" --register-link-binaries=", registerFileRel);
      cleanFiles.push_back(registerFileRel);
    }

    std::string command =
      cmStrCat(Makefile->GetRequiredDefinition("CMAKE_CUDA_DEVICE_LINKER"),
               " -arch=sm_", architecture, registerFileCmd, " -o=$@ ",
               cmJoin(linkDeps, " "));

    LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, cubin, linkDeps,
                                  { command }, false);
  }

  // Combine all architectures into a single fatbinary.
  const std::string fatbinaryCommand =
    cmStrCat(Makefile->GetRequiredDefinition("CMAKE_CUDA_FATBINARY"),
             " -64 -cmdline=--compile-only -compress-all -link "
             "--embedded-fatbin=$@",
             profiles);
  const std::string fatbinaryOutput =
    cmStrCat(objectDir, "cmake_cuda_fatbin.h");
  const std::string fatbinaryOutputRel =
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), fatbinaryOutput);

  LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, fatbinaryOutputRel,
                                fatbinaryDepends, { fatbinaryCommand }, false);

  // Compile the stub that registers the kernels and contains the fatbinaries.
  cmRulePlaceholderExpander::RuleVariables vars;
  vars.CMTargetName = GetGeneratorTarget()->GetName().c_str();
  vars.CMTargetType =
    cmState::GetTargetTypeName(GetGeneratorTarget()->GetType()).c_str();

  vars.Language = "CUDA";
  vars.Object = output.c_str();
  vars.Fatbinary = fatbinaryOutput.c_str();
  vars.RegisterFile = registerFile.c_str();

  std::string flags = GetFlags("CUDA", GetConfigName());
  vars.Flags = flags.c_str();

  std::string compileCmd = GetLinkRule("CMAKE_CUDA_DEVICE_LINK_COMPILE");
  std::unique_ptr<cmRulePlaceholderExpander> rulePlaceholderExpander(
    LocalGenerator->CreateRulePlaceholderExpander());
  rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator, compileCmd,
                                               vars);

  commands.emplace_back(compileCmd);
  LocalGenerator->WriteMakeRule(
    *BuildFileStream, nullptr, output,
    { cmStrCat(relObjectDir, "cmake_cuda_fatbin.h") }, commands, false);

  // Clean all the possible executable names and symlinks.
  CleanFiles.insert(cleanFiles.begin(), cleanFiles.end());
}

void cmMakefileTargetGenerator::GenerateCustomRuleFile(
  cmCustomCommandGenerator const& ccg)
{
  // Collect the commands.
  std::vector<std::string> commands;
  std::string comment = LocalGenerator->ConstructComment(ccg);
  if (!comment.empty()) {
    // add in a progress call if needed
    NumberOfProgressActions++;
    if (!NoRuleMessages) {
      cmLocalUnixMakefileGenerator3::EchoProgress progress;
      MakeEchoProgress(progress);
      LocalGenerator->AppendEcho(commands, comment,
                                 cmLocalUnixMakefileGenerator3::EchoGenerate,
                                 &progress);
    }
  }

  // Now append the actual user-specified commands.
  std::ostringstream content;
  LocalGenerator->AppendCustomCommand(commands, ccg, GeneratorTarget,
                                      LocalGenerator->GetBinaryDirectory(),
                                      false, &content);

  // Collect the dependencies.
  std::vector<std::string> depends;
  LocalGenerator->AppendCustomDepend(depends, ccg);

  // Write the rule.
  const std::vector<std::string>& outputs = ccg.GetOutputs();
  bool symbolic =
    WriteMakeRule(*BuildFileStream, nullptr, outputs, depends, commands);

  // Symbolic inputs are not expected to exist, so add dummy rules.
  if (CMP0113New && !depends.empty()) {
    std::vector<std::string> no_depends;
    std::vector<std::string> no_commands;
    for (std::string const& dep : depends) {
      if (cmSourceFile* dsf =
            Makefile->GetSource(dep, cmSourceFileLocationKind::Known)) {
        if (dsf->GetPropertyAsBool("SYMBOLIC")) {
          LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, dep,
                                        no_depends, no_commands, true);
        }
      }
    }
  }

  // If the rule has changed make sure the output is rebuilt.
  if (!symbolic) {
    GlobalGenerator->AddRuleHash(ccg.GetOutputs(), content.str());
  }

  // Setup implicit dependency scanning.
  for (auto const& idi : ccg.GetCC().GetImplicitDepends()) {
    std::string objFullPath = cmSystemTools::CollapseFullPath(
      outputs[0], LocalGenerator->GetCurrentBinaryDirectory());
    std::string srcFullPath = cmSystemTools::CollapseFullPath(
      idi.second, LocalGenerator->GetCurrentBinaryDirectory());
    LocalGenerator->AddImplicitDepends(GeneratorTarget, idi.first, objFullPath,
                                       srcFullPath);
  }

  CustomCommandOutputs.insert(outputs.begin(), outputs.end());
}

void cmMakefileTargetGenerator::MakeEchoProgress(
  cmLocalUnixMakefileGenerator3::EchoProgress& progress) const
{
  progress.Dir = cmStrCat(LocalGenerator->GetBinaryDirectory(), "/CMakeFiles");
  std::ostringstream progressArg;
  progressArg << "$(CMAKE_PROGRESS_" << NumberOfProgressActions << ")";
  progress.Arg = progressArg.str();
}

void cmMakefileTargetGenerator::WriteObjectsVariable(
  std::string& variableName, std::string& variableNameExternal,
  bool useWatcomQuote)
{
  // Write a make variable assignment that lists all objects for the
  // target.
  variableName =
    LocalGenerator->CreateMakeVariable(GeneratorTarget->GetName(), "_OBJECTS");
  *BuildFileStream << "# Object files for target "
                   << GeneratorTarget->GetName() << "\n"
                   << variableName << " =";
  std::string object;
  std::string lineContinue;
  if (cmProp p = Makefile->GetDefinition("CMAKE_MAKE_LINE_CONTINUE")) {
    lineContinue = *p;
  } else {
    lineContinue = "\\";
  }

  cmProp pchExtension = Makefile->GetDefinition("CMAKE_PCH_EXTENSION");

  for (std::string const& obj : Objects) {
    if (cmSystemTools::StringEndsWith(obj, cmToCStr(pchExtension))) {
      continue;
    }
    *BuildFileStream << " " << lineContinue << "\n";
    *BuildFileStream
      << cmLocalUnixMakefileGenerator3::ConvertToQuotedOutputPath(
           obj, useWatcomQuote);
  }
  *BuildFileStream << "\n";

  // Write a make variable assignment that lists all external objects
  // for the target.
  variableNameExternal = LocalGenerator->CreateMakeVariable(
    GeneratorTarget->GetName(), "_EXTERNAL_OBJECTS");
  /* clang-format off */
  *BuildFileStream
    << "\n"
    << "# External object files for target "
    << GeneratorTarget->GetName() << "\n"
    << variableNameExternal << " =";
  /* clang-format on */
  std::string currentBinDir = LocalGenerator->GetCurrentBinaryDirectory();
  for (std::string const& obj : ExternalObjects) {
    object = LocalGenerator->MaybeConvertToRelativePath(currentBinDir, obj);
    *BuildFileStream << " " << lineContinue << "\n";
    *BuildFileStream
      << cmLocalUnixMakefileGenerator3::ConvertToQuotedOutputPath(
           obj, useWatcomQuote);
  }
  *BuildFileStream << "\n"
                   << "\n";
}

class cmMakefileTargetGeneratorObjectStrings
{
public:
  cmMakefileTargetGeneratorObjectStrings(std::vector<std::string>& strings,
                                         cmOutputConverter* outputConverter,
                                         cmStateDirectory const& stateDir,
                                         std::string::size_type limit)
    : Strings(strings)
    , OutputConverter(outputConverter)
    , StateDir(stateDir)
    , LengthLimit(limit)
  {
    Space = "";
  }
  void Feed(std::string const& obj)
  {
    // Construct the name of the next object.
    NextObject = OutputConverter->ConvertToOutputFormat(
      MaybeConvertToRelativePath(obj), cmOutputConverter::RESPONSE);

    // Roll over to next string if the limit will be exceeded.
    if (LengthLimit != std::string::npos &&
        (CurrentString.length() + 1 + NextObject.length() > LengthLimit)) {
      Strings.push_back(CurrentString);
      CurrentString.clear();
      Space = "";
    }

    // Separate from previous object.
    CurrentString += Space;
    Space = " ";

    // Append this object.
    CurrentString += NextObject;
  }
  void Done() { Strings.push_back(CurrentString); }

private:
  std::string MaybeConvertToRelativePath(std::string const& obj)
  {
    if (!StateDir.ContainsBoth(StateDir.GetCurrentBinary(), obj)) {
      return obj;
    }
    return cmSystemTools::ForceToRelativePath(StateDir.GetCurrentBinary(),
                                              obj);
  }

  std::vector<std::string>& Strings;
  cmOutputConverter* OutputConverter;
  cmStateDirectory StateDir;
  std::string::size_type LengthLimit;
  std::string CurrentString;
  std::string NextObject;
  const char* Space;
};

void cmMakefileTargetGenerator::WriteObjectsStrings(
  std::vector<std::string>& objStrings, std::string::size_type limit)
{
  cmProp pchExtension = Makefile->GetDefinition("CMAKE_PCH_EXTENSION");

  cmMakefileTargetGeneratorObjectStrings helper(
    objStrings, LocalGenerator,
    LocalGenerator->GetStateSnapshot().GetDirectory(), limit);
  for (std::string const& obj : Objects) {
    if (cmSystemTools::StringEndsWith(obj, cmToCStr(pchExtension))) {
      continue;
    }
    helper.Feed(obj);
  }
  for (std::string const& obj : ExternalObjects) {
    helper.Feed(obj);
  }
  auto ispcAdditionalObjs =
    GeneratorTarget->GetGeneratedISPCObjects(GetConfigName());
  for (std::string const& obj : ispcAdditionalObjs) {
    helper.Feed(obj);
  }
  helper.Done();
}

void cmMakefileTargetGenerator::WriteTargetDriverRule(
  const std::string& main_output, bool relink)
{
  // Compute the name of the driver target.
  std::string dir =
    LocalGenerator->GetRelativeTargetDirectory(GeneratorTarget);
  std::string buildTargetRuleName =
    cmStrCat(dir, relink ? "/preinstall" : "/build");
  buildTargetRuleName = LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetBinaryDirectory(), buildTargetRuleName);

  // Build the list of target outputs to drive.
  std::vector<std::string> depends;
  depends.push_back(main_output);

  const char* comment = nullptr;
  if (relink) {
    // Setup the comment for the preinstall driver.
    comment = "Rule to relink during preinstall.";
  } else {
    // Setup the comment for the main build driver.
    comment = "Rule to build all files generated by this target.";

    // Make sure all custom command outputs in this target are built.
    if (CustomCommandDriver == OnBuild) {
      DriveCustomCommands(depends);
    }

    // Make sure the extra files are built.
    cm::append(depends, ExtraFiles);
  }

  // Write the driver rule.
  std::vector<std::string> no_commands;
  LocalGenerator->WriteMakeRule(*BuildFileStream, comment, buildTargetRuleName,
                                depends, no_commands, true);
}

void cmMakefileTargetGenerator::AppendTargetDepends(
  std::vector<std::string>& depends, bool ignoreType)
{
  // Static libraries never depend on anything for linking.
  if (GeneratorTarget->GetType() == cmStateEnums::STATIC_LIBRARY &&
      !ignoreType) {
    return;
  }

  // Loop over all library dependencies.
  const std::string& cfg = GetConfigName();
  if (cmComputeLinkInformation* cli =
        GeneratorTarget->GetLinkInformation(cfg)) {
    cm::append(depends, cli->GetDepends());
  }
}

void cmMakefileTargetGenerator::AppendObjectDepends(
  std::vector<std::string>& depends)
{
  // Add dependencies on the compiled object files.
  std::string const& relPath = LocalGenerator->GetHomeRelativeOutputPath();
  for (std::string const& obj : Objects) {
    std::string objTarget = cmStrCat(relPath, obj);
    depends.push_back(std::move(objTarget));
  }

  // Add dependencies on the external object files.
  cm::append(depends, ExternalObjects);

  // Add a dependency on the rule file itself.
  LocalGenerator->AppendRuleDepend(depends, BuildFileNameFull.c_str());
}

void cmMakefileTargetGenerator::AppendLinkDepends(
  std::vector<std::string>& depends, const std::string& linkLanguage)
{
  AppendObjectDepends(depends);

  // Add dependencies on targets that must be built first.
  AppendTargetDepends(depends);

  // Add a dependency on the link definitions file, if any.
  if (cmGeneratorTarget::ModuleDefinitionInfo const* mdi =
        GeneratorTarget->GetModuleDefinitionInfo(GetConfigName())) {
    for (cmSourceFile const* src : mdi->Sources) {
      depends.push_back(src->GetFullPath());
    }
  }

  // Add a dependency on user-specified manifest files, if any.
  std::vector<cmSourceFile const*> manifest_srcs;
  GeneratorTarget->GetManifests(manifest_srcs, GetConfigName());
  for (cmSourceFile const* manifest_src : manifest_srcs) {
    depends.push_back(manifest_src->GetFullPath());
  }

  // Add user-specified dependencies.
  GeneratorTarget->GetLinkDepends(depends, GetConfigName(), linkLanguage);
}

std::string cmMakefileTargetGenerator::GetLinkRule(
  const std::string& linkRuleVar)
{
  std::string linkRule = Makefile->GetRequiredDefinition(linkRuleVar);
  if (GeneratorTarget->HasImplibGNUtoMS(GetConfigName())) {
    std::string ruleVar =
      cmStrCat("CMAKE_", GeneratorTarget->GetLinkerLanguage(GetConfigName()),
               "_GNUtoMS_RULE");
    if (cmProp rule = Makefile->GetDefinition(ruleVar)) {
      linkRule += *rule;
    }
  }
  return linkRule;
}

void cmMakefileTargetGenerator::CloseFileStreams()
{
  BuildFileStream.reset();
  InfoFileStream.reset();
  FlagFileStream.reset();
}

void cmMakefileTargetGenerator::CreateLinkScript(
  const char* name, std::vector<std::string> const& link_commands,
  std::vector<std::string>& makefile_commands,
  std::vector<std::string>& makefile_depends)
{
  // Create the link script file.
  std::string linkScriptName = cmStrCat(TargetBuildDirectoryFull, '/', name);
  cmGeneratedFileStream linkScriptStream(linkScriptName);
  linkScriptStream.SetCopyIfDifferent(true);
  for (std::string const& link_command : link_commands) {
    // Do not write out empty commands or commands beginning in the
    // shell no-op ":".
    if (!link_command.empty() && link_command[0] != ':') {
      linkScriptStream << link_command << "\n";
    }
  }

  // Create the makefile command to invoke the link script.
  std::string link_command =
    cmStrCat("$(CMAKE_COMMAND) -E cmake_link_script ",
             LocalGenerator->ConvertToOutputFormat(
               LocalGenerator->MaybeConvertToRelativePath(
                 LocalGenerator->GetCurrentBinaryDirectory(), linkScriptName),
               cmOutputConverter::SHELL),
             " --verbose=$(VERBOSE)");
  makefile_commands.push_back(std::move(link_command));
  makefile_depends.push_back(std::move(linkScriptName));
}

bool cmMakefileTargetGenerator::CheckUseResponseFileForObjects(
  std::string const& l) const
{
  // Check for an explicit setting one way or the other.
  std::string const responseVar =
    "CMAKE_" + l + "_USE_RESPONSE_FILE_FOR_OBJECTS";
  if (cmProp val = Makefile->GetDefinition(responseVar)) {
    if (!val->empty()) {
      return cmIsOn(val);
    }
  }

  // Check for a system limit.
  if (size_t const limit = cmSystemTools::CalculateCommandLineLengthLimit()) {
    // Compute the total length of our list of object files with room
    // for argument separation and quoting.  This does not convert paths
    // relative to CMAKE_CURRENT_BINARY_DIR like the final list will be, so the
    // actual list will likely be much shorter than this.  However, in the
    // worst case all objects will remain as absolute paths.
    size_t length = 0;
    for (std::string const& obj : Objects) {
      length += obj.size() + 3;
    }
    for (std::string const& ext_obj : ExternalObjects) {
      length += ext_obj.size() + 3;
    }

    // We need to guarantee room for both objects and libraries, so
    // if the objects take up more than half then use a response file
    // for them.
    if (length > (limit / 2)) {
      return true;
    }
  }

  // We do not need a response file for objects.
  return false;
}

bool cmMakefileTargetGenerator::CheckUseResponseFileForLibraries(
  std::string const& l) const
{
  // Check for an explicit setting one way or the other.
  std::string const responseVar =
    "CMAKE_" + l + "_USE_RESPONSE_FILE_FOR_LIBRARIES";
  if (cmProp val = Makefile->GetDefinition(responseVar)) {
    if (!val->empty()) {
      return cmIsOn(val);
    }
  }

  // We do not need a response file for libraries.
  return false;
}

std::string cmMakefileTargetGenerator::CreateResponseFile(
  const char* name, std::string const& options,
  std::vector<std::string>& makefile_depends)
{
  // Create the response file.
  std::string responseFileNameFull =
    cmStrCat(TargetBuildDirectoryFull, '/', name);
  cmGeneratedFileStream responseStream(responseFileNameFull);
  responseStream.SetCopyIfDifferent(true);
  responseStream << options << "\n";

  // Add a dependency so the target will rebuild when the set of
  // objects changes.
  makefile_depends.push_back(std::move(responseFileNameFull));

  // Construct the name to be used on the command line.
  std::string responseFileName = cmStrCat(TargetBuildDirectory, '/', name);
  return responseFileName;
}

std::unique_ptr<cmLinkLineComputer>
cmMakefileTargetGenerator::CreateLinkLineComputer(
  cmOutputConverter* outputConverter, cmStateDirectory const& stateDir)
{
  if (Makefile->IsOn("MSVC60")) {
    return GlobalGenerator->CreateMSVC60LinkLineComputer(outputConverter,
                                                         stateDir);
  }
  return GlobalGenerator->CreateLinkLineComputer(outputConverter, stateDir);
}

void cmMakefileTargetGenerator::CreateLinkLibs(
  cmLinkLineComputer* linkLineComputer, std::string& linkLibs,
  bool useResponseFile, std::vector<std::string>& makefile_depends)
{
  std::string frameworkPath;
  std::string linkPath;
  cmComputeLinkInformation* pcli =
    GeneratorTarget->GetLinkInformation(GetConfigName());
  LocalGenerator->OutputLinkLibraries(pcli, linkLineComputer, linkLibs,
                                      frameworkPath, linkPath);
  linkLibs = frameworkPath + linkPath + linkLibs;

  if (useResponseFile &&
      linkLibs.find_first_not_of(' ') != std::string::npos) {
    // Lookup the response file reference flag.
    std::string responseFlagVar =
      cmStrCat("CMAKE_", GeneratorTarget->GetLinkerLanguage(GetConfigName()),
               "_RESPONSE_FILE_LINK_FLAG");
    std::string responseFlag;
    if (cmProp p = Makefile->GetDefinition(responseFlagVar)) {
      responseFlag = *p;
    } else {
      responseFlag = "@";
    }

    // Create this response file.
    std::string link_rsp =
      CreateResponseFile("linklibs.rsp", linkLibs, makefile_depends);

    // Reference the response file.
    linkLibs = cmStrCat(responseFlag,
                        LocalGenerator->ConvertToOutputFormat(
                          link_rsp, cmOutputConverter::SHELL));
  }
}

void cmMakefileTargetGenerator::CreateObjectLists(
  bool useLinkScript, bool useArchiveRules, bool useResponseFile,
  std::string& buildObjs, std::vector<std::string>& makefile_depends,
  bool useWatcomQuote)
{
  std::string variableName;
  std::string variableNameExternal;
  WriteObjectsVariable(variableName, variableNameExternal, useWatcomQuote);
  if (useResponseFile) {
    // MSVC response files cannot exceed 128K.
    std::string::size_type const responseFileLimit = 131000;

    // Construct the individual object list strings.
    std::vector<std::string> object_strings;
    WriteObjectsStrings(object_strings, responseFileLimit);

    // Lookup the response file reference flag.
    std::string responseFlagVar =
      cmStrCat("CMAKE_", GeneratorTarget->GetLinkerLanguage(GetConfigName()),
               "_RESPONSE_FILE_LINK_FLAG");
    std::string responseFlag;
    if (cmProp p = Makefile->GetDefinition(responseFlagVar)) {
      responseFlag = *p;
    } else {
      responseFlag = "@";
    }

    // Write a response file for each string.
    const char* sep = "";
    for (unsigned int i = 0; i < object_strings.size(); ++i) {
      // Number the response files.
      char rsp[32];
      sprintf(rsp, "objects%u.rsp", i + 1);

      // Create this response file.
      std::string objects_rsp =
        CreateResponseFile(rsp, object_strings[i], makefile_depends);

      // Separate from previous response file references.
      buildObjs += sep;
      sep = " ";

      // Reference the response file.
      buildObjs += responseFlag;
      buildObjs += LocalGenerator->ConvertToOutputFormat(
        objects_rsp, cmOutputConverter::SHELL);
    }
  } else if (useLinkScript) {
    if (!useArchiveRules) {
      std::vector<std::string> objStrings;
      WriteObjectsStrings(objStrings);
      buildObjs = objStrings[0];
    }
  } else {
    buildObjs =
      cmStrCat("$(", variableName, ") $(", variableNameExternal, ')');
  }
}

void cmMakefileTargetGenerator::AddIncludeFlags(std::string& flags,
                                                const std::string& lang,
                                                const std::string& /*config*/)
{
  std::string responseVar =
    cmStrCat("CMAKE_", lang, "_USE_RESPONSE_FILE_FOR_INCLUDES");
  bool useResponseFile = Makefile->IsOn(responseVar);

  std::vector<std::string> includes;
  LocalGenerator->GetIncludeDirectories(includes, GeneratorTarget, lang,
                                        GetConfigName());

  std::string includeFlags = LocalGenerator->GetIncludeFlags(
    includes, GeneratorTarget, lang, false, useResponseFile, GetConfigName());
  if (includeFlags.empty()) {
    return;
  }

  if (useResponseFile) {
    std::string const responseFlagVar =
      "CMAKE_" + lang + "_RESPONSE_FILE_FLAG";
    std::string responseFlag = Makefile->GetSafeDefinition(responseFlagVar);
    if (responseFlag.empty()) {
      responseFlag = "@";
    }
    std::string name = cmStrCat("includes_", lang, ".rsp");
    std::string arg = std::move(responseFlag) +
      CreateResponseFile(name.c_str(), includeFlags, FlagFileDepends[lang]);
    LocalGenerator->AppendFlags(flags, arg);
  } else {
    LocalGenerator->AppendFlags(flags, includeFlags);
  }
}

void cmMakefileTargetGenerator::GenDefFile(
  std::vector<std::string>& real_link_commands)
{
  cmGeneratorTarget::ModuleDefinitionInfo const* mdi =
    GeneratorTarget->GetModuleDefinitionInfo(GetConfigName());
  if (!mdi || !mdi->DefFileGenerated) {
    return;
  }
  std::string cmd = cmSystemTools::GetCMakeCommand();
  cmd = cmStrCat(
    LocalGenerator->ConvertToOutputFormat(cmd, cmOutputConverter::SHELL),
    " -E __create_def ",
    LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), mdi->DefFile),
      cmOutputConverter::SHELL),
    ' ');
  std::string objlist_file = mdi->DefFile + ".objs";
  cmd += LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), objlist_file),
    cmOutputConverter::SHELL);
  cmProp nm_executable = Makefile->GetDefinition("CMAKE_NM");
  if (cmNonempty(nm_executable)) {
    cmd += " --nm=";
    cmd += LocalCommonGenerator->ConvertToOutputFormat(
      *nm_executable, cmOutputConverter::SHELL);
  }
  real_link_commands.insert(real_link_commands.begin(), cmd);
  // create a list of obj files for the -E __create_def to read
  cmGeneratedFileStream fout(objlist_file);

  if (mdi->WindowsExportAllSymbols) {
    for (std::string const& obj : Objects) {
      if (cmHasLiteralSuffix(obj, ".obj")) {
        fout << obj << "\n";
      }
    }
    for (std::string const& obj : ExternalObjects) {
      fout << obj << "\n";
    }
  }

  for (cmSourceFile const* src : mdi->Sources) {
    fout << src->GetFullPath() << "\n";
  }
}
