/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmMakefileLibraryTargetGenerator.h"

#include <cstddef>
#include <set>
#include <sstream>
#include <utility>
#include <vector>

#include <cm/memory>
#include <cmext/algorithm>

#include "cmGeneratedFileStream.h"
#include "cmGeneratorTarget.h"
#include "cmGlobalUnixMakefileGenerator3.h"
#include "cmLinkLineComputer.h"
#include "cmLinkLineDeviceComputer.h"
#include "cmLocalGenerator.h"
#include "cmLocalUnixMakefileGenerator3.h"
#include "cmMakefile.h"
#include "cmOSXBundleGenerator.h"
#include "cmOutputConverter.h"
#include "cmRulePlaceholderExpander.h"
#include "cmState.h"
#include "cmStateDirectory.h"
#include "cmStateSnapshot.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

cmMakefileLibraryTargetGenerator::cmMakefileLibraryTargetGenerator(
  cmGeneratorTarget* target)
  : cmMakefileTargetGenerator(target)
{
  CustomCommandDriver = OnDepends;
  if (GeneratorTarget->GetType() != cmStateEnums::INTERFACE_LIBRARY) {
    TargetNames = GeneratorTarget->GetLibraryNames(GetConfigName());
  }

  OSXBundleGenerator = cm::make_unique<cmOSXBundleGenerator>(target);
  OSXBundleGenerator->SetMacContentFolders(&MacContentFolders);
}

cmMakefileLibraryTargetGenerator::~cmMakefileLibraryTargetGenerator() =
  default;

void cmMakefileLibraryTargetGenerator::WriteRuleFiles()
{
  // create the build.make file and directory, put in the common blocks
  CreateRuleFile();

  // write rules used to help build object files
  WriteCommonCodeRules();

  // write the per-target per-language flags
  WriteTargetLanguageFlags();

  // write in rules for object files and custom commands
  WriteTargetBuildRules();

  // write the link rules
  // Write the rule for this target type.
  switch (GeneratorTarget->GetType()) {
    case cmStateEnums::STATIC_LIBRARY:
      WriteStaticLibraryRules();
      break;
    case cmStateEnums::SHARED_LIBRARY:
      WriteSharedLibraryRules(false);
      if (GeneratorTarget->NeedRelinkBeforeInstall(GetConfigName())) {
        // Write rules to link an installable version of the target.
        WriteSharedLibraryRules(true);
      }
      break;
    case cmStateEnums::MODULE_LIBRARY:
      WriteModuleLibraryRules(false);
      if (GeneratorTarget->NeedRelinkBeforeInstall(GetConfigName())) {
        // Write rules to link an installable version of the target.
        WriteModuleLibraryRules(true);
      }
      break;
    case cmStateEnums::OBJECT_LIBRARY:
      WriteObjectLibraryRules();
      break;
    default:
      // If language is not known, this is an error.
      cmSystemTools::Error("Unknown Library Type");
      break;
  }

  // Write clean target
  WriteTargetCleanRules();

  // Write the dependency generation rule.  This must be done last so
  // that multiple output pair information is available.
  WriteTargetDependRules();

  // close the streams
  CloseFileStreams();
}

void cmMakefileLibraryTargetGenerator::WriteObjectLibraryRules()
{
  std::vector<std::string> commands;
  std::vector<std::string> depends;

  // Add post-build rules.
  LocalGenerator->AppendCustomCommands(
    commands, GeneratorTarget->GetPostBuildCommands(), GeneratorTarget,
    LocalGenerator->GetBinaryDirectory());

  // Depend on the object files.
  AppendObjectDepends(depends);

  // Write the rule.
  LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr,
                                GeneratorTarget->GetName(), depends, commands,
                                true);

  // Write the main driver rule to build everything in this target.
  WriteTargetDriverRule(GeneratorTarget->GetName(), false);
}

void cmMakefileLibraryTargetGenerator::WriteStaticLibraryRules()
{
  const bool requiresDeviceLinking =
    requireDeviceLinking(*GeneratorTarget, *LocalGenerator, GetConfigName());
  if (requiresDeviceLinking) {
    WriteDeviceLibraryRules("CMAKE_CUDA_DEVICE_LINK_LIBRARY", false);
  }

  std::string linkLanguage =
    GeneratorTarget->GetLinkerLanguage(GetConfigName());

  std::string linkRuleVar =
    GeneratorTarget->GetCreateRuleVariable(linkLanguage, GetConfigName());

  std::string extraFlags;
  LocalGenerator->GetStaticLibraryFlags(extraFlags, GetConfigName(),
                                        linkLanguage, GeneratorTarget);
  WriteLibraryRules(linkRuleVar, extraFlags, false);
}

void cmMakefileLibraryTargetGenerator::WriteSharedLibraryRules(bool relink)
{
  if (GeneratorTarget->IsFrameworkOnApple()) {
    WriteFrameworkRules(relink);
    return;
  }

  if (!relink) {
    const bool requiresDeviceLinking =
      requireDeviceLinking(*GeneratorTarget, *LocalGenerator, GetConfigName());
    if (requiresDeviceLinking) {
      WriteDeviceLibraryRules("CMAKE_CUDA_DEVICE_LINK_LIBRARY", relink);
    }
  }

  std::string linkLanguage =
    GeneratorTarget->GetLinkerLanguage(GetConfigName());
  std::string linkRuleVar =
    cmStrCat("CMAKE_", linkLanguage, "_CREATE_SHARED_LIBRARY");

  std::string extraFlags;
  GetTargetLinkFlags(extraFlags, linkLanguage);
  LocalGenerator->AddConfigVariableFlags(
    extraFlags, "CMAKE_SHARED_LINKER_FLAGS", GetConfigName());

  std::unique_ptr<cmLinkLineComputer> linkLineComputer =
    CreateLinkLineComputer(LocalGenerator,
                           LocalGenerator->GetStateSnapshot().GetDirectory());

  AddModuleDefinitionFlag(linkLineComputer.get(), extraFlags, GetConfigName());

  if (GeneratorTarget->GetPropertyAsBool("LINK_WHAT_YOU_USE")) {
    LocalGenerator->AppendFlags(extraFlags, " -Wl,--no-as-needed");
  }
  WriteLibraryRules(linkRuleVar, extraFlags, relink);
}

void cmMakefileLibraryTargetGenerator::WriteModuleLibraryRules(bool relink)
{
  if (!relink) {
    const bool requiresDeviceLinking =
      requireDeviceLinking(*GeneratorTarget, *LocalGenerator, GetConfigName());
    if (requiresDeviceLinking) {
      WriteDeviceLibraryRules("CMAKE_CUDA_DEVICE_LINK_LIBRARY", relink);
    }
  }

  std::string linkLanguage =
    GeneratorTarget->GetLinkerLanguage(GetConfigName());
  std::string linkRuleVar =
    cmStrCat("CMAKE_", linkLanguage, "_CREATE_SHARED_MODULE");

  std::string extraFlags;
  GetTargetLinkFlags(extraFlags, linkLanguage);
  LocalGenerator->AddConfigVariableFlags(
    extraFlags, "CMAKE_MODULE_LINKER_FLAGS", GetConfigName());

  std::unique_ptr<cmLinkLineComputer> linkLineComputer =
    CreateLinkLineComputer(LocalGenerator,
                           LocalGenerator->GetStateSnapshot().GetDirectory());

  AddModuleDefinitionFlag(linkLineComputer.get(), extraFlags, GetConfigName());

  WriteLibraryRules(linkRuleVar, extraFlags, relink);
}

void cmMakefileLibraryTargetGenerator::WriteFrameworkRules(bool relink)
{
  std::string linkLanguage =
    GeneratorTarget->GetLinkerLanguage(GetConfigName());
  std::string linkRuleVar =
    cmStrCat("CMAKE_", linkLanguage, "_CREATE_MACOSX_FRAMEWORK");

  std::string extraFlags;
  GetTargetLinkFlags(extraFlags, linkLanguage);
  LocalGenerator->AddConfigVariableFlags(
    extraFlags, "CMAKE_MACOSX_FRAMEWORK_LINKER_FLAGS", GetConfigName());

  WriteLibraryRules(linkRuleVar, extraFlags, relink);
}

void cmMakefileLibraryTargetGenerator::WriteDeviceLibraryRules(
  const std::string& linkRuleVar, bool relink)
{
#ifndef CMAKE_BOOTSTRAP
  // TODO: Merge the methods that call this method to avoid
  // code duplication.
  std::vector<std::string> commands;
  std::string const objExt =
    Makefile->GetSafeDefinition("CMAKE_CUDA_OUTPUT_EXTENSION");

  // Get the name of the device object to generate.
  std::string const targetOutput =
    GeneratorTarget->ObjectDirectory + "cmake_device_link" + objExt;
  DeviceLinkObject = targetOutput;

  NumberOfProgressActions++;
  if (!NoRuleMessages) {
    cmLocalUnixMakefileGenerator3::EchoProgress progress;
    MakeEchoProgress(progress);
    // Add the link message.
    std::string buildEcho = cmStrCat(
      "Linking CUDA device code ",
      LocalGenerator->ConvertToOutputFormat(
        LocalGenerator->MaybeConvertToRelativePath(
          LocalGenerator->GetCurrentBinaryDirectory(), DeviceLinkObject),
        cmOutputConverter::SHELL));
    LocalGenerator->AppendEcho(
      commands, buildEcho, cmLocalUnixMakefileGenerator3::EchoLink, &progress);
  }

  if (Makefile->GetSafeDefinition("CMAKE_CUDA_COMPILER_ID") == "Clang") {
    WriteDeviceLinkRule(commands, targetOutput);
  } else {
    WriteNvidiaDeviceLibraryRules(linkRuleVar, relink, commands, targetOutput);
  }

  // Write the main driver rule to build everything in this target.
  WriteTargetDriverRule(targetOutput, relink);
}

void cmMakefileLibraryTargetGenerator::WriteNvidiaDeviceLibraryRules(
  const std::string& linkRuleVar, bool relink,
  std::vector<std::string>& commands, const std::string& targetOutput)
{
  std::string linkLanguage = "CUDA";

  // Build list of dependencies.
  std::vector<std::string> depends;
  AppendLinkDepends(depends, linkLanguage);

  // Add language-specific flags.
  std::string langFlags;
  LocalGenerator->AddLanguageFlagsForLinking(langFlags, GeneratorTarget,
                                             linkLanguage, GetConfigName());

  // Create set of linking flags.
  std::string linkFlags;
  GetDeviceLinkFlags(linkFlags, linkLanguage);

  // Clean files associated with this library.
  std::set<std::string> libCleanFiles;
  libCleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), targetOutput));

  // Determine whether a link script will be used.
  bool useLinkScript = GlobalGenerator->GetUseLinkScript();

  bool useResponseFileForObjects =
    CheckUseResponseFileForObjects(linkLanguage);
  bool const useResponseFileForLibs =
    CheckUseResponseFileForLibraries(linkLanguage);

  cmRulePlaceholderExpander::RuleVariables vars;
  vars.Language = linkLanguage.c_str();

  // Expand the rule variables.
  std::vector<std::string> real_link_commands;
  {
    bool useWatcomQuote = Makefile->IsOn(linkRuleVar + "_USE_WATCOM_QUOTE");

    // Set path conversion for link script shells.
    LocalGenerator->SetLinkScriptShell(useLinkScript);

    // Collect up flags to link in needed libraries.
    std::string linkLibs;
    std::unique_ptr<cmLinkLineComputer> linkLineComputer(
      new cmLinkLineDeviceComputer(
        LocalGenerator, LocalGenerator->GetStateSnapshot().GetDirectory()));
    linkLineComputer->SetForResponse(useResponseFileForLibs);
    linkLineComputer->SetUseWatcomQuote(useWatcomQuote);
    linkLineComputer->SetRelink(relink);

    CreateLinkLibs(linkLineComputer.get(), linkLibs, useResponseFileForLibs,
                   depends);

    // Construct object file lists that may be needed to expand the
    // rule.
    std::string buildObjs;
    CreateObjectLists(useLinkScript, false, // useArchiveRules
                      useResponseFileForObjects, buildObjs, depends,
                      useWatcomQuote);

    cmOutputConverter::OutputFormat output = (useWatcomQuote)
      ? cmOutputConverter::WATCOMQUOTE
      : cmOutputConverter::SHELL;

    std::string objectDir = GeneratorTarget->GetSupportDirectory();
    objectDir = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), objectDir),
      cmOutputConverter::SHELL);

    std::string target = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), targetOutput),
      output);

    std::string targetFullPathCompilePDB =
      ComputeTargetCompilePDB(GetConfigName());
    std::string targetOutPathCompilePDB =
      LocalGenerator->ConvertToOutputFormat(targetFullPathCompilePDB,
                                            cmOutputConverter::SHELL);

    vars.Objects = buildObjs.c_str();
    vars.ObjectDir = objectDir.c_str();
    vars.Target = target.c_str();
    vars.LinkLibraries = linkLibs.c_str();
    vars.ObjectsQuoted = buildObjs.c_str();
    vars.LanguageCompileFlags = langFlags.c_str();
    vars.LinkFlags = linkFlags.c_str();
    vars.TargetCompilePDB = targetOutPathCompilePDB.c_str();

    std::string launcher;
    const char* val =
      LocalGenerator->GetRuleLauncher(GeneratorTarget, "RULE_LAUNCH_LINK");
    if (cmNonempty(val)) {
      launcher = cmStrCat(val, ' ');
    }

    std::unique_ptr<cmRulePlaceholderExpander> rulePlaceholderExpander(
      LocalGenerator->CreateRulePlaceholderExpander());

    // Construct the main link rule and expand placeholders.
    rulePlaceholderExpander->SetTargetImpLib(targetOutput);
    std::string linkRule = GetLinkRule(linkRuleVar);
    cmExpandList(linkRule, real_link_commands);

    // Expand placeholders.
    for (std::string& real_link_command : real_link_commands) {
      real_link_command = cmStrCat(launcher, real_link_command);
      rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                   real_link_command, vars);
    }
    // Restore path conversion to normal shells.
    LocalGenerator->SetLinkScriptShell(false);

    // Clean all the possible library names and symlinks.
    CleanFiles.insert(libCleanFiles.begin(), libCleanFiles.end());
  }

  std::vector<std::string> commands1;
  // Optionally convert the build rule to use a script to avoid long
  // command lines in the make shell.
  if (useLinkScript) {
    // Use a link script.
    const char* name = (relink ? "drelink.txt" : "dlink.txt");
    CreateLinkScript(name, real_link_commands, commands1, depends);
  } else {
    // No link script.  Just use the link rule directly.
    commands1 = real_link_commands;
  }
  LocalGenerator->CreateCDCommand(commands1,
                                  Makefile->GetCurrentBinaryDirectory(),
                                  LocalGenerator->GetBinaryDirectory());
  cm::append(commands, commands1);
  commands1.clear();

  // Compute the list of outputs.
  std::vector<std::string> outputs(1, targetOutput);

  // Write the build rule.
  WriteMakeRule(*BuildFileStream, nullptr, outputs, depends, commands, false);
#else
  static_cast<void>(linkRuleVar);
  static_cast<void>(relink);
#endif
}

void cmMakefileLibraryTargetGenerator::WriteLibraryRules(
  const std::string& linkRuleVar, const std::string& extraFlags, bool relink)
{
  // TODO: Merge the methods that call this method to avoid
  // code duplication.
  std::vector<std::string> commands;

  // Get the language to use for linking this library.
  std::string linkLanguage =
    GeneratorTarget->GetLinkerLanguage(GetConfigName());

  // Make sure we have a link language.
  if (linkLanguage.empty()) {
    cmSystemTools::Error("Cannot determine link language for target \"" +
                         GeneratorTarget->GetName() + "\".");
    return;
  }

  // Build list of dependencies.
  std::vector<std::string> depends;
  AppendLinkDepends(depends, linkLanguage);
  if (!DeviceLinkObject.empty()) {
    depends.push_back(DeviceLinkObject);
  }

  // Create set of linking flags.
  std::string linkFlags;
  LocalGenerator->AppendFlags(linkFlags, extraFlags);
  LocalGenerator->AppendIPOLinkerFlags(linkFlags, GeneratorTarget,
                                       GetConfigName(), linkLanguage);

  // Add OSX version flags, if any.
  if (GeneratorTarget->GetType() == cmStateEnums::SHARED_LIBRARY ||
      GeneratorTarget->GetType() == cmStateEnums::MODULE_LIBRARY) {
    AppendOSXVerFlag(linkFlags, linkLanguage, "COMPATIBILITY", true);
    AppendOSXVerFlag(linkFlags, linkLanguage, "CURRENT", false);
  }

  // Construct the name of the library.
  GeneratorTarget->GetLibraryNames(GetConfigName());

  // Construct the full path version of the names.
  std::string outpath;
  std::string outpathImp;
  if (GeneratorTarget->IsFrameworkOnApple()) {
    outpath = GeneratorTarget->GetDirectory(GetConfigName());
    OSXBundleGenerator->CreateFramework(TargetNames.Output, outpath,
                                        GetConfigName());
    outpath += '/';
  } else if (GeneratorTarget->IsCFBundleOnApple()) {
    outpath = GeneratorTarget->GetDirectory(GetConfigName());
    OSXBundleGenerator->CreateCFBundle(TargetNames.Output, outpath,
                                       GetConfigName());
    outpath += '/';
  } else if (relink) {
    outpath = cmStrCat(Makefile->GetCurrentBinaryDirectory(),
                       "/CMakeFiles/CMakeRelink.dir");
    cmSystemTools::MakeDirectory(outpath);
    outpath += '/';
    if (!TargetNames.ImportLibrary.empty()) {
      outpathImp = outpath;
    }
  } else {
    outpath = GeneratorTarget->GetDirectory(GetConfigName());
    cmSystemTools::MakeDirectory(outpath);
    outpath += '/';
    if (!TargetNames.ImportLibrary.empty()) {
      outpathImp = GeneratorTarget->GetDirectory(
        GetConfigName(), cmStateEnums::ImportLibraryArtifact);
      cmSystemTools::MakeDirectory(outpathImp);
      outpathImp += '/';
    }
  }

  std::string compilePdbOutputPath =
    GeneratorTarget->GetCompilePDBDirectory(GetConfigName());
  cmSystemTools::MakeDirectory(compilePdbOutputPath);

  std::string pdbOutputPath =
    GeneratorTarget->GetPDBDirectory(GetConfigName());
  cmSystemTools::MakeDirectory(pdbOutputPath);
  pdbOutputPath += "/";

  std::string targetFullPath = outpath + TargetNames.Output;
  std::string targetFullPathPDB = pdbOutputPath + TargetNames.PDB;
  std::string targetFullPathSO = outpath + TargetNames.SharedObject;
  std::string targetFullPathReal = outpath + TargetNames.Real;
  std::string targetFullPathImport = outpathImp + TargetNames.ImportLibrary;

  // Construct the output path version of the names for use in command
  // arguments.
  std::string targetOutPathPDB = LocalGenerator->ConvertToOutputFormat(
    targetFullPathPDB, cmOutputConverter::SHELL);

  std::string targetOutPath = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPath),
    cmOutputConverter::SHELL);
  std::string targetOutPathSO = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathSO),
    cmOutputConverter::SHELL);
  std::string targetOutPathReal = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal),
    cmOutputConverter::SHELL);
  std::string targetOutPathImport = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathImport),
    cmOutputConverter::SHELL);

  NumberOfProgressActions++;
  if (!NoRuleMessages) {
    cmLocalUnixMakefileGenerator3::EchoProgress progress;
    MakeEchoProgress(progress);
    // Add the link message.
    std::string buildEcho = cmStrCat("Linking ", linkLanguage);
    switch (GeneratorTarget->GetType()) {
      case cmStateEnums::STATIC_LIBRARY:
        buildEcho += " static library ";
        break;
      case cmStateEnums::SHARED_LIBRARY:
        buildEcho += " shared library ";
        break;
      case cmStateEnums::MODULE_LIBRARY:
        if (GeneratorTarget->IsCFBundleOnApple()) {
          buildEcho += " CFBundle";
        }
        buildEcho += " shared module ";
        break;
      default:
        buildEcho += " library ";
        break;
    }
    buildEcho += targetOutPath;
    LocalGenerator->AppendEcho(
      commands, buildEcho, cmLocalUnixMakefileGenerator3::EchoLink, &progress);
  }

  // Clean files associated with this library.
  std::set<std::string> libCleanFiles;
  libCleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal));

  std::vector<std::string> commands1;
  // Add a command to remove any existing files for this library.
  // for static libs only
  if (GeneratorTarget->GetType() == cmStateEnums::STATIC_LIBRARY) {
    LocalGenerator->AppendCleanCommand(commands1, libCleanFiles,
                                       GeneratorTarget, "target");
    LocalGenerator->CreateCDCommand(commands1,
                                    Makefile->GetCurrentBinaryDirectory(),
                                    LocalGenerator->GetBinaryDirectory());
    cm::append(commands, commands1);
    commands1.clear();
  }

  if (TargetNames.Output != TargetNames.Real) {
    libCleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPath));
  }
  if (TargetNames.SharedObject != TargetNames.Real &&
      TargetNames.SharedObject != TargetNames.Output) {
    libCleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathSO));
  }
  if (!TargetNames.ImportLibrary.empty()) {
    libCleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathImport));
    std::string implib;
    if (GeneratorTarget->GetImplibGNUtoMS(GetConfigName(),
                                          targetFullPathImport, implib)) {
      libCleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), implib));
    }
  }

  // List the PDB for cleaning only when the whole target is
  // cleaned.  We do not want to delete the .pdb file just before
  // linking the target.
  CleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathPDB));

#ifdef _WIN32
  // There may be a manifest file for this target.  Add it to the
  // clean set just in case.
  if (this->GeneratorTarget->GetType() != cmStateEnums::STATIC_LIBRARY) {
    libCleanFiles.insert(this->LocalGenerator->MaybeConvertToRelativePath(
      this->LocalGenerator->GetCurrentBinaryDirectory(),
      targetFullPath + ".manifest"));
  }
#endif

  // Add the pre-build and pre-link rules building but not when relinking.
  if (!relink) {
    LocalGenerator->AppendCustomCommands(
      commands, GeneratorTarget->GetPreBuildCommands(), GeneratorTarget,
      LocalGenerator->GetBinaryDirectory());
    LocalGenerator->AppendCustomCommands(
      commands, GeneratorTarget->GetPreLinkCommands(), GeneratorTarget,
      LocalGenerator->GetBinaryDirectory());
  }

  // Determine whether a link script will be used.
  bool useLinkScript = GlobalGenerator->GetUseLinkScript();

  bool useResponseFileForObjects =
    CheckUseResponseFileForObjects(linkLanguage);
  bool const useResponseFileForLibs =
    CheckUseResponseFileForLibraries(linkLanguage);

  // For static libraries there might be archiving rules.
  bool haveStaticLibraryRule = false;
  std::vector<std::string> archiveCreateCommands;
  std::vector<std::string> archiveAppendCommands;
  std::vector<std::string> archiveFinishCommands;
  std::string::size_type archiveCommandLimit = std::string::npos;
  if (GeneratorTarget->GetType() == cmStateEnums::STATIC_LIBRARY) {
    haveStaticLibraryRule = Makefile->IsDefinitionSet(linkRuleVar);
    std::string arCreateVar =
      cmStrCat("CMAKE_", linkLanguage, "_ARCHIVE_CREATE");

    arCreateVar = GeneratorTarget->GetFeatureSpecificLinkRuleVariable(
      arCreateVar, linkLanguage, GetConfigName());

    Makefile->GetDefExpandList(arCreateVar, archiveCreateCommands);
    std::string arAppendVar =
      cmStrCat("CMAKE_", linkLanguage, "_ARCHIVE_APPEND");

    arAppendVar = GeneratorTarget->GetFeatureSpecificLinkRuleVariable(
      arAppendVar, linkLanguage, GetConfigName());

    Makefile->GetDefExpandList(arAppendVar, archiveAppendCommands);
    std::string arFinishVar =
      cmStrCat("CMAKE_", linkLanguage, "_ARCHIVE_FINISH");

    arFinishVar = GeneratorTarget->GetFeatureSpecificLinkRuleVariable(
      arFinishVar, linkLanguage, GetConfigName());

    Makefile->GetDefExpandList(arFinishVar, archiveFinishCommands);
  }

  // Decide whether to use archiving rules.
  bool useArchiveRules = !haveStaticLibraryRule &&
    !archiveCreateCommands.empty() && !archiveAppendCommands.empty();
  if (useArchiveRules) {
    // Archiving rules are always run with a link script.
    useLinkScript = true;

    // Archiving rules never use a response file.
    useResponseFileForObjects = false;

    // Limit the length of individual object lists to less than half of
    // the command line length limit (leaving half for other flags).
    // This may result in several calls to the archiver.
    if (size_t limit = cmSystemTools::CalculateCommandLineLengthLimit()) {
      archiveCommandLimit = limit / 2;
    } else {
      archiveCommandLimit = 8000;
    }
  }

  // Expand the rule variables.
  std::vector<std::string> real_link_commands;
  {
    bool useWatcomQuote = Makefile->IsOn(linkRuleVar + "_USE_WATCOM_QUOTE");

    // Set path conversion for link script shells.
    LocalGenerator->SetLinkScriptShell(useLinkScript);

    // Collect up flags to link in needed libraries.
    std::string linkLibs;
    if (GeneratorTarget->GetType() != cmStateEnums::STATIC_LIBRARY) {

      std::unique_ptr<cmLinkLineComputer> linkLineComputer =
        CreateLinkLineComputer(
          LocalGenerator, LocalGenerator->GetStateSnapshot().GetDirectory());
      linkLineComputer->SetForResponse(useResponseFileForLibs);
      linkLineComputer->SetUseWatcomQuote(useWatcomQuote);
      linkLineComputer->SetRelink(relink);

      CreateLinkLibs(linkLineComputer.get(), linkLibs, useResponseFileForLibs,
                     depends);
    }

    // Construct object file lists that may be needed to expand the
    // rule.
    std::string buildObjs;
    CreateObjectLists(useLinkScript, useArchiveRules,
                      useResponseFileForObjects, buildObjs, depends,
                      useWatcomQuote);
    if (!DeviceLinkObject.empty()) {
      buildObjs += " " +
        LocalGenerator->ConvertToOutputFormat(
          LocalGenerator->MaybeConvertToRelativePath(
            LocalGenerator->GetCurrentBinaryDirectory(), DeviceLinkObject),
          cmOutputConverter::SHELL);
    }

    std::string const& aixExports = GetAIXExports(GetConfigName());

    // maybe create .def file from list of objects
    GenDefFile(real_link_commands);

    std::string manifests = GetManifests(GetConfigName());

    cmRulePlaceholderExpander::RuleVariables vars;
    vars.TargetPDB = targetOutPathPDB.c_str();

    // Setup the target version.
    std::string targetVersionMajor;
    std::string targetVersionMinor;
    {
      std::ostringstream majorStream;
      std::ostringstream minorStream;
      int major;
      int minor;
      GeneratorTarget->GetTargetVersion(major, minor);
      majorStream << major;
      minorStream << minor;
      targetVersionMajor = majorStream.str();
      targetVersionMinor = minorStream.str();
    }
    vars.TargetVersionMajor = targetVersionMajor.c_str();
    vars.TargetVersionMinor = targetVersionMinor.c_str();

    vars.CMTargetName = GeneratorTarget->GetName().c_str();
    vars.CMTargetType =
      cmState::GetTargetTypeName(GeneratorTarget->GetType()).c_str();
    vars.Language = linkLanguage.c_str();
    vars.AIXExports = aixExports.c_str();
    vars.Objects = buildObjs.c_str();
    std::string objectDir = GeneratorTarget->GetSupportDirectory();

    objectDir = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), objectDir),
      cmOutputConverter::SHELL);

    vars.ObjectDir = objectDir.c_str();
    cmOutputConverter::OutputFormat output = (useWatcomQuote)
      ? cmOutputConverter::WATCOMQUOTE
      : cmOutputConverter::SHELL;
    std::string target = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal),
      output);
    vars.Target = target.c_str();
    vars.LinkLibraries = linkLibs.c_str();
    vars.ObjectsQuoted = buildObjs.c_str();
    if (GeneratorTarget->HasSOName(GetConfigName())) {
      vars.SONameFlag = Makefile->GetSONameFlag(linkLanguage);
      vars.TargetSOName = TargetNames.SharedObject.c_str();
    }
    vars.LinkFlags = linkFlags.c_str();

    vars.Manifests = manifests.c_str();

    // Compute the directory portion of the install_name setting.
    std::string install_name_dir;
    if (GeneratorTarget->GetType() == cmStateEnums::SHARED_LIBRARY) {
      // Get the install_name directory for the build tree.
      install_name_dir =
        GeneratorTarget->GetInstallNameDirForBuildTree(GetConfigName());

      // Set the rule variable replacement value.
      if (install_name_dir.empty()) {
        vars.TargetInstallNameDir = "";
      } else {
        // Convert to a path for the native build tool.
        install_name_dir = LocalGenerator->ConvertToOutputFormat(
          install_name_dir, cmOutputConverter::SHELL);
        vars.TargetInstallNameDir = install_name_dir.c_str();
      }
    }

    // Add language-specific flags.
    std::string langFlags;
    LocalGenerator->AddLanguageFlagsForLinking(langFlags, GeneratorTarget,
                                               linkLanguage, GetConfigName());

    LocalGenerator->AddArchitectureFlags(langFlags, GeneratorTarget,
                                         linkLanguage, GetConfigName());

    vars.LanguageCompileFlags = langFlags.c_str();

    std::string launcher;
    const char* val =
      LocalGenerator->GetRuleLauncher(GeneratorTarget, "RULE_LAUNCH_LINK");
    if (cmNonempty(val)) {
      launcher = cmStrCat(val, ' ');
    }

    std::unique_ptr<cmRulePlaceholderExpander> rulePlaceholderExpander(
      LocalGenerator->CreateRulePlaceholderExpander());
    // Construct the main link rule and expand placeholders.
    rulePlaceholderExpander->SetTargetImpLib(targetOutPathImport);
    if (useArchiveRules) {
      // Construct the individual object list strings.
      std::vector<std::string> object_strings;
      WriteObjectsStrings(object_strings, archiveCommandLimit);

      // Add the cuda device object to the list of archive files. This will
      // only occur on archives which have CUDA_RESOLVE_DEVICE_SYMBOLS enabled
      if (!DeviceLinkObject.empty()) {
        object_strings.push_back(LocalGenerator->ConvertToOutputFormat(
          LocalGenerator->MaybeConvertToRelativePath(
            LocalGenerator->GetCurrentBinaryDirectory(), DeviceLinkObject),
          cmOutputConverter::SHELL));
      }

      // Create the archive with the first set of objects.
      auto osi = object_strings.begin();
      {
        vars.Objects = osi->c_str();
        for (std::string const& acc : archiveCreateCommands) {
          std::string cmd = launcher + acc;
          rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator, cmd,
                                                       vars);
          real_link_commands.push_back(std::move(cmd));
        }
      }
      // Append to the archive with the other object sets.
      for (++osi; osi != object_strings.end(); ++osi) {
        vars.Objects = osi->c_str();
        for (std::string const& aac : archiveAppendCommands) {
          std::string cmd = launcher + aac;
          rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator, cmd,
                                                       vars);
          real_link_commands.push_back(std::move(cmd));
        }
      }
      // Finish the archive.
      vars.Objects = "";
      for (std::string const& afc : archiveFinishCommands) {
        std::string cmd = launcher + afc;
        rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator, cmd,
                                                     vars);
        // If there is no ranlib the command will be ":".  Skip it.
        if (!cmd.empty() && cmd[0] != ':') {
          real_link_commands.push_back(std::move(cmd));
        }
      }
    } else {
      // Get the set of commands.
      std::string linkRule = GetLinkRule(linkRuleVar);
      cmExpandList(linkRule, real_link_commands);
      if (GeneratorTarget->GetPropertyAsBool("LINK_WHAT_YOU_USE") &&
          (GeneratorTarget->GetType() == cmStateEnums::SHARED_LIBRARY)) {
        std::string cmakeCommand = cmStrCat(
          LocalGenerator->ConvertToOutputFormat(
            cmSystemTools::GetCMakeCommand(), cmLocalGenerator::SHELL),
          " -E __run_co_compile --lwyu=", targetOutPathReal);
        real_link_commands.push_back(std::move(cmakeCommand));
      }

      // Expand placeholders.
      for (std::string& real_link_command : real_link_commands) {
        real_link_command = cmStrCat(launcher, real_link_command);
        rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                     real_link_command, vars);
      }
    }

    // Restore path conversion to normal shells.
    LocalGenerator->SetLinkScriptShell(false);
  }

  // Optionally convert the build rule to use a script to avoid long
  // command lines in the make shell.
  if (useLinkScript) {
    // Use a link script.
    const char* name = (relink ? "relink.txt" : "link.txt");
    CreateLinkScript(name, real_link_commands, commands1, depends);
  } else {
    // No link script.  Just use the link rule directly.
    commands1 = real_link_commands;
  }
  LocalGenerator->CreateCDCommand(commands1,
                                  Makefile->GetCurrentBinaryDirectory(),
                                  LocalGenerator->GetBinaryDirectory());
  cm::append(commands, commands1);
  commands1.clear();

  // Add a rule to create necessary symlinks for the library.
  // Frameworks are handled by cmOSXBundleGenerator.
  if (targetOutPath != targetOutPathReal &&
      !GeneratorTarget->IsFrameworkOnApple()) {
    std::string symlink =
      cmStrCat("$(CMAKE_COMMAND) -E cmake_symlink_library ", targetOutPathReal,
               ' ', targetOutPathSO, ' ', targetOutPath);
    commands1.push_back(std::move(symlink));
    LocalGenerator->CreateCDCommand(commands1,
                                    Makefile->GetCurrentBinaryDirectory(),
                                    LocalGenerator->GetBinaryDirectory());
    cm::append(commands, commands1);
    commands1.clear();
  }

  // Add the post-build rules when building but not when relinking.
  if (!relink) {
    LocalGenerator->AppendCustomCommands(
      commands, GeneratorTarget->GetPostBuildCommands(), GeneratorTarget,
      LocalGenerator->GetBinaryDirectory());
  }

  // Compute the list of outputs.
  std::vector<std::string> outputs(1, targetFullPathReal);
  if (TargetNames.SharedObject != TargetNames.Real) {
    outputs.push_back(targetFullPathSO);
  }
  if (TargetNames.Output != TargetNames.SharedObject &&
      TargetNames.Output != TargetNames.Real) {
    outputs.push_back(targetFullPath);
  }

  // Write the build rule.
  WriteMakeRule(*BuildFileStream, nullptr, outputs, depends, commands, false);

  // Write the main driver rule to build everything in this target.
  WriteTargetDriverRule(targetFullPath, relink);

  // Clean all the possible library names and symlinks.
  CleanFiles.insert(libCleanFiles.begin(), libCleanFiles.end());
}
