/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmMakefileExecutableTargetGenerator.h"

#include <set>
#include <sstream>
#include <string>
#include <utility>
#include <vector>

#include <cm/memory>
#include <cmext/algorithm>

#include "cmGeneratedFileStream.h"
#include "cmGeneratorTarget.h"
#include "cmGlobalUnixMakefileGenerator3.h"
#include "cmLinkLineComputer.h"
#include "cmLinkLineDeviceComputer.h"
#include "cmLocalGenerator.h"
#include "cmLocalUnixMakefileGenerator3.h"
#include "cmMakefile.h"
#include "cmOSXBundleGenerator.h"
#include "cmOutputConverter.h"
#include "cmRulePlaceholderExpander.h"
#include "cmState.h"
#include "cmStateDirectory.h"
#include "cmStateSnapshot.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

cmMakefileExecutableTargetGenerator::cmMakefileExecutableTargetGenerator(
  cmGeneratorTarget* target)
  : cmMakefileTargetGenerator(target)
{
  CustomCommandDriver = OnDepends;
  TargetNames = GeneratorTarget->GetExecutableNames(GetConfigName());

  OSXBundleGenerator = cm::make_unique<cmOSXBundleGenerator>(target);
  OSXBundleGenerator->SetMacContentFolders(&MacContentFolders);
}

cmMakefileExecutableTargetGenerator::~cmMakefileExecutableTargetGenerator() =
  default;

void cmMakefileExecutableTargetGenerator::WriteRuleFiles()
{
  // create the build.make file and directory, put in the common blocks
  CreateRuleFile();

  // write rules used to help build object files
  WriteCommonCodeRules();

  // write the per-target per-language flags
  WriteTargetLanguageFlags();

  // write in rules for object files and custom commands
  WriteTargetBuildRules();

  // write the device link rules
  WriteDeviceExecutableRule(false);

  // write the link rules
  WriteExecutableRule(false);
  if (GeneratorTarget->NeedRelinkBeforeInstall(GetConfigName())) {
    // Write rules to link an installable version of the target.
    WriteExecutableRule(true);
  }

  // Write clean target
  WriteTargetCleanRules();

  // Write the dependency generation rule.  This must be done last so
  // that multiple output pair information is available.
  WriteTargetDependRules();

  // close the streams
  CloseFileStreams();
}

void cmMakefileExecutableTargetGenerator::WriteDeviceExecutableRule(
  bool relink)
{
#ifndef CMAKE_BOOTSTRAP
  const bool requiresDeviceLinking =
    requireDeviceLinking(*GeneratorTarget, *LocalGenerator, GetConfigName());
  if (!requiresDeviceLinking) {
    return;
  }

  std::vector<std::string> commands;

  // Get the name of the device object to generate.
  std::string const& objExt =
    Makefile->GetSafeDefinition("CMAKE_CUDA_OUTPUT_EXTENSION");
  std::string const targetOutput =
    GeneratorTarget->ObjectDirectory + "cmake_device_link" + objExt;
  DeviceLinkObject = targetOutput;

  NumberOfProgressActions++;
  if (!NoRuleMessages) {
    cmLocalUnixMakefileGenerator3::EchoProgress progress;
    MakeEchoProgress(progress);
    // Add the link message.
    std::string buildEcho = cmStrCat(
      "Linking CUDA device code ",
      LocalGenerator->ConvertToOutputFormat(
        LocalGenerator->MaybeConvertToRelativePath(
          LocalGenerator->GetCurrentBinaryDirectory(), DeviceLinkObject),
        cmOutputConverter::SHELL));
    LocalGenerator->AppendEcho(
      commands, buildEcho, cmLocalUnixMakefileGenerator3::EchoLink, &progress);
  }

  if (Makefile->GetSafeDefinition("CMAKE_CUDA_COMPILER_ID") == "Clang") {
    WriteDeviceLinkRule(commands, targetOutput);
  } else {
    WriteNvidiaDeviceExecutableRule(relink, commands, targetOutput);
  }

  // Write the main driver rule to build everything in this target.
  WriteTargetDriverRule(targetOutput, relink);
#else
  static_cast<void>(relink);
#endif
}

void cmMakefileExecutableTargetGenerator::WriteNvidiaDeviceExecutableRule(
  bool relink, std::vector<std::string>& commands,
  const std::string& targetOutput)
{
  const std::string linkLanguage = "CUDA";

  // Build list of dependencies.
  std::vector<std::string> depends;
  AppendLinkDepends(depends, linkLanguage);

  // Build a list of compiler flags and linker flags.
  std::string langFlags;
  std::string linkFlags;

  // Add language feature flags.
  LocalGenerator->AddLanguageFlagsForLinking(langFlags, GeneratorTarget,
                                             linkLanguage, GetConfigName());

  // Add device-specific linker flags.
  GetDeviceLinkFlags(linkFlags, linkLanguage);

  // Construct a list of files associated with this executable that
  // may need to be cleaned.
  std::vector<std::string> exeCleanFiles;
  exeCleanFiles.push_back(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), targetOutput));

  // Determine whether a link script will be used.
  bool useLinkScript = GlobalGenerator->GetUseLinkScript();

  // Construct the main link rule.
  std::vector<std::string> real_link_commands;
  const std::string linkRuleVar = "CMAKE_CUDA_DEVICE_LINK_EXECUTABLE";
  const std::string linkRule = GetLinkRule(linkRuleVar);
  std::vector<std::string> commands1;
  cmExpandList(linkRule, real_link_commands);

  bool useResponseFileForObjects =
    CheckUseResponseFileForObjects(linkLanguage);
  bool const useResponseFileForLibs =
    CheckUseResponseFileForLibraries(linkLanguage);

  // Expand the rule variables.
  {
    bool useWatcomQuote = Makefile->IsOn(linkRuleVar + "_USE_WATCOM_QUOTE");

    // Set path conversion for link script shells.
    LocalGenerator->SetLinkScriptShell(useLinkScript);

    std::unique_ptr<cmLinkLineComputer> linkLineComputer(
      new cmLinkLineDeviceComputer(
        LocalGenerator, LocalGenerator->GetStateSnapshot().GetDirectory()));
    linkLineComputer->SetForResponse(useResponseFileForLibs);
    linkLineComputer->SetUseWatcomQuote(useWatcomQuote);
    linkLineComputer->SetRelink(relink);

    // Collect up flags to link in needed libraries.
    std::string linkLibs;
    CreateLinkLibs(linkLineComputer.get(), linkLibs, useResponseFileForLibs,
                   depends);

    // Construct object file lists that may be needed to expand the
    // rule.
    std::string buildObjs;
    CreateObjectLists(useLinkScript, false, useResponseFileForObjects,
                      buildObjs, depends, useWatcomQuote);

    std::string const& aixExports = GetAIXExports(GetConfigName());

    cmRulePlaceholderExpander::RuleVariables vars;
    std::string objectDir = GeneratorTarget->GetSupportDirectory();

    objectDir = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), objectDir),
      cmOutputConverter::SHELL);

    cmOutputConverter::OutputFormat output = (useWatcomQuote)
      ? cmOutputConverter::WATCOMQUOTE
      : cmOutputConverter::SHELL;
    std::string target = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), targetOutput),
      output);

    std::string targetFullPathCompilePDB =
      ComputeTargetCompilePDB(GetConfigName());
    std::string targetOutPathCompilePDB =
      LocalGenerator->ConvertToOutputFormat(targetFullPathCompilePDB,
                                            cmOutputConverter::SHELL);

    vars.Language = linkLanguage.c_str();
    vars.AIXExports = aixExports.c_str();
    vars.Objects = buildObjs.c_str();
    vars.ObjectDir = objectDir.c_str();
    vars.Target = target.c_str();
    vars.LinkLibraries = linkLibs.c_str();
    vars.LanguageCompileFlags = langFlags.c_str();
    vars.LinkFlags = linkFlags.c_str();
    vars.TargetCompilePDB = targetOutPathCompilePDB.c_str();

    std::string launcher;

    const char* val =
      LocalGenerator->GetRuleLauncher(GeneratorTarget, "RULE_LAUNCH_LINK");
    if (cmNonempty(val)) {
      launcher = cmStrCat(val, ' ');
    }

    std::unique_ptr<cmRulePlaceholderExpander> rulePlaceholderExpander(
      LocalGenerator->CreateRulePlaceholderExpander());

    // Expand placeholders in the commands.
    rulePlaceholderExpander->SetTargetImpLib(targetOutput);
    for (std::string& real_link_command : real_link_commands) {
      real_link_command = cmStrCat(launcher, real_link_command);
      rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                   real_link_command, vars);
    }

    // Restore path conversion to normal shells.
    LocalGenerator->SetLinkScriptShell(false);
  }

  // Optionally convert the build rule to use a script to avoid long
  // command lines in the make shell.
  if (useLinkScript) {
    // Use a link script.
    const char* name = (relink ? "drelink.txt" : "dlink.txt");
    CreateLinkScript(name, real_link_commands, commands1, depends);
  } else {
    // No link script.  Just use the link rule directly.
    commands1 = real_link_commands;
  }
  LocalGenerator->CreateCDCommand(commands1,
                                  Makefile->GetCurrentBinaryDirectory(),
                                  LocalGenerator->GetBinaryDirectory());
  cm::append(commands, commands1);
  commands1.clear();

  // Write the build rule.
  LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, targetOutput,
                                depends, commands, false);

  // Clean all the possible executable names and symlinks.
  CleanFiles.insert(exeCleanFiles.begin(), exeCleanFiles.end());
}

void cmMakefileExecutableTargetGenerator::WriteExecutableRule(bool relink)
{
  std::vector<std::string> commands;

  // Get the name of the executable to generate.
  cmGeneratorTarget::Names targetNames =
    GeneratorTarget->GetExecutableNames(GetConfigName());

  // Construct the full path version of the names.
  std::string outpath = GeneratorTarget->GetDirectory(GetConfigName());
  if (GeneratorTarget->IsAppBundleOnApple()) {
    OSXBundleGenerator->CreateAppBundle(targetNames.Output, outpath,
                                        GetConfigName());
  }
  outpath += '/';
  std::string outpathImp;
  if (relink) {
    outpath = cmStrCat(Makefile->GetCurrentBinaryDirectory(),
                       "/CMakeFiles/CMakeRelink.dir");
    cmSystemTools::MakeDirectory(outpath);
    outpath += '/';
    if (!targetNames.ImportLibrary.empty()) {
      outpathImp = outpath;
    }
  } else {
    cmSystemTools::MakeDirectory(outpath);
    if (!targetNames.ImportLibrary.empty()) {
      outpathImp = GeneratorTarget->GetDirectory(
        GetConfigName(), cmStateEnums::ImportLibraryArtifact);
      cmSystemTools::MakeDirectory(outpathImp);
      outpathImp += '/';
    }
  }

  std::string compilePdbOutputPath =
    GeneratorTarget->GetCompilePDBDirectory(GetConfigName());
  cmSystemTools::MakeDirectory(compilePdbOutputPath);

  std::string pdbOutputPath =
    GeneratorTarget->GetPDBDirectory(GetConfigName());
  cmSystemTools::MakeDirectory(pdbOutputPath);
  pdbOutputPath += '/';

  std::string targetFullPath = outpath + targetNames.Output;
  std::string targetFullPathReal = outpath + targetNames.Real;
  std::string targetFullPathPDB = pdbOutputPath + targetNames.PDB;
  std::string targetFullPathImport = outpathImp + targetNames.ImportLibrary;
  std::string targetOutPathPDB = LocalGenerator->ConvertToOutputFormat(
    targetFullPathPDB, cmOutputConverter::SHELL);
  // Convert to the output path to use in constructing commands.
  std::string targetOutPath = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPath),
    cmOutputConverter::SHELL);
  std::string targetOutPathReal = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal),
    cmOutputConverter::SHELL);
  std::string targetOutPathImport = LocalGenerator->ConvertToOutputFormat(
    LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathImport),
    cmOutputConverter::SHELL);

  // Get the language to use for linking this executable.
  std::string linkLanguage =
    GeneratorTarget->GetLinkerLanguage(GetConfigName());

  // Make sure we have a link language.
  if (linkLanguage.empty()) {
    cmSystemTools::Error("Cannot determine link language for target \"" +
                         GeneratorTarget->GetName() + "\".");
    return;
  }

  // Build list of dependencies.
  std::vector<std::string> depends;
  AppendLinkDepends(depends, linkLanguage);
  if (!DeviceLinkObject.empty()) {
    depends.push_back(DeviceLinkObject);
  }

  NumberOfProgressActions++;
  if (!NoRuleMessages) {
    cmLocalUnixMakefileGenerator3::EchoProgress progress;
    MakeEchoProgress(progress);
    // Add the link message.
    std::string buildEcho =
      cmStrCat("Linking ", linkLanguage, " executable ", targetOutPath);
    LocalGenerator->AppendEcho(
      commands, buildEcho, cmLocalUnixMakefileGenerator3::EchoLink, &progress);
  }

  // Build a list of compiler flags and linker flags.
  std::string flags;
  std::string linkFlags;

  // Add flags to create an executable.
  LocalGenerator->AddConfigVariableFlags(linkFlags, "CMAKE_EXE_LINKER_FLAGS",
                                         GetConfigName());

  if (GeneratorTarget->IsWin32Executable(
        Makefile->GetSafeDefinition("CMAKE_BUILD_TYPE"))) {
    LocalGenerator->AppendFlags(
      linkFlags, Makefile->GetSafeDefinition("CMAKE_CREATE_WIN32_EXE"));
  } else {
    LocalGenerator->AppendFlags(
      linkFlags, Makefile->GetSafeDefinition("CMAKE_CREATE_CONSOLE_EXE"));
  }

  // Add symbol export flags if necessary.
  if (GeneratorTarget->IsExecutableWithExports()) {
    std::string export_flag_var =
      cmStrCat("CMAKE_EXE_EXPORTS_", linkLanguage, "_FLAG");
    LocalGenerator->AppendFlags(linkFlags,
                                Makefile->GetSafeDefinition(export_flag_var));
  }

  LocalGenerator->AppendFlags(
    linkFlags,
    LocalGenerator->GetLinkLibsCMP0065(linkLanguage, *GeneratorTarget));

  if (GeneratorTarget->GetPropertyAsBool("LINK_WHAT_YOU_USE")) {
    LocalGenerator->AppendFlags(linkFlags, " -Wl,--no-as-needed");
  }

  // Add language feature flags.
  LocalGenerator->AddLanguageFlagsForLinking(flags, GeneratorTarget,
                                             linkLanguage, GetConfigName());

  LocalGenerator->AddArchitectureFlags(flags, GeneratorTarget, linkLanguage,
                                       GetConfigName());

  // Add target-specific linker flags.
  GetTargetLinkFlags(linkFlags, linkLanguage);

  {
    std::unique_ptr<cmLinkLineComputer> linkLineComputer =
      CreateLinkLineComputer(
        LocalGenerator, LocalGenerator->GetStateSnapshot().GetDirectory());

    AddModuleDefinitionFlag(linkLineComputer.get(), linkFlags,
                            GetConfigName());
  }

  LocalGenerator->AppendIPOLinkerFlags(linkFlags, GeneratorTarget,
                                       GetConfigName(), linkLanguage);

  // Construct a list of files associated with this executable that
  // may need to be cleaned.
  std::vector<std::string> exeCleanFiles;
  exeCleanFiles.push_back(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), targetFullPath));
#ifdef _WIN32
  // There may be a manifest file for this target.  Add it to the
  // clean set just in case.
  exeCleanFiles.push_back(this->LocalGenerator->MaybeConvertToRelativePath(
    this->LocalGenerator->GetCurrentBinaryDirectory(),
    targetFullPath + ".manifest"));
#endif
  if (TargetNames.Real != TargetNames.Output) {
    exeCleanFiles.push_back(LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal));
  }
  if (!TargetNames.ImportLibrary.empty()) {
    exeCleanFiles.push_back(LocalGenerator->MaybeConvertToRelativePath(
      LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathImport));
    std::string implib;
    if (GeneratorTarget->GetImplibGNUtoMS(GetConfigName(),
                                          targetFullPathImport, implib)) {
      exeCleanFiles.push_back(LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), implib));
    }
  }

  // List the PDB for cleaning only when the whole target is
  // cleaned.  We do not want to delete the .pdb file just before
  // linking the target.
  CleanFiles.insert(LocalGenerator->MaybeConvertToRelativePath(
    LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathPDB));

  // Add the pre-build and pre-link rules building but not when relinking.
  if (!relink) {
    LocalGenerator->AppendCustomCommands(
      commands, GeneratorTarget->GetPreBuildCommands(), GeneratorTarget,
      LocalGenerator->GetBinaryDirectory());
    LocalGenerator->AppendCustomCommands(
      commands, GeneratorTarget->GetPreLinkCommands(), GeneratorTarget,
      LocalGenerator->GetBinaryDirectory());
  }

  // Determine whether a link script will be used.
  bool useLinkScript = GlobalGenerator->GetUseLinkScript();

  // Construct the main link rule.
  std::vector<std::string> real_link_commands;
  std::string linkRuleVar =
    GeneratorTarget->GetCreateRuleVariable(linkLanguage, GetConfigName());
  std::string linkRule = GetLinkRule(linkRuleVar);
  std::vector<std::string> commands1;
  cmExpandList(linkRule, real_link_commands);
  if (GeneratorTarget->IsExecutableWithExports()) {
    // If a separate rule for creating an import library is specified
    // add it now.
    std::string implibRuleVar =
      cmStrCat("CMAKE_", linkLanguage, "_CREATE_IMPORT_LIBRARY");
    Makefile->GetDefExpandList(implibRuleVar, real_link_commands);
  }

  bool useResponseFileForObjects =
    CheckUseResponseFileForObjects(linkLanguage);
  bool const useResponseFileForLibs =
    CheckUseResponseFileForLibraries(linkLanguage);

  // Expand the rule variables.
  {
    bool useWatcomQuote = Makefile->IsOn(linkRuleVar + "_USE_WATCOM_QUOTE");

    // Set path conversion for link script shells.
    LocalGenerator->SetLinkScriptShell(useLinkScript);

    std::unique_ptr<cmLinkLineComputer> linkLineComputer =
      CreateLinkLineComputer(
        LocalGenerator, LocalGenerator->GetStateSnapshot().GetDirectory());
    linkLineComputer->SetForResponse(useResponseFileForLibs);
    linkLineComputer->SetUseWatcomQuote(useWatcomQuote);
    linkLineComputer->SetRelink(relink);

    // Collect up flags to link in needed libraries.
    std::string linkLibs;
    CreateLinkLibs(linkLineComputer.get(), linkLibs, useResponseFileForLibs,
                   depends);

    // Construct object file lists that may be needed to expand the
    // rule.
    std::string buildObjs;
    CreateObjectLists(useLinkScript, false, useResponseFileForObjects,
                      buildObjs, depends, useWatcomQuote);
    if (!DeviceLinkObject.empty()) {
      buildObjs += " " +
        LocalGenerator->ConvertToOutputFormat(
          LocalGenerator->MaybeConvertToRelativePath(
            LocalGenerator->GetCurrentBinaryDirectory(), DeviceLinkObject),
          cmOutputConverter::SHELL);
    }

    // maybe create .def file from list of objects
    GenDefFile(real_link_commands);

    std::string manifests = GetManifests(GetConfigName());

    cmRulePlaceholderExpander::RuleVariables vars;
    vars.CMTargetName = GeneratorTarget->GetName().c_str();
    vars.CMTargetType =
      cmState::GetTargetTypeName(GeneratorTarget->GetType()).c_str();
    vars.Language = linkLanguage.c_str();
    vars.Objects = buildObjs.c_str();
    std::string objectDir = GeneratorTarget->GetSupportDirectory();

    objectDir = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), objectDir),
      cmOutputConverter::SHELL);
    vars.ObjectDir = objectDir.c_str();
    cmOutputConverter::OutputFormat output = (useWatcomQuote)
      ? cmOutputConverter::WATCOMQUOTE
      : cmOutputConverter::SHELL;
    std::string target = LocalGenerator->ConvertToOutputFormat(
      LocalGenerator->MaybeConvertToRelativePath(
        LocalGenerator->GetCurrentBinaryDirectory(), targetFullPathReal),
      output);
    vars.Target = target.c_str();
    vars.TargetPDB = targetOutPathPDB.c_str();

    // Setup the target version.
    std::string targetVersionMajor;
    std::string targetVersionMinor;
    {
      std::ostringstream majorStream;
      std::ostringstream minorStream;
      int major;
      int minor;
      GeneratorTarget->GetTargetVersion(major, minor);
      majorStream << major;
      minorStream << minor;
      targetVersionMajor = majorStream.str();
      targetVersionMinor = minorStream.str();
    }
    vars.TargetVersionMajor = targetVersionMajor.c_str();
    vars.TargetVersionMinor = targetVersionMinor.c_str();

    vars.LinkLibraries = linkLibs.c_str();
    vars.Flags = flags.c_str();
    vars.LinkFlags = linkFlags.c_str();
    vars.Manifests = manifests.c_str();

    if (GeneratorTarget->GetPropertyAsBool("LINK_WHAT_YOU_USE")) {
      std::string cmakeCommand =
        cmStrCat(LocalGenerator->ConvertToOutputFormat(
                   cmSystemTools::GetCMakeCommand(), cmLocalGenerator::SHELL),
                 " -E __run_co_compile --lwyu=", targetOutPathReal);
      real_link_commands.push_back(std::move(cmakeCommand));
    }

    std::string launcher;

    const char* val =
      LocalGenerator->GetRuleLauncher(GeneratorTarget, "RULE_LAUNCH_LINK");
    if (cmNonempty(val)) {
      launcher = cmStrCat(val, ' ');
    }

    std::unique_ptr<cmRulePlaceholderExpander> rulePlaceholderExpander(
      LocalGenerator->CreateRulePlaceholderExpander());

    // Expand placeholders in the commands.
    rulePlaceholderExpander->SetTargetImpLib(targetOutPathImport);
    for (std::string& real_link_command : real_link_commands) {
      real_link_command = cmStrCat(launcher, real_link_command);
      rulePlaceholderExpander->ExpandRuleVariables(LocalGenerator,
                                                   real_link_command, vars);
    }

    // Restore path conversion to normal shells.
    LocalGenerator->SetLinkScriptShell(false);
  }

  // Optionally convert the build rule to use a script to avoid long
  // command lines in the make shell.
  if (useLinkScript) {
    // Use a link script.
    const char* name = (relink ? "relink.txt" : "link.txt");
    CreateLinkScript(name, real_link_commands, commands1, depends);
  } else {
    // No link script.  Just use the link rule directly.
    commands1 = real_link_commands;
  }
  LocalGenerator->CreateCDCommand(commands1,
                                  Makefile->GetCurrentBinaryDirectory(),
                                  LocalGenerator->GetBinaryDirectory());
  cm::append(commands, commands1);
  commands1.clear();

  // Add a rule to create necessary symlinks for the library.
  if (targetOutPath != targetOutPathReal) {
    std::string symlink =
      cmStrCat("$(CMAKE_COMMAND) -E cmake_symlink_executable ",
               targetOutPathReal, ' ', targetOutPath);
    commands1.push_back(std::move(symlink));
    LocalGenerator->CreateCDCommand(commands1,
                                    Makefile->GetCurrentBinaryDirectory(),
                                    LocalGenerator->GetBinaryDirectory());
    cm::append(commands, commands1);
    commands1.clear();
  }

  // Add the post-build rules when building but not when relinking.
  if (!relink) {
    LocalGenerator->AppendCustomCommands(
      commands, GeneratorTarget->GetPostBuildCommands(), GeneratorTarget,
      LocalGenerator->GetBinaryDirectory());
  }

  // Write the build rule.
  LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, targetFullPathReal,
                                depends, commands, false);

  // The symlink name for the target should depend on the real target
  // so if the target version changes it rebuilds and recreates the
  // symlink.
  if (targetFullPath != targetFullPathReal) {
    depends.clear();
    commands.clear();
    depends.push_back(targetFullPathReal);
    LocalGenerator->WriteMakeRule(*BuildFileStream, nullptr, targetFullPath,
                                  depends, commands, false);
  }

  // Write the main driver rule to build everything in this target.
  WriteTargetDriverRule(targetFullPath, relink);

  // Clean all the possible executable names and symlinks.
  CleanFiles.insert(exeCleanFiles.begin(), exeCleanFiles.end());
}
