/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmGeneratorExpressionParser.h"

#include <cassert>
#include <cstddef>
#include <utility>

#include <cm/memory>
#include <cmext/algorithm>
#include <cmext/memory>

#include "cmGeneratorExpressionEvaluator.h"

cmGeneratorExpressionParser::cmGeneratorExpressionParser(
  std::vector<cmGeneratorExpressionToken> tokens)
  : Tokens(std::move(tokens))
  , NestingLevel(0)
{
}

void cmGeneratorExpressionParser::Parse(
  cmGeneratorExpressionEvaluatorVector& result)
{
  it = Tokens.begin();

  while (it != Tokens.end()) {
    ParseContent(result);
  }
}

static void extendText(
  cmGeneratorExpressionEvaluatorVector& result,
  std::vector<cmGeneratorExpressionToken>::const_iterator it)
{
  if (!result.empty() &&
      (*(result.end() - 1))->GetType() ==
        cmGeneratorExpressionEvaluator::Text) {
    cm::static_reference_cast<TextContent>(*(result.end() - 1))
      .Extend(it->Length);
  } else {
    auto textContent = cm::make_unique<TextContent>(it->Content, it->Length);
    result.push_back(std::move(textContent));
  }
}

static void extendResult(
  cmGeneratorExpressionParser::cmGeneratorExpressionEvaluatorVector& result,
  cmGeneratorExpressionParser::cmGeneratorExpressionEvaluatorVector&& contents)
{
  if (!result.empty() &&
      (*(result.end() - 1))->GetType() ==
        cmGeneratorExpressionEvaluator::Text &&
      contents.front()->GetType() == cmGeneratorExpressionEvaluator::Text) {
    cm::static_reference_cast<TextContent>(*(result.end() - 1))
      .Extend(
        cm::static_reference_cast<TextContent>(contents.front()).GetLength());
    contents.erase(contents.begin());
  }
  cm::append(result, std::move(contents));
}

void cmGeneratorExpressionParser::ParseGeneratorExpression(
  cmGeneratorExpressionEvaluatorVector& result)
{
  assert(it != Tokens.end());
  unsigned int nestedLevel = NestingLevel;
  ++NestingLevel;

  auto startToken = it - 1;

  cmGeneratorExpressionEvaluatorVector identifier;
  while (it->TokenType != cmGeneratorExpressionToken::EndExpression &&
         it->TokenType != cmGeneratorExpressionToken::ColonSeparator) {
    if (it->TokenType == cmGeneratorExpressionToken::CommaSeparator) {
      extendText(identifier, it);
      ++it;
    } else {
      ParseContent(identifier);
    }
    if (it == Tokens.end()) {
      break;
    }
  }
  if (identifier.empty()) {
    // ERROR
  }

  if (it != Tokens.end() &&
      it->TokenType == cmGeneratorExpressionToken::EndExpression) {
    auto content = cm::make_unique<GeneratorExpressionContent>(
      startToken->Content, it->Content - startToken->Content + it->Length);
    assert(it != Tokens.end());
    ++it;
    --NestingLevel;
    content->SetIdentifier(std::move(identifier));
    result.push_back(std::move(content));
    return;
  }

  std::vector<cmGeneratorExpressionEvaluatorVector> parameters;
  std::vector<std::vector<cmGeneratorExpressionToken>::const_iterator>
    commaTokens;
  std::vector<cmGeneratorExpressionToken>::const_iterator colonToken;

  bool emptyParamTermination = false;

  if (it != Tokens.end() &&
      it->TokenType == cmGeneratorExpressionToken::ColonSeparator) {
    colonToken = it;
    parameters.resize(parameters.size() + 1);
    assert(it != Tokens.end());
    ++it;
    if (it == Tokens.end()) {
      emptyParamTermination = true;
    }

    while (it != Tokens.end() &&
           it->TokenType == cmGeneratorExpressionToken::CommaSeparator) {
      commaTokens.push_back(it);
      parameters.resize(parameters.size() + 1);
      assert(it != Tokens.end());
      ++it;
      if (it == Tokens.end()) {
        emptyParamTermination = true;
      }
    }
    while (it != Tokens.end() &&
           it->TokenType == cmGeneratorExpressionToken::ColonSeparator) {
      extendText(*(parameters.end() - 1), it);
      assert(it != Tokens.end());
      ++it;
    }
    while (it != Tokens.end() &&
           it->TokenType != cmGeneratorExpressionToken::EndExpression) {
      ParseContent(*(parameters.end() - 1));
      if (it == Tokens.end()) {
        break;
      }
      while (it != Tokens.end() &&
             it->TokenType == cmGeneratorExpressionToken::CommaSeparator) {
        commaTokens.push_back(it);
        parameters.resize(parameters.size() + 1);
        assert(it != Tokens.end());
        ++it;
        if (it == Tokens.end()) {
          emptyParamTermination = true;
        }
      }
      while (it != Tokens.end() &&
             it->TokenType == cmGeneratorExpressionToken::ColonSeparator) {
        extendText(*(parameters.end() - 1), it);
        assert(it != Tokens.end());
        ++it;
      }
    }
    if (it != Tokens.end() &&
        it->TokenType == cmGeneratorExpressionToken::EndExpression) {
      --NestingLevel;
      assert(it != Tokens.end());
      ++it;
    }
  }

  if (nestedLevel != NestingLevel) {
    // There was a '$<' in the text, but no corresponding '>'. Rebuild to
    // treat the '$<' as having been plain text, along with the
    // corresponding : and , tokens that might have been found.
    extendText(result, startToken);
    extendResult(result, std::move(identifier));
    if (!parameters.empty()) {
      extendText(result, colonToken);

      auto pit = parameters.begin();
      const auto pend = parameters.end();
      auto commaIt = commaTokens.begin();
      assert(parameters.size() > commaTokens.size());
      for (; pit != pend; ++pit, ++commaIt) {
        if (!pit->empty() && !emptyParamTermination) {
          extendResult(result, std::move(*pit));
        }
        if (commaIt != commaTokens.end()) {
          extendText(result, *commaIt);
        } else {
          break;
        }
      }
    }
    return;
  }

  size_t contentLength =
    ((it - 1)->Content - startToken->Content) + (it - 1)->Length;
  auto content = cm::make_unique<GeneratorExpressionContent>(
    startToken->Content, contentLength);
  content->SetIdentifier(std::move(identifier));
  content->SetParameters(std::move(parameters));
  result.push_back(std::move(content));
}

void cmGeneratorExpressionParser::ParseContent(
  cmGeneratorExpressionEvaluatorVector& result)
{
  assert(it != Tokens.end());
  switch (it->TokenType) {
    case cmGeneratorExpressionToken::Text: {
      if (NestingLevel == 0) {
        if (!result.empty() &&
            (*(result.end() - 1))->GetType() ==
              cmGeneratorExpressionEvaluator::Text) {
          // A comma in 'plain text' could have split text that should
          // otherwise be continuous. Extend the last text content instead of
          // creating a new one.
          cm::static_reference_cast<TextContent>(*(result.end() - 1))
            .Extend(it->Length);
          assert(it != Tokens.end());
          ++it;
          return;
        }
      }
      auto n = cm::make_unique<TextContent>(it->Content, it->Length);
      result.push_back(std::move(n));
      assert(it != Tokens.end());
      ++it;
      return;
    }
    case cmGeneratorExpressionToken::BeginExpression:
      assert(it != Tokens.end());
      ++it;
      ParseGeneratorExpression(result);
      return;
    case cmGeneratorExpressionToken::EndExpression:
    case cmGeneratorExpressionToken::ColonSeparator:
    case cmGeneratorExpressionToken::CommaSeparator:
      if (NestingLevel == 0) {
        extendText(result, it);
      } else {
        assert(false && "Got unexpected syntax token.");
      }
      assert(it != Tokens.end());
      ++it;
      return;
  }
  assert(false && "Unhandled token in generator expression.");
}
