/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmGccDepfileLexerHelper.h"

#include <cstdio>
#include <memory>
#include <string>
#include <vector>

#include "cmGccDepfileReaderTypes.h"

#include "LexerParser/cmGccDepfileLexer.h"

#ifdef _WIN32
#  include "cmsys/Encoding.h"
#endif

bool cmGccDepfileLexerHelper::readFile(const char* filePath)
{
#ifdef _WIN32
  wchar_t* wpath = cmsysEncoding_DupToWide(filePath);
  FILE* file = _wfopen(wpath, L"rb");
  free(wpath);
#else
  FILE* file = fopen(filePath, "r");
#endif
  if (!file) {
    return false;
  }
  newEntry();
  yyscan_t scanner;
  cmGccDepfile_yylex_init(&scanner);
  cmGccDepfile_yyset_extra(this, scanner);
  cmGccDepfile_yyrestart(file, scanner);
  cmGccDepfile_yylex(scanner);
  cmGccDepfile_yylex_destroy(scanner);
  sanitizeContent();
  fclose(file);
  return true;
}

void cmGccDepfileLexerHelper::newEntry()
{
  HelperState = State::Rule;
  Content.emplace_back();
  newRule();
}

void cmGccDepfileLexerHelper::newRule()
{
  auto& entry = Content.back();
  if (entry.rules.empty() || !entry.rules.back().empty()) {
    entry.rules.emplace_back();
  }
}

void cmGccDepfileLexerHelper::newDependency()
{
  // printf("NEW DEP\n");
  HelperState = State::Dependency;
  if (Content.back().paths.empty() || !Content.back().paths.back().empty()) {
    Content.back().paths.emplace_back();
  }
}

void cmGccDepfileLexerHelper::newRuleOrDependency()
{
  if (HelperState == State::Rule) {
    newRule();
  } else {
    newDependency();
  }
}

void cmGccDepfileLexerHelper::addToCurrentPath(const char* s)
{
  if (Content.empty()) {
    return;
  }
  cmGccStyleDependency* dep = &Content.back();
  std::string* dst = nullptr;
  switch (HelperState) {
    case State::Rule: {
      if (dep->rules.empty()) {
        return;
      }
      dst = &dep->rules.back();
    } break;
    case State::Dependency: {
      if (dep->paths.empty()) {
        return;
      }
      dst = &dep->paths.back();
    } break;
  }
  dst->append(s);
}

void cmGccDepfileLexerHelper::sanitizeContent()
{
  for (auto it = Content.begin(); it != Content.end();) {
    // Remove empty rules
    for (auto rit = it->rules.begin(); rit != it->rules.end();) {
      if (rit->empty()) {
        rit = it->rules.erase(rit);
      } else {
        ++rit;
      }
    }
    // Remove the entry if rules are empty
    if (it->rules.empty()) {
      it = Content.erase(it);
    } else {
      // Remove empty paths
      for (auto pit = it->paths.begin(); pit != it->paths.end();) {
        if (pit->empty()) {
          pit = it->paths.erase(pit);
        } else {
          ++pit;
        }
      }
      ++it;
    }
  }
}
