/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmFileLockPool.h"

#include <cassert>
#include <utility>

#include "cmFileLock.h"
#include "cmFileLockResult.h"

cmFileLockPool::cmFileLockPool() = default;

cmFileLockPool::~cmFileLockPool() = default;

void cmFileLockPool::PushFunctionScope()
{
  FunctionScopes.push_back(ScopePool());
}

void cmFileLockPool::PopFunctionScope()
{
  assert(!FunctionScopes.empty());
  FunctionScopes.pop_back();
}

void cmFileLockPool::PushFileScope()
{
  FileScopes.push_back(ScopePool());
}

void cmFileLockPool::PopFileScope()
{
  assert(!FileScopes.empty());
  FileScopes.pop_back();
}

cmFileLockResult cmFileLockPool::LockFunctionScope(const std::string& filename,
                                                   unsigned long timeoutSec)
{
  if (IsAlreadyLocked(filename)) {
    return cmFileLockResult::MakeAlreadyLocked();
  }
  if (FunctionScopes.empty()) {
    return cmFileLockResult::MakeNoFunction();
  }
  return FunctionScopes.back().Lock(filename, timeoutSec);
}

cmFileLockResult cmFileLockPool::LockFileScope(const std::string& filename,
                                               unsigned long timeoutSec)
{
  if (IsAlreadyLocked(filename)) {
    return cmFileLockResult::MakeAlreadyLocked();
  }
  assert(!FileScopes.empty());
  return FileScopes.back().Lock(filename, timeoutSec);
}

cmFileLockResult cmFileLockPool::LockProcessScope(const std::string& filename,
                                                  unsigned long timeoutSec)
{
  if (IsAlreadyLocked(filename)) {
    return cmFileLockResult::MakeAlreadyLocked();
  }
  return ProcessScope.Lock(filename, timeoutSec);
}

cmFileLockResult cmFileLockPool::Release(const std::string& filename)
{
  for (auto& funcScope : FunctionScopes) {
    const cmFileLockResult result = funcScope.Release(filename);
    if (!result.IsOk()) {
      return result;
    }
  }

  for (auto& fileScope : FileScopes) {
    const cmFileLockResult result = fileScope.Release(filename);
    if (!result.IsOk()) {
      return result;
    }
  }

  return ProcessScope.Release(filename);
}

bool cmFileLockPool::IsAlreadyLocked(const std::string& filename) const
{
  for (auto const& funcScope : FunctionScopes) {
    const bool result = funcScope.IsAlreadyLocked(filename);
    if (result) {
      return true;
    }
  }

  for (auto const& fileScope : FileScopes) {
    const bool result = fileScope.IsAlreadyLocked(filename);
    if (result) {
      return true;
    }
  }

  return ProcessScope.IsAlreadyLocked(filename);
}

cmFileLockPool::ScopePool::ScopePool() = default;

cmFileLockPool::ScopePool::~ScopePool() = default;

cmFileLockPool::ScopePool::ScopePool(ScopePool&&) noexcept = default;

cmFileLockPool::ScopePool& cmFileLockPool::ScopePool::operator=(
  ScopePool&& other) noexcept
{
  if (this != &other) {
    Locks = std::move(other.Locks);
  }

  return *this;
}

cmFileLockResult cmFileLockPool::ScopePool::Lock(const std::string& filename,
                                                 unsigned long timeoutSec)
{
  cmFileLock lock;
  const cmFileLockResult result = lock.Lock(filename, timeoutSec);
  if (result.IsOk()) {
    Locks.push_back(std::move(lock));
    return cmFileLockResult::MakeOk();
  }
  return result;
}

cmFileLockResult cmFileLockPool::ScopePool::Release(
  const std::string& filename)
{
  for (auto& lock : Locks) {
    if (lock.IsLocked(filename)) {
      return lock.Release();
    }
  }
  return cmFileLockResult::MakeOk();
}

bool cmFileLockPool::ScopePool::IsAlreadyLocked(
  const std::string& filename) const
{
  for (auto const& lock : Locks) {
    if (lock.IsLocked(filename)) {
      return true;
    }
  }
  return false;
}
