/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmFileLock.h"

#include <cassert>
#include <utility>

#include "cmFileLockResult.h"

// Common implementation

cmFileLock::cmFileLock(cmFileLock&& other) noexcept
{
  File = other.File;
#if defined(_WIN32)
  other.File = INVALID_HANDLE_VALUE;
#else
  other.File = -1;
#endif
  Filename = std::move(other.Filename);
}

cmFileLock::~cmFileLock()
{
  if (!Filename.empty()) {
    const cmFileLockResult result = Release();
    static_cast<void>(result);
    assert(result.IsOk());
  }
}

cmFileLock& cmFileLock::operator=(cmFileLock&& other) noexcept
{
  File = other.File;
#if defined(_WIN32)
  other.File = INVALID_HANDLE_VALUE;
#else
  other.File = -1;
#endif
  Filename = std::move(other.Filename);

  return *this;
}

cmFileLockResult cmFileLock::Lock(const std::string& filename,
                                  unsigned long timeout)
{
  if (filename.empty()) {
    // Error is internal since all the directories and file must be created
    // before actual lock called.
    return cmFileLockResult::MakeInternal();
  }

  if (!Filename.empty()) {
    // Error is internal since double-lock must be checked in class
    // cmFileLockPool by the cmFileLock::IsLocked method.
    return cmFileLockResult::MakeInternal();
  }

  Filename = filename;
  cmFileLockResult result = OpenFile();
  if (result.IsOk()) {
    if (timeout == static_cast<unsigned long>(-1)) {
      result = LockWithoutTimeout();
    } else {
      result = LockWithTimeout(timeout);
    }
  }

  if (!result.IsOk()) {
    Filename.clear();
  }

  return result;
}

bool cmFileLock::IsLocked(const std::string& filename) const
{
  return filename == Filename;
}

#if defined(_WIN32)
#  include "cmFileLockWin32.cxx"
#else
#  include "cmFileLockUnix.cxx"
#endif
