/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCustomCommandGenerator.h"

#include <cstddef>
#include <memory>
#include <utility>

#include <cmext/algorithm>

#include "cmCustomCommand.h"
#include "cmCustomCommandLines.h"
#include "cmGeneratorExpression.h"
#include "cmGeneratorTarget.h"
#include "cmLocalGenerator.h"
#include "cmMakefile.h"
#include "cmProperty.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

namespace {
void AppendPaths(const std::vector<std::string>& inputs,
                 cmGeneratorExpression const& ge, cmLocalGenerator* lg,
                 std::string const& config, std::vector<std::string>& output)
{
  for (std::string const& in : inputs) {
    std::unique_ptr<cmCompiledGeneratorExpression> cge = ge.Parse(in);
    std::vector<std::string> result =
      cmExpandedList(cge->Evaluate(lg, config));
    for (std::string& it : result) {
      cmSystemTools::ConvertToUnixSlashes(it);
      if (cmSystemTools::FileIsFullPath(it)) {
        it = cmSystemTools::CollapseFullPath(
          it, lg->GetMakefile()->GetHomeOutputDirectory());
      }
    }
    cm::append(output, result);
  }
}
}

cmCustomCommandGenerator::cmCustomCommandGenerator(cmCustomCommand const& cc,
                                                   std::string config,
                                                   cmLocalGenerator* lg)
  : CC(cc)
  , Config(std::move(config))
  , LG(lg)
  , OldStyle(cc.GetEscapeOldStyle())
  , MakeVars(cc.GetEscapeAllowMakeVars())
  , EmulatorsWithArguments(cc.GetCommandLines().size())
{
  cmGeneratorExpression ge(cc.GetBacktrace());

  const cmCustomCommandLines& cmdlines = CC.GetCommandLines();
  for (cmCustomCommandLine const& cmdline : cmdlines) {
    cmCustomCommandLine argv;
    for (std::string const& clarg : cmdline) {
      std::unique_ptr<cmCompiledGeneratorExpression> cge = ge.Parse(clarg);
      std::string parsed_arg = cge->Evaluate(LG, Config);
      if (CC.GetCommandExpandLists()) {
        cm::append(argv, cmExpandedList(parsed_arg));
      } else {
        argv.push_back(std::move(parsed_arg));
      }
    }

    // Later code assumes at least one entry exists, but expanding
    // lists on an empty command may have left this empty.
    // FIXME: Should we define behavior for removing empty commands?
    if (argv.empty()) {
      argv.emplace_back();
    }

    CommandLines.push_back(std::move(argv));
  }

  AppendPaths(cc.GetByproducts(), ge, LG, Config, Byproducts);
  AppendPaths(cc.GetDepends(), ge, LG, Config, Depends);

  const std::string& workingdirectory = CC.GetWorkingDirectory();
  if (!workingdirectory.empty()) {
    std::unique_ptr<cmCompiledGeneratorExpression> cge =
      ge.Parse(workingdirectory);
    WorkingDirectory = cge->Evaluate(LG, Config);
    // Convert working directory to a full path.
    if (!WorkingDirectory.empty()) {
      std::string const& build_dir = LG->GetCurrentBinaryDirectory();
      WorkingDirectory =
        cmSystemTools::CollapseFullPath(WorkingDirectory, build_dir);
    }
  }

  FillEmulatorsWithArguments();
}

unsigned int cmCustomCommandGenerator::GetNumberOfCommands() const
{
  return static_cast<unsigned int>(CC.GetCommandLines().size());
}

void cmCustomCommandGenerator::FillEmulatorsWithArguments()
{
  if (!LG->GetMakefile()->IsOn("CMAKE_CROSSCOMPILING")) {
    return;
  }

  for (unsigned int c = 0; c < GetNumberOfCommands(); ++c) {
    std::string const& argv0 = CommandLines[c][0];
    cmGeneratorTarget* target = LG->FindGeneratorTargetToUse(argv0);
    if (target && target->GetType() == cmStateEnums::EXECUTABLE &&
        !target->IsImported()) {

      cmProp emulator_property =
        target->GetProperty("CROSSCOMPILING_EMULATOR");
      if (!emulator_property) {
        continue;
      }

      cmExpandList(*emulator_property, EmulatorsWithArguments[c]);
    }
  }
}

std::vector<std::string> cmCustomCommandGenerator::GetCrossCompilingEmulator(
  unsigned int c) const
{
  if (c >= EmulatorsWithArguments.size()) {
    return std::vector<std::string>();
  }
  return EmulatorsWithArguments[c];
}

const char* cmCustomCommandGenerator::GetArgv0Location(unsigned int c) const
{
  std::string const& argv0 = CommandLines[c][0];
  cmGeneratorTarget* target = LG->FindGeneratorTargetToUse(argv0);
  if (target && target->GetType() == cmStateEnums::EXECUTABLE &&
      (target->IsImported() ||
       target->GetProperty("CROSSCOMPILING_EMULATOR") ||
       !LG->GetMakefile()->IsOn("CMAKE_CROSSCOMPILING"))) {
    return target->GetLocation(Config).c_str();
  }
  return nullptr;
}

bool cmCustomCommandGenerator::HasOnlyEmptyCommandLines() const
{
  for (size_t i = 0; i < CommandLines.size(); ++i) {
    for (size_t j = 0; j < CommandLines[i].size(); ++j) {
      if (!CommandLines[i][j].empty()) {
        return false;
      }
    }
  }
  return true;
}

std::string cmCustomCommandGenerator::GetCommand(unsigned int c) const
{
  std::vector<std::string> emulator = GetCrossCompilingEmulator(c);
  if (!emulator.empty()) {
    return emulator[0];
  }
  if (const char* location = GetArgv0Location(c)) {
    return std::string(location);
  }

  return CommandLines[c][0];
}

std::string escapeForShellOldStyle(const std::string& str)
{
  std::string result;
#if defined(_WIN32) && !defined(__CYGWIN__)
  // if there are spaces
  std::string temp = str;
  if (temp.find(" ") != std::string::npos &&
      temp.find("\"") == std::string::npos) {
    result = cmStrCat('"', str, '"');
    return result;
  }
  return str;
#else
  for (const char* ch = str.c_str(); *ch != '\0'; ++ch) {
    if (*ch == ' ') {
      result += '\\';
    }
    result += *ch;
  }
  return result;
#endif
}

void cmCustomCommandGenerator::AppendArguments(unsigned int c,
                                               std::string& cmd) const
{
  unsigned int offset = 1;
  std::vector<std::string> emulator = GetCrossCompilingEmulator(c);
  if (!emulator.empty()) {
    for (unsigned j = 1; j < emulator.size(); ++j) {
      cmd += " ";
      if (OldStyle) {
        cmd += escapeForShellOldStyle(emulator[j]);
      } else {
        cmd += LG->EscapeForShell(emulator[j], MakeVars, false, false,
                                  MakeVars && LG->IsNinjaMulti());
      }
    }

    offset = 0;
  }

  cmCustomCommandLine const& commandLine = CommandLines[c];
  for (unsigned int j = offset; j < commandLine.size(); ++j) {
    std::string arg;
    if (const char* location = j == 0 ? GetArgv0Location(c) : nullptr) {
      // GetCommand returned the emulator instead of the argv0 location,
      // so transform the latter now.
      arg = location;
    } else {
      arg = commandLine[j];
    }
    cmd += " ";
    if (OldStyle) {
      cmd += escapeForShellOldStyle(arg);
    } else {
      cmd += LG->EscapeForShell(arg, MakeVars, false, false,
                                MakeVars && LG->IsNinjaMulti());
    }
  }
}

const char* cmCustomCommandGenerator::GetComment() const
{
  return CC.GetComment();
}

std::string cmCustomCommandGenerator::GetWorkingDirectory() const
{
  return WorkingDirectory;
}

std::vector<std::string> const& cmCustomCommandGenerator::GetOutputs() const
{
  return CC.GetOutputs();
}

std::vector<std::string> const& cmCustomCommandGenerator::GetByproducts() const
{
  return Byproducts;
}

std::vector<std::string> const& cmCustomCommandGenerator::GetDepends() const
{
  return Depends;
}
