/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCustomCommand.h"

#include <utility>

#include <cmext/algorithm>

cmCustomCommand::cmCustomCommand(std::vector<std::string> outputs,
                                 std::vector<std::string> byproducts,
                                 std::vector<std::string> depends,
                                 cmCustomCommandLines commandLines,
                                 cmListFileBacktrace lfbt, const char* comment,
                                 const char* workingDirectory,
                                 bool stdPipesUTF8)
  : Outputs(std::move(outputs))
  , Byproducts(std::move(byproducts))
  , Depends(std::move(depends))
  , CommandLines(std::move(commandLines))
  , Backtrace(std::move(lfbt))
  , Comment(comment ? comment : "")
  , WorkingDirectory(workingDirectory ? workingDirectory : "")
  , HaveComment(comment != nullptr)
  , StdPipesUTF8(stdPipesUTF8)
{
}

const std::vector<std::string>& cmCustomCommand::GetOutputs() const
{
  return Outputs;
}

const std::vector<std::string>& cmCustomCommand::GetByproducts() const
{
  return Byproducts;
}

const std::vector<std::string>& cmCustomCommand::GetDepends() const
{
  return Depends;
}

const cmCustomCommandLines& cmCustomCommand::GetCommandLines() const
{
  return CommandLines;
}

const char* cmCustomCommand::GetComment() const
{
  const char* no_comment = nullptr;
  return HaveComment ? Comment.c_str() : no_comment;
}

void cmCustomCommand::AppendCommands(const cmCustomCommandLines& commandLines)
{
  cm::append(CommandLines, commandLines);
}

void cmCustomCommand::AppendDepends(const std::vector<std::string>& depends)
{
  cm::append(Depends, depends);
}

bool cmCustomCommand::GetEscapeOldStyle() const
{
  return EscapeOldStyle;
}

void cmCustomCommand::SetEscapeOldStyle(bool b)
{
  EscapeOldStyle = b;
}

bool cmCustomCommand::GetEscapeAllowMakeVars() const
{
  return EscapeAllowMakeVars;
}

void cmCustomCommand::SetEscapeAllowMakeVars(bool b)
{
  EscapeAllowMakeVars = b;
}

cmListFileBacktrace const& cmCustomCommand::GetBacktrace() const
{
  return Backtrace;
}

cmImplicitDependsList const& cmCustomCommand::GetImplicitDepends() const
{
  return ImplicitDepends;
}

void cmCustomCommand::SetImplicitDepends(cmImplicitDependsList const& l)
{
  ImplicitDepends = l;
}

void cmCustomCommand::AppendImplicitDepends(cmImplicitDependsList const& l)
{
  cm::append(ImplicitDepends, l);
}

bool cmCustomCommand::GetUsesTerminal() const
{
  return UsesTerminal;
}

void cmCustomCommand::SetUsesTerminal(bool b)
{
  UsesTerminal = b;
}

bool cmCustomCommand::GetCommandExpandLists() const
{
  return CommandExpandLists;
}

void cmCustomCommand::SetCommandExpandLists(bool b)
{
  CommandExpandLists = b;
}

const std::string& cmCustomCommand::GetDepfile() const
{
  return Depfile;
}

void cmCustomCommand::SetDepfile(const std::string& depfile)
{
  Depfile = depfile;
}

const std::string& cmCustomCommand::GetJobPool() const
{
  return JobPool;
}

void cmCustomCommand::SetJobPool(const std::string& job_pool)
{
  JobPool = job_pool;
}
