/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCursesLongMessageForm.h"

#include <cstdio>
#include <cstring>

#include "cmCursesForm.h"
#include "cmCursesMainForm.h"
#include "cmCursesStandardIncludes.h"
#include "cmStringAlgorithms.h"
#include "cmVersion.h"

inline int ctrl(int z)
{
  return (z & 037);
}

cmCursesLongMessageForm::cmCursesLongMessageForm(
  std::vector<std::string> const& messages, const char* title,
  ScrollBehavior scrollBehavior)
  : Scrolling(scrollBehavior)
{
  // Append all messages into on big string
  Messages = cmJoin(messages, "\n");
  Title = title;
  Fields[0] = nullptr;
  Fields[1] = nullptr;
}

cmCursesLongMessageForm::~cmCursesLongMessageForm()
{
  if (Fields[0]) {
    free_field(Fields[0]);
  }
}

void cmCursesLongMessageForm::UpdateContent(std::string const& output,
                                            std::string const& title)
{
  Title = title;

  if (!output.empty() && Messages.size() < MAX_CONTENT_SIZE) {
    Messages.push_back('\n');
    Messages.append(output);
    form_driver(Form, REQ_NEW_LINE);
    DrawMessage(output.c_str());
  }

  UpdateStatusBar();
  touchwin(stdscr);
  refresh();
}

void cmCursesLongMessageForm::UpdateStatusBar()
{
  int x;
  int y;
  getmaxyx(stdscr, y, x);

  char bar[cmCursesMainForm::MAX_WIDTH];
  size_t size = Title.size();
  if (size >= cmCursesMainForm::MAX_WIDTH) {
    size = cmCursesMainForm::MAX_WIDTH - 1;
  }
  strncpy(bar, Title.c_str(), size);
  for (size_t i = size; i < cmCursesMainForm::MAX_WIDTH; i++) {
    bar[i] = ' ';
  }
  int width;
  if (x >= 0 && x < cmCursesMainForm::MAX_WIDTH) {
    width = x;
  } else {
    width = cmCursesMainForm::MAX_WIDTH - 1;
  }

  bar[width] = '\0';

  char version[cmCursesMainForm::MAX_WIDTH];
  char vertmp[128];
  sprintf(vertmp, "CMake Version %s", cmVersion::GetCMakeVersion());
  size_t sideSpace = (width - strlen(vertmp));
  for (size_t i = 0; i < sideSpace; i++) {
    version[i] = ' ';
  }
  sprintf(version + sideSpace, "%s", vertmp);
  version[width] = '\0';

  char fmt_s[] = "%s";
  curses_move(y - 4, 0);
  attron(A_STANDOUT);
  printw(fmt_s, bar);
  attroff(A_STANDOUT);
  curses_move(y - 3, 0);
  printw(fmt_s, version);
  pos_form_cursor(Form);
}

void cmCursesLongMessageForm::PrintKeys()
{
  int x;
  int y;
  getmaxyx(stdscr, y, x);
  if (x < cmCursesMainForm::MIN_WIDTH || y < cmCursesMainForm::MIN_HEIGHT) {
    return;
  }
  char firstLine[512];
  sprintf(firstLine, "Press [e] to exit screen");

  char fmt_s[] = "%s";
  curses_move(y - 2, 0);
  printw(fmt_s, firstLine);
  pos_form_cursor(Form);
}

void cmCursesLongMessageForm::Render(int /*left*/, int /*top*/, int /*width*/,
                                     int /*height*/)
{
  int x;
  int y;
  getmaxyx(stdscr, y, x);

  if (Form) {
    unpost_form(Form);
    free_form(Form);
    Form = nullptr;
  }

  if (Fields[0]) {
    free_field(Fields[0]);
    Fields[0] = nullptr;
  }

  Fields[0] = new_field(y - 6, x - 2, 1, 1, 0, 0);

  field_opts_off(Fields[0], O_STATIC);

  Form = new_form(Fields);
  post_form(Form);

  form_driver(Form, REQ_BEG_FIELD);
  DrawMessage(Messages.c_str());

  UpdateStatusBar();
  touchwin(stdscr);
  refresh();
}

void cmCursesLongMessageForm::DrawMessage(const char* msg) const
{
  int i = 0;
  while (msg[i] != '\0' && i < MAX_CONTENT_SIZE) {
    if (msg[i] == '\n' && msg[i + 1] != '\0') {
      form_driver(Form, REQ_NEW_LINE);
    } else {
      form_driver(Form, msg[i]);
    }
    i++;
  }
  if (Scrolling == ScrollBehavior::ScrollDown) {
    form_driver(Form, REQ_END_FIELD);
  } else {
    form_driver(Form, REQ_BEG_FIELD);
  }
}

void cmCursesLongMessageForm::HandleInput()
{
  if (!Form) {
    return;
  }

  char debugMessage[128];

  for (;;) {
    PrintKeys();
    int key = getch();

    sprintf(debugMessage, "Message widget handling input, key: %d", key);
    cmCursesForm::LogMessage(debugMessage);

    // quit
    if (key == 'o' || key == 'e') {
      break;
    }
    if (key == KEY_DOWN || key == ctrl('n')) {
      form_driver(Form, REQ_SCR_FLINE);
    } else if (key == KEY_UP || key == ctrl('p')) {
      form_driver(Form, REQ_SCR_BLINE);
    } else if (key == KEY_NPAGE || key == ctrl('d')) {
      form_driver(Form, REQ_SCR_FPAGE);
    } else if (key == KEY_PPAGE || key == ctrl('u')) {
      form_driver(Form, REQ_SCR_BPAGE);
    }

    UpdateStatusBar();
    touchwin(stdscr);
    wrefresh(stdscr);
  }
}
