/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCursesCacheEntryComposite.h"

#include <cassert>
#include <utility>
#include <vector>

#include <cm/memory>

#include "cmCursesBoolWidget.h"
#include "cmCursesFilePathWidget.h"
#include "cmCursesLabelWidget.h"
#include "cmCursesOptionsWidget.h"
#include "cmCursesPathWidget.h"
#include "cmCursesStringWidget.h"
#include "cmCursesWidget.h"
#include "cmProperty.h"
#include "cmState.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

cmCursesCacheEntryComposite::cmCursesCacheEntryComposite(
  const std::string& key, int labelwidth, int entrywidth)
  : Key(key)
  , LabelWidth(labelwidth)
  , EntryWidth(entrywidth)
{
  Label = cm::make_unique<cmCursesLabelWidget>(LabelWidth, 1, 1, 1, key);
  IsNewLabel = cm::make_unique<cmCursesLabelWidget>(1, 1, 1, 1, " ");
  Entry = cm::make_unique<cmCursesStringWidget>(EntryWidth, 1, 1, 1);
}

cmCursesCacheEntryComposite::cmCursesCacheEntryComposite(
  const std::string& key, cmState* state, bool isNew, int labelwidth,
  int entrywidth)
  : Key(key)
  , LabelWidth(labelwidth)
  , EntryWidth(entrywidth)
{
  Label = cm::make_unique<cmCursesLabelWidget>(LabelWidth, 1, 1, 1, key);
  if (isNew) {
    IsNewLabel = cm::make_unique<cmCursesLabelWidget>(1, 1, 1, 1, "*");
  } else {
    IsNewLabel = cm::make_unique<cmCursesLabelWidget>(1, 1, 1, 1, " ");
  }

  cmProp value = state->GetCacheEntryValue(key);
  assert(value);
  switch (state->GetCacheEntryType(key)) {
    case cmStateEnums::BOOL: {
      auto bw = cm::make_unique<cmCursesBoolWidget>(EntryWidth, 1, 1, 1);
      bw->SetValueAsBool(cmIsOn(*value));
      Entry = std::move(bw);
      break;
    }
    case cmStateEnums::PATH: {
      auto pw = cm::make_unique<cmCursesPathWidget>(EntryWidth, 1, 1, 1);
      pw->SetString(*value);
      Entry = std::move(pw);
      break;
    }
    case cmStateEnums::FILEPATH: {
      auto fpw = cm::make_unique<cmCursesFilePathWidget>(EntryWidth, 1, 1, 1);
      fpw->SetString(*value);
      Entry = std::move(fpw);
      break;
    }
    case cmStateEnums::STRING: {
      cmProp stringsProp = state->GetCacheEntryProperty(key, "STRINGS");
      if (stringsProp) {
        auto ow = cm::make_unique<cmCursesOptionsWidget>(EntryWidth, 1, 1, 1);
        for (std::string const& opt : cmExpandedList(*stringsProp)) {
          ow->AddOption(opt);
        }
        ow->SetOption(*value);
        Entry = std::move(ow);
      } else {
        auto sw = cm::make_unique<cmCursesStringWidget>(EntryWidth, 1, 1, 1);
        sw->SetString(*value);
        Entry = std::move(sw);
      }
      break;
    }
    case cmStateEnums::UNINITIALIZED:
      cmSystemTools::Error("Found an undefined variable: " + key);
      break;
    default:
      // TODO : put warning message here
      break;
  }
}

cmCursesCacheEntryComposite::~cmCursesCacheEntryComposite() = default;

const char* cmCursesCacheEntryComposite::GetValue()
{
  if (Label) {
    return Label->GetValue();
  }
  return nullptr;
}
