/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestUpdateHandler.h"

#include <chrono>
#include <sstream>

#include <cm/memory>

#include "cmCLocaleEnvironmentScope.h"
#include "cmCTest.h"
#include "cmCTestBZR.h"
#include "cmCTestCVS.h"
#include "cmCTestGIT.h"
#include "cmCTestHG.h"
#include "cmCTestP4.h"
#include "cmCTestSVN.h"
#include "cmCTestVC.h"
#include "cmGeneratedFileStream.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmVersion.h"
#include "cmXMLWriter.h"

static const char* cmCTestUpdateHandlerUpdateStrings[] = {
  "Unknown", "CVS", "SVN", "BZR", "GIT", "HG", "P4"
};

static const char* cmCTestUpdateHandlerUpdateToString(int type)
{
  if (type < cmCTestUpdateHandler::e_UNKNOWN ||
      type >= cmCTestUpdateHandler::e_LAST) {
    return cmCTestUpdateHandlerUpdateStrings[cmCTestUpdateHandler::e_UNKNOWN];
  }
  return cmCTestUpdateHandlerUpdateStrings[type];
}

cmCTestUpdateHandler::cmCTestUpdateHandler() = default;

void cmCTestUpdateHandler::Initialize()
{
  Superclass::Initialize();
  UpdateCommand.clear();
  UpdateType = e_CVS;
}

int cmCTestUpdateHandler::DetermineType(const char* cmd, const char* type)
{
  cmCTestOptionalLog(CTest, DEBUG,
                     "Determine update type from command: "
                       << cmd << " and type: " << type << std::endl,
                     Quiet);
  if (type && *type) {
    cmCTestOptionalLog(CTest, DEBUG, "Type specified: " << type << std::endl,
                       Quiet);
    std::string stype = cmSystemTools::LowerCase(type);
    if (stype.find("cvs") != std::string::npos) {
      return cmCTestUpdateHandler::e_CVS;
    }
    if (stype.find("svn") != std::string::npos) {
      return cmCTestUpdateHandler::e_SVN;
    }
    if (stype.find("bzr") != std::string::npos) {
      return cmCTestUpdateHandler::e_BZR;
    }
    if (stype.find("git") != std::string::npos) {
      return cmCTestUpdateHandler::e_GIT;
    }
    if (stype.find("hg") != std::string::npos) {
      return cmCTestUpdateHandler::e_HG;
    }
    if (stype.find("p4") != std::string::npos) {
      return cmCTestUpdateHandler::e_P4;
    }
  } else {
    cmCTestOptionalLog(
      CTest, DEBUG, "Type not specified, check command: " << cmd << std::endl,
      Quiet);
    std::string stype = cmSystemTools::LowerCase(cmd);
    if (stype.find("cvs") != std::string::npos) {
      return cmCTestUpdateHandler::e_CVS;
    }
    if (stype.find("svn") != std::string::npos) {
      return cmCTestUpdateHandler::e_SVN;
    }
    if (stype.find("bzr") != std::string::npos) {
      return cmCTestUpdateHandler::e_BZR;
    }
    if (stype.find("git") != std::string::npos) {
      return cmCTestUpdateHandler::e_GIT;
    }
    if (stype.find("hg") != std::string::npos) {
      return cmCTestUpdateHandler::e_HG;
    }
    if (stype.find("p4") != std::string::npos) {
      return cmCTestUpdateHandler::e_P4;
    }
  }
  return cmCTestUpdateHandler::e_UNKNOWN;
}

// clearly it would be nice if this were broken up into a few smaller
// functions and commented...
int cmCTestUpdateHandler::ProcessHandler()
{
  // Make sure VCS tool messages are in English so we can parse them.
  cmCLocaleEnvironmentScope fixLocale;
  static_cast<void>(fixLocale);

  // Get source dir
  const char* sourceDirectory = GetOption("SourceDirectory");
  if (!sourceDirectory) {
    cmCTestLog(CTest, ERROR_MESSAGE,
               "Cannot find SourceDirectory  key in the DartConfiguration.tcl"
                 << std::endl);
    return -1;
  }

  cmGeneratedFileStream ofs;
  if (!CTest->GetShowOnly()) {
    StartLogFile("Update", ofs);
  }

  cmCTestOptionalLog(
    CTest, HANDLER_OUTPUT,
    "   Updating the repository: " << sourceDirectory << std::endl, Quiet);

  if (!SelectVCS()) {
    return -1;
  }

  cmCTestOptionalLog(CTest, HANDLER_OUTPUT,
                     "   Use "
                       << cmCTestUpdateHandlerUpdateToString(UpdateType)
                       << " repository type" << std::endl;
                     , Quiet);

  // Create an object to interact with the VCS tool.
  std::unique_ptr<cmCTestVC> vc;
  switch (UpdateType) {
    case e_CVS:
      vc = cm::make_unique<cmCTestCVS>(CTest, ofs);
      break;
    case e_SVN:
      vc = cm::make_unique<cmCTestSVN>(CTest, ofs);
      break;
    case e_BZR:
      vc = cm::make_unique<cmCTestBZR>(CTest, ofs);
      break;
    case e_GIT:
      vc = cm::make_unique<cmCTestGIT>(CTest, ofs);
      break;
    case e_HG:
      vc = cm::make_unique<cmCTestHG>(CTest, ofs);
      break;
    case e_P4:
      vc = cm::make_unique<cmCTestP4>(CTest, ofs);
      break;
    default:
      vc = cm::make_unique<cmCTestVC>(CTest, ofs);
      break;
  }
  vc->SetCommandLineTool(UpdateCommand);
  vc->SetSourceDirectory(sourceDirectory);

  // Cleanup the working tree.
  vc->Cleanup();

  //
  // Now update repository and remember what files were updated
  //
  cmGeneratedFileStream os;
  if (!StartResultingXML(cmCTest::PartUpdate, "Update", os)) {
    cmCTestLog(CTest, ERROR_MESSAGE, "Cannot open log file" << std::endl);
    return -1;
  }
  std::string start_time = CTest->CurrentTime();
  auto start_time_time = std::chrono::system_clock::now();
  auto elapsed_time_start = std::chrono::steady_clock::now();

  bool updated = vc->Update();
  std::string buildname =
    cmCTest::SafeBuildIdField(CTest->GetCTestConfiguration("BuildName"));

  cmXMLWriter xml(os);
  xml.StartDocument();
  xml.StartElement("Update");
  xml.Attribute("mode", "Client");
  xml.Attribute("Generator",
                std::string("ctest-") + cmVersion::GetCMakeVersion());
  xml.Element("Site", CTest->GetCTestConfiguration("Site"));
  xml.Element("BuildName", buildname);
  xml.Element("BuildStamp",
              CTest->GetCurrentTag() + "-" + CTest->GetTestModelString());
  xml.Element("StartDateTime", start_time);
  xml.Element("StartTime", start_time_time);
  xml.Element("UpdateCommand", vc->GetUpdateCommandLine());
  xml.Element("UpdateType", cmCTestUpdateHandlerUpdateToString(UpdateType));
  std::string changeId = CTest->GetCTestConfiguration("ChangeId");
  if (!changeId.empty()) {
    xml.Element("ChangeId", changeId);
  }

  bool loadedMods = vc->WriteXML(xml);

  int localModifications = 0;
  int numUpdated = vc->GetPathCount(cmCTestVC::PathUpdated);
  if (numUpdated) {
    cmCTestOptionalLog(CTest, HANDLER_OUTPUT,
                       "   Found " << numUpdated << " updated files\n", Quiet);
  }
  if (int numModified = vc->GetPathCount(cmCTestVC::PathModified)) {
    cmCTestOptionalLog(
      CTest, HANDLER_OUTPUT,
      "   Found " << numModified << " locally modified files\n", Quiet);
    localModifications += numModified;
  }
  if (int numConflicting = vc->GetPathCount(cmCTestVC::PathConflicting)) {
    cmCTestOptionalLog(CTest, HANDLER_OUTPUT,
                       "   Found " << numConflicting << " conflicting files\n",
                       Quiet);
    localModifications += numConflicting;
  }

  cmCTestOptionalLog(CTest, DEBUG, "End" << std::endl, Quiet);
  std::string end_time = CTest->CurrentTime();
  xml.Element("EndDateTime", end_time);
  xml.Element("EndTime", std::chrono::system_clock::now());
  xml.Element("ElapsedMinutes",
              std::chrono::duration_cast<std::chrono::minutes>(
                std::chrono::steady_clock::now() - elapsed_time_start)
                .count());

  xml.StartElement("UpdateReturnStatus");
  if (localModifications) {
    xml.Content("Update error: "
                "There are modified or conflicting files in the repository");
    cmCTestLog(CTest, ERROR_MESSAGE,
               "   There are modified or conflicting files in the repository"
                 << std::endl);
  }
  if (!updated) {
    xml.Content("Update command failed:\n");
    xml.Content(vc->GetUpdateCommandLine());
    cmCTestLog(CTest, HANDLER_OUTPUT,
               "   Update command failed: " << vc->GetUpdateCommandLine()
                                            << "\n");
  }
  xml.EndElement(); // UpdateReturnStatus
  xml.EndElement(); // Update
  xml.EndDocument();
  return updated && loadedMods ? numUpdated : -1;
}

int cmCTestUpdateHandler::DetectVCS(const char* dir)
{
  std::string sourceDirectory = dir;
  cmCTestOptionalLog(
    CTest, DEBUG, "Check directory: " << sourceDirectory << std::endl, Quiet);
  sourceDirectory += "/.svn";
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_SVN;
  }
  sourceDirectory = cmStrCat(dir, "/CVS");
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_CVS;
  }
  sourceDirectory = cmStrCat(dir, "/.bzr");
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_BZR;
  }
  sourceDirectory = cmStrCat(dir, "/.git");
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_GIT;
  }
  sourceDirectory = cmStrCat(dir, "/.hg");
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_HG;
  }
  sourceDirectory = cmStrCat(dir, "/.p4");
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_P4;
  }
  sourceDirectory = cmStrCat(dir, "/.p4config");
  if (cmSystemTools::FileExists(sourceDirectory)) {
    return cmCTestUpdateHandler::e_P4;
  }
  return cmCTestUpdateHandler::e_UNKNOWN;
}

bool cmCTestUpdateHandler::SelectVCS()
{
  // Get update command
  UpdateCommand = CTest->GetCTestConfiguration("UpdateCommand");

  // Detect the VCS managing the source tree.
  UpdateType = DetectVCS(GetOption("SourceDirectory"));
  if (UpdateType == e_UNKNOWN) {
    // The source tree does not have a recognized VCS.  Check the
    // configuration value or command name.
    UpdateType =
      DetermineType(UpdateCommand.c_str(),
                    CTest->GetCTestConfiguration("UpdateType").c_str());
  }

  // If no update command was specified, lookup one for this VCS tool.
  if (UpdateCommand.empty()) {
    const char* key = nullptr;
    switch (UpdateType) {
      case e_CVS:
        key = "CVSCommand";
        break;
      case e_SVN:
        key = "SVNCommand";
        break;
      case e_BZR:
        key = "BZRCommand";
        break;
      case e_GIT:
        key = "GITCommand";
        break;
      case e_HG:
        key = "HGCommand";
        break;
      case e_P4:
        key = "P4Command";
        break;
      default:
        break;
    }
    if (key) {
      UpdateCommand = CTest->GetCTestConfiguration(key);
    }
    if (UpdateCommand.empty()) {
      std::ostringstream e;
      e << "Cannot find UpdateCommand ";
      if (key) {
        e << "or " << key;
      }
      e << " configuration key.";
      cmCTestLog(CTest, ERROR_MESSAGE, e.str() << std::endl);
      return false;
    }
  }

  return true;
}
