/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestSubmitCommand.h"

#include <set>
#include <sstream>
#include <utility>

#include <cm/memory>
#include <cm/vector>
#include <cmext/algorithm>
#include <cmext/string_view>

#include "cmCTest.h"
#include "cmCTestSubmitHandler.h"
#include "cmCommand.h"
#include "cmMakefile.h"
#include "cmMessageType.h"
#include "cmProperty.h"
#include "cmRange.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

class cmExecutionStatus;

/**
 * This is a virtual constructor for the command.
 */
std::unique_ptr<cmCommand> cmCTestSubmitCommand::Clone()
{
  auto ni = cm::make_unique<cmCTestSubmitCommand>();
  ni->CTest = CTest;
  ni->CTestScriptHandler = CTestScriptHandler;
  return std::unique_ptr<cmCommand>(std::move(ni));
}

cmCTestGenericHandler* cmCTestSubmitCommand::InitializeHandler()
{
  const std::string* submitURL = !SubmitURL.empty()
    ? &SubmitURL
    : Makefile->GetDefinition("CTEST_SUBMIT_URL");

  if (submitURL) {
    CTest->SetCTestConfiguration("SubmitURL", *submitURL, Quiet);
  } else {
    CTest->SetCTestConfigurationFromCMakeVariable(Makefile, "DropMethod",
                                                  "CTEST_DROP_METHOD", Quiet);
    CTest->SetCTestConfigurationFromCMakeVariable(
      Makefile, "DropSiteUser", "CTEST_DROP_SITE_USER", Quiet);
    CTest->SetCTestConfigurationFromCMakeVariable(
      Makefile, "DropSitePassword", "CTEST_DROP_SITE_PASSWORD", Quiet);
    CTest->SetCTestConfigurationFromCMakeVariable(Makefile, "DropSite",
                                                  "CTEST_DROP_SITE", Quiet);
    CTest->SetCTestConfigurationFromCMakeVariable(
      Makefile, "DropLocation", "CTEST_DROP_LOCATION", Quiet);
  }

  CTest->SetCTestConfigurationFromCMakeVariable(Makefile, "CurlOptions",
                                                "CTEST_CURL_OPTIONS", Quiet);

  cmProp notesFilesVariable = Makefile->GetDefinition("CTEST_NOTES_FILES");
  if (notesFilesVariable) {
    std::vector<std::string> notesFiles = cmExpandedList(*notesFilesVariable);
    CTest->GenerateNotesFile(notesFiles);
  }

  cmProp extraFilesVariable =
    Makefile->GetDefinition("CTEST_EXTRA_SUBMIT_FILES");
  if (extraFilesVariable) {
    std::vector<std::string> extraFiles = cmExpandedList(*extraFilesVariable);
    if (!CTest->SubmitExtraFiles(extraFiles)) {
      SetError("problem submitting extra files.");
      return nullptr;
    }
  }

  cmCTestSubmitHandler* handler = CTest->GetSubmitHandler();
  handler->Initialize();

  // If no FILES or PARTS given, *all* PARTS are submitted by default.
  //
  // If FILES are given, but not PARTS, only the FILES are submitted
  // and *no* PARTS are submitted.
  //  (This is why we select the empty "noParts" set in the
  //   FilesMentioned block below...)
  //
  // If PARTS are given, only the selected PARTS are submitted.
  //
  // If both PARTS and FILES are given, only the selected PARTS *and*
  // all the given FILES are submitted.

  // If given explicit FILES to submit, pass them to the handler.
  //
  if (FilesMentioned) {
    // Intentionally select *no* PARTS. (Pass an empty set.) If PARTS
    // were also explicitly mentioned, they will be selected below...
    // But FILES with no PARTS mentioned should just submit the FILES
    // without any of the default parts.
    //
    handler->SelectParts(std::set<cmCTest::Part>());
    handler->SelectFiles(std::set<std::string>(Files.begin(), Files.end()));
  }

  // If a PARTS option was given, select only the named parts for submission.
  //
  if (PartsMentioned) {
    auto parts = cmMakeRange(Parts).transform(
      [this](std::string const& arg) { return CTest->GetPartFromName(arg); });
    handler->SelectParts(std::set<cmCTest::Part>(parts.begin(), parts.end()));
  }

  // Pass along any HTTPHEADER to the handler if this option was given.
  if (!HttpHeaders.empty()) {
    handler->SetHttpHeaders(HttpHeaders);
  }

  handler->SetOption("RetryDelay", RetryDelay.c_str());
  handler->SetOption("RetryCount", RetryCount.c_str());
  handler->SetOption("InternalTest", InternalTest ? "ON" : "OFF");

  handler->SetQuiet(Quiet);

  if (CDashUpload) {
    handler->SetOption("CDashUploadFile", CDashUploadFile.c_str());
    handler->SetOption("CDashUploadType", CDashUploadType.c_str());
  }
  return handler;
}

bool cmCTestSubmitCommand::InitialPass(std::vector<std::string> const& args,
                                       cmExecutionStatus& status)
{
  CDashUpload = !args.empty() && args[0] == "CDASH_UPLOAD";

  bool ret = cmCTestHandlerCommand::InitialPass(args, status);

  if (!BuildID.empty()) {
    Makefile->AddDefinition(BuildID, CTest->GetBuildID());
  }

  return ret;
}

void cmCTestSubmitCommand::BindArguments()
{
  if (CDashUpload) {
    // Arguments specific to the CDASH_UPLOAD signature.
    Bind("CDASH_UPLOAD", CDashUploadFile);
    Bind("CDASH_UPLOAD_TYPE", CDashUploadType);
  } else {
    // Arguments that cannot be used with CDASH_UPLOAD.
    Bind("PARTS"_s, Parts);
    Bind("FILES"_s, Files);
  }
  // Arguments used by both modes.
  Bind("BUILD_ID"_s, BuildID);
  Bind("HTTPHEADER"_s, HttpHeaders);
  Bind("RETRY_COUNT"_s, RetryCount);
  Bind("RETRY_DELAY"_s, RetryDelay);
  Bind("SUBMIT_URL"_s, SubmitURL);
  Bind("INTERNAL_TEST_CHECKSUM", InternalTest);

  // Look for other arguments.
  cmCTestHandlerCommand::BindArguments();
}

void cmCTestSubmitCommand::CheckArguments(
  std::vector<std::string> const& keywords)
{
  PartsMentioned = !Parts.empty() || cm::contains(keywords, "PARTS");
  FilesMentioned = !Files.empty() || cm::contains(keywords, "FILES");

  cm::erase_if(Parts, [this](std::string const& arg) -> bool {
    cmCTest::Part p = CTest->GetPartFromName(arg);
    if (p == cmCTest::PartCount) {
      std::ostringstream e;
      e << "Part name \"" << arg << "\" is invalid.";
      Makefile->IssueMessage(MessageType::FATAL_ERROR, e.str());
      return true;
    }
    return false;
  });

  cm::erase_if(Files, [this](std::string const& arg) -> bool {
    if (!cmSystemTools::FileExists(arg)) {
      std::ostringstream e;
      e << "File \"" << arg << "\" does not exist. Cannot submit "
        << "a non-existent file.";
      Makefile->IssueMessage(MessageType::FATAL_ERROR, e.str());
      return true;
    }
    return false;
  });
}
