/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestSVN.h"

#include <cstdlib>
#include <cstring>
#include <map>
#include <ostream>

#include <cmext/algorithm>

#include "cmsys/RegularExpression.hxx"

#include "cmCTest.h"
#include "cmCTestVC.h"
#include "cmProcessTools.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmXMLParser.h"
#include "cmXMLWriter.h"

struct cmCTestSVN::Revision : public cmCTestVC::Revision
{
  cmCTestSVN::SVNInfo* SVNInfo;
};

cmCTestSVN::cmCTestSVN(cmCTest* ct, std::ostream& log)
  : cmCTestGlobalVC(ct, log)
{
  PriorRev = Unknown;
}

cmCTestSVN::~cmCTestSVN() = default;

void cmCTestSVN::CleanupImpl()
{
  std::vector<const char*> svn_cleanup;
  svn_cleanup.push_back("cleanup");
  OutputLogger out(Log, "cleanup-out> ");
  OutputLogger err(Log, "cleanup-err> ");
  RunSVNCommand(svn_cleanup, &out, &err);
}

class cmCTestSVN::InfoParser : public cmCTestVC::LineParser
{
public:
  InfoParser(cmCTestSVN* svn, const char* prefix, std::string& rev,
             SVNInfo& svninfo)
    : Rev(rev)
    , SVNRepo(svninfo)
  {
    SetLog(&svn->Log, prefix);
    RegexRev.compile("^Revision: ([0-9]+)");
    RegexURL.compile("^URL: +([^ ]+) *$");
    RegexRoot.compile("^Repository Root: +([^ ]+) *$");
  }

private:
  std::string& Rev;
  cmCTestSVN::SVNInfo& SVNRepo;
  cmsys::RegularExpression RegexRev;
  cmsys::RegularExpression RegexURL;
  cmsys::RegularExpression RegexRoot;
  bool ProcessLine() override
  {
    if (RegexRev.find(Line)) {
      Rev = RegexRev.match(1);
    } else if (RegexURL.find(Line)) {
      SVNRepo.URL = RegexURL.match(1);
    } else if (RegexRoot.find(Line)) {
      SVNRepo.Root = RegexRoot.match(1);
    }
    return true;
  }
};

static bool cmCTestSVNPathStarts(std::string const& p1, std::string const& p2)
{
  // Does path p1 start with path p2?
  if (p1.size() == p2.size()) {
    return p1 == p2;
  }
  if (p1.size() > p2.size() && p1[p2.size()] == '/') {
    return strncmp(p1.c_str(), p2.c_str(), p2.size()) == 0;
  }
  return false;
}

std::string cmCTestSVN::LoadInfo(SVNInfo& svninfo)
{
  // Run "svn info" to get the repository info from the work tree.
  std::vector<const char*> svn_info;
  svn_info.push_back("info");
  svn_info.push_back(svninfo.LocalPath.c_str());
  std::string rev;
  InfoParser out(this, "info-out> ", rev, svninfo);
  OutputLogger err(Log, "info-err> ");
  RunSVNCommand(svn_info, &out, &err);
  return rev;
}

bool cmCTestSVN::NoteOldRevision()
{
  if (!LoadRepositories()) {
    return false;
  }

  for (SVNInfo& svninfo : Repositories) {
    svninfo.OldRevision = LoadInfo(svninfo);
    Log << "Revision for repository '" << svninfo.LocalPath
        << "' before update: " << svninfo.OldRevision << "\n";
    cmCTestLog(CTest, HANDLER_OUTPUT,
               "   Old revision of external repository '"
                 << svninfo.LocalPath << "' is: " << svninfo.OldRevision
                 << "\n");
  }

  // Set the global old revision to the one of the root
  OldRevision = RootInfo->OldRevision;
  PriorRev.Rev = OldRevision;
  return true;
}

bool cmCTestSVN::NoteNewRevision()
{
  if (!LoadRepositories()) {
    return false;
  }

  for (SVNInfo& svninfo : Repositories) {
    svninfo.NewRevision = LoadInfo(svninfo);
    Log << "Revision for repository '" << svninfo.LocalPath
        << "' after update: " << svninfo.NewRevision << "\n";
    cmCTestLog(CTest, HANDLER_OUTPUT,
               "   New revision of external repository '"
                 << svninfo.LocalPath << "' is: " << svninfo.NewRevision
                 << "\n");

    // svninfo.Root = ""; // uncomment to test GuessBase
    Log << "Repository '" << svninfo.LocalPath << "' URL = " << svninfo.URL
        << "\n";
    Log << "Repository '" << svninfo.LocalPath << "' Root = " << svninfo.Root
        << "\n";

    // Compute the base path the working tree has checked out under
    // the repository root.
    if (!svninfo.Root.empty() &&
        cmCTestSVNPathStarts(svninfo.URL, svninfo.Root)) {
      svninfo.Base = cmStrCat(
        cmCTest::DecodeURL(svninfo.URL.substr(svninfo.Root.size())), '/');
    }
    Log << "Repository '" << svninfo.LocalPath << "' Base = " << svninfo.Base
        << "\n";
  }

  // Set the global new revision to the one of the root
  NewRevision = RootInfo->NewRevision;
  return true;
}

void cmCTestSVN::GuessBase(SVNInfo& svninfo,
                           std::vector<Change> const& changes)
{
  // Subversion did not give us a good repository root so we need to
  // guess the base path from the URL and the paths in a revision with
  // changes under it.

  // Consider each possible URL suffix from longest to shortest.
  for (std::string::size_type slash = svninfo.URL.find('/');
       svninfo.Base.empty() && slash != std::string::npos;
       slash = svninfo.URL.find('/', slash + 1)) {
    // If the URL suffix is a prefix of at least one path then it is the base.
    std::string base = cmCTest::DecodeURL(svninfo.URL.substr(slash));
    for (auto ci = changes.begin();
         svninfo.Base.empty() && ci != changes.end(); ++ci) {
      if (cmCTestSVNPathStarts(ci->Path, base)) {
        svninfo.Base = base;
      }
    }
  }

  // We always append a slash so that we know paths beginning in the
  // base lie under its path.  If no base was found then the working
  // tree must be a checkout of the entire repo and this will match
  // the leading slash in all paths.
  svninfo.Base += "/";

  Log << "Guessed Base = " << svninfo.Base << "\n";
}

class cmCTestSVN::UpdateParser : public cmCTestVC::LineParser
{
public:
  UpdateParser(cmCTestSVN* svn, const char* prefix)
    : SVN(svn)
  {
    SetLog(&svn->Log, prefix);
    RegexUpdate.compile("^([ADUCGE ])([ADUCGE ])[B ] +(.+)$");
  }

private:
  cmCTestSVN* SVN;
  cmsys::RegularExpression RegexUpdate;

  bool ProcessLine() override
  {
    if (RegexUpdate.find(Line)) {
      DoPath(RegexUpdate.match(1)[0], RegexUpdate.match(2)[0],
             RegexUpdate.match(3));
    }
    return true;
  }

  void DoPath(char path_status, char prop_status, std::string const& path)
  {
    char status = (path_status != ' ') ? path_status : prop_status;
    std::string dir = cmSystemTools::GetFilenamePath(path);
    std::string name = cmSystemTools::GetFilenameName(path);
    // See "svn help update".
    switch (status) {
      case 'G':
        SVN->Dirs[dir][name].Status = PathModified;
        break;
      case 'C':
        SVN->Dirs[dir][name].Status = PathConflicting;
        break;
      case 'A':
      case 'D':
      case 'U':
        SVN->Dirs[dir][name].Status = PathUpdated;
        break;
      case 'E': // TODO?
      case '?':
      case ' ':
      default:
        break;
    }
  }
};

bool cmCTestSVN::UpdateImpl()
{
  // Get user-specified update options.
  std::string opts = CTest->GetCTestConfiguration("UpdateOptions");
  if (opts.empty()) {
    opts = CTest->GetCTestConfiguration("SVNUpdateOptions");
  }
  std::vector<std::string> args = cmSystemTools::ParseArguments(opts);

  // Specify the start time for nightly testing.
  if (CTest->GetTestModel() == cmCTest::NIGHTLY) {
    args.push_back("-r{" + GetNightlyTime() + " +0000}");
  }

  std::vector<char const*> svn_update;
  svn_update.push_back("update");
  for (std::string const& arg : args) {
    svn_update.push_back(arg.c_str());
  }

  UpdateParser out(this, "up-out> ");
  OutputLogger err(Log, "up-err> ");
  return RunSVNCommand(svn_update, &out, &err);
}

bool cmCTestSVN::RunSVNCommand(std::vector<char const*> const& parameters,
                               OutputParser* out, OutputParser* err)
{
  if (parameters.empty()) {
    return false;
  }

  std::vector<char const*> args;
  args.push_back(CommandLineTool.c_str());
  cm::append(args, parameters);
  args.push_back("--non-interactive");

  std::string userOptions = CTest->GetCTestConfiguration("SVNOptions");

  std::vector<std::string> parsedUserOptions =
    cmSystemTools::ParseArguments(userOptions);
  for (std::string const& opt : parsedUserOptions) {
    args.push_back(opt.c_str());
  }

  args.push_back(nullptr);

  if (strcmp(parameters[0], "update") == 0) {
    return RunUpdateCommand(&args[0], out, err);
  }
  return RunChild(&args[0], out, err);
}

class cmCTestSVN::LogParser
  : public cmCTestVC::OutputLogger
  , private cmXMLParser
{
public:
  LogParser(cmCTestSVN* svn, const char* prefix, SVNInfo& svninfo)
    : OutputLogger(svn->Log, prefix)
    , SVN(svn)
    , SVNRepo(svninfo)
  {
    InitializeParser();
  }
  ~LogParser() override { CleanupParser(); }

private:
  cmCTestSVN* SVN;
  cmCTestSVN::SVNInfo& SVNRepo;

  using Revision = cmCTestSVN::Revision;
  using Change = cmCTestSVN::Change;
  Revision Rev;
  std::vector<Change> Changes;
  Change CurChange;
  std::vector<char> CData;

  bool ProcessChunk(const char* data, int length) override
  {
    OutputLogger::ProcessChunk(data, length);
    ParseChunk(data, length);
    return true;
  }

  void StartElement(const std::string& name, const char** atts) override
  {
    CData.clear();
    if (name == "logentry") {
      Rev = Revision();
      Rev.SVNInfo = &SVNRepo;
      if (const char* rev =
            cmCTestSVN::LogParser::FindAttribute(atts, "revision")) {
        Rev.Rev = rev;
      }
      Changes.clear();
    } else if (name == "path") {
      CurChange = Change();
      if (const char* action =
            cmCTestSVN::LogParser::FindAttribute(atts, "action")) {
        CurChange.Action = action[0];
      }
    }
  }

  void CharacterDataHandler(const char* data, int length) override
  {
    cm::append(CData, data, data + length);
  }

  void EndElement(const std::string& name) override
  {
    if (name == "logentry") {
      SVN->DoRevisionSVN(Rev, Changes);
    } else if (!CData.empty() && name == "path") {
      std::string orig_path(&CData[0], CData.size());
      std::string new_path = SVNRepo.BuildLocalPath(orig_path);
      CurChange.Path.assign(new_path);
      Changes.push_back(CurChange);
    } else if (!CData.empty() && name == "author") {
      Rev.Author.assign(&CData[0], CData.size());
    } else if (!CData.empty() && name == "date") {
      Rev.Date.assign(&CData[0], CData.size());
    } else if (!CData.empty() && name == "msg") {
      Rev.Log.assign(&CData[0], CData.size());
    }
    CData.clear();
  }

  void ReportError(int /*line*/, int /*column*/, const char* msg) override
  {
    SVN->Log << "Error parsing svn log xml: " << msg << "\n";
  }
};

bool cmCTestSVN::LoadRevisions()
{
  bool result = true;
  // Get revisions for all the external repositories
  for (SVNInfo& svninfo : Repositories) {
    result = LoadRevisions(svninfo) && result;
  }
  return result;
}

bool cmCTestSVN::LoadRevisions(SVNInfo& svninfo)
{
  // We are interested in every revision included in the update.
  std::string revs;
  if (atoi(svninfo.OldRevision.c_str()) < atoi(svninfo.NewRevision.c_str())) {
    revs = "-r" + svninfo.OldRevision + ":" + svninfo.NewRevision;
  } else {
    revs = "-r" + svninfo.NewRevision;
  }

  // Run "svn log" to get all global revisions of interest.
  std::vector<const char*> svn_log;
  svn_log.push_back("log");
  svn_log.push_back("--xml");
  svn_log.push_back("-v");
  svn_log.push_back(revs.c_str());
  svn_log.push_back(svninfo.LocalPath.c_str());
  LogParser out(this, "log-out> ", svninfo);
  OutputLogger err(Log, "log-err> ");
  return RunSVNCommand(svn_log, &out, &err);
}

void cmCTestSVN::DoRevisionSVN(Revision const& revision,
                               std::vector<Change> const& changes)
{
  // Guess the base checkout path from the changes if necessary.
  if (RootInfo->Base.empty() && !changes.empty()) {
    GuessBase(*RootInfo, changes);
  }

  // Ignore changes in the old revision for external repositories
  if (revision.Rev == revision.SVNInfo->OldRevision &&
      !revision.SVNInfo->LocalPath.empty()) {
    return;
  }

  cmCTestGlobalVC::DoRevision(revision, changes);
}

class cmCTestSVN::StatusParser : public cmCTestVC::LineParser
{
public:
  StatusParser(cmCTestSVN* svn, const char* prefix)
    : SVN(svn)
  {
    SetLog(&svn->Log, prefix);
    RegexStatus.compile("^([ACDIMRX?!~ ])([CM ])[ L]... +(.+)$");
  }

private:
  cmCTestSVN* SVN;
  cmsys::RegularExpression RegexStatus;
  bool ProcessLine() override
  {
    if (RegexStatus.find(Line)) {
      DoPath(RegexStatus.match(1)[0], RegexStatus.match(2)[0],
             RegexStatus.match(3));
    }
    return true;
  }

  void DoPath(char path_status, char prop_status, std::string const& path)
  {
    char status = (path_status != ' ') ? path_status : prop_status;
    // See "svn help status".
    switch (status) {
      case 'M':
      case '!':
      case 'A':
      case 'D':
      case 'R':
        SVN->DoModification(PathModified, path);
        break;
      case 'C':
      case '~':
        SVN->DoModification(PathConflicting, path);
        break;
      case 'X':
      case 'I':
      case '?':
      case ' ':
      default:
        break;
    }
  }
};

bool cmCTestSVN::LoadModifications()
{
  // Run "svn status" which reports local modifications.
  std::vector<const char*> svn_status;
  svn_status.push_back("status");
  StatusParser out(this, "status-out> ");
  OutputLogger err(Log, "status-err> ");
  RunSVNCommand(svn_status, &out, &err);
  return true;
}

void cmCTestSVN::WriteXMLGlobal(cmXMLWriter& xml)
{
  cmCTestGlobalVC::WriteXMLGlobal(xml);

  xml.Element("SVNPath", RootInfo->Base);
}

class cmCTestSVN::ExternalParser : public cmCTestVC::LineParser
{
public:
  ExternalParser(cmCTestSVN* svn, const char* prefix)
    : SVN(svn)
  {
    SetLog(&svn->Log, prefix);
    RegexExternal.compile("^X..... +(.+)$");
  }

private:
  cmCTestSVN* SVN;
  cmsys::RegularExpression RegexExternal;
  bool ProcessLine() override
  {
    if (RegexExternal.find(Line)) {
      DoPath(RegexExternal.match(1));
    }
    return true;
  }

  void DoPath(std::string const& path)
  {
    // Get local path relative to the source directory
    std::string local_path;
    if (path.size() > SVN->SourceDirectory.size() &&
        strncmp(path.c_str(), SVN->SourceDirectory.c_str(),
                SVN->SourceDirectory.size()) == 0) {
      local_path = path.substr(SVN->SourceDirectory.size() + 1);
    } else {
      local_path = path;
    }
    SVN->Repositories.emplace_back(local_path);
  }
};

bool cmCTestSVN::LoadRepositories()
{
  if (!Repositories.empty()) {
    return true;
  }

  // Info for root repository
  Repositories.emplace_back();
  RootInfo = &(Repositories.back());

  // Run "svn status" to get the list of external repositories
  std::vector<const char*> svn_status;
  svn_status.push_back("status");
  ExternalParser out(this, "external-out> ");
  OutputLogger err(Log, "external-err> ");
  return RunSVNCommand(svn_status, &out, &err);
}

std::string cmCTestSVN::SVNInfo::BuildLocalPath(std::string const& path) const
{
  std::string local_path;

  // Add local path prefix if not empty
  if (!LocalPath.empty()) {
    local_path += LocalPath;
    local_path += "/";
  }

  // Add path with base prefix removed
  if (path.size() > Base.size() &&
      strncmp(path.c_str(), Base.c_str(), Base.size()) == 0) {
    local_path += path.substr(Base.size());
  } else {
    local_path += path;
  }

  return local_path;
}
