/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestGlobalVC.h"

#include <ostream>
#include <utility>

#include "cmCTest.h"
#include "cmSystemTools.h"
#include "cmXMLWriter.h"

cmCTestGlobalVC::cmCTestGlobalVC(cmCTest* ct, std::ostream& log)
  : cmCTestVC(ct, log)
{
  PriorRev = Unknown;
}

cmCTestGlobalVC::~cmCTestGlobalVC() = default;

const char* cmCTestGlobalVC::LocalPath(std::string const& path)
{
  return path.c_str();
}

void cmCTestGlobalVC::DoRevision(Revision const& revision,
                                 std::vector<Change> const& changes)
{
  // Ignore changes in the old revision.
  if (revision.Rev == OldRevision) {
    PriorRev = revision;
    return;
  }

  // Indicate we found a revision.
  cmCTestLog(CTest, HANDLER_OUTPUT, "." << std::flush);

  // Store the revision.
  Revisions.push_back(revision);

  // Report this revision.
  Revision const& rev = Revisions.back();
  /* clang-format off */
  Log << "Found revision " << rev.Rev << "\n"
            << "  author = " << rev.Author << "\n"
            << "  date = " << rev.Date << "\n";
  /* clang-format on */

  // Update information about revisions of the changed files.
  for (Change const& c : changes) {
    if (const char* local = LocalPath(c.Path)) {
      std::string dir = cmSystemTools::GetFilenamePath(local);
      std::string name = cmSystemTools::GetFilenameName(local);
      File& file = Dirs[dir][name];
      file.PriorRev = file.Rev ? file.Rev : &PriorRev;
      file.Rev = &rev;
      Log << "  " << c.Action << " " << local << " "
          << "\n";
    }
  }
}

void cmCTestGlobalVC::DoModification(PathStatus status,
                                     std::string const& path)
{
  std::string dir = cmSystemTools::GetFilenamePath(path);
  std::string name = cmSystemTools::GetFilenameName(path);
  File& file = Dirs[dir][name];
  file.Status = status;
  // For local modifications the current rev is unknown and the
  // prior rev is the latest from svn.
  if (!file.Rev && !file.PriorRev) {
    file.PriorRev = &PriorRev;
  }
}

void cmCTestGlobalVC::WriteXMLDirectory(cmXMLWriter& xml,
                                        std::string const& path,
                                        Directory const& dir)
{
  const char* slash = path.empty() ? "" : "/";
  xml.StartElement("Directory");
  xml.Element("Name", path);
  for (auto const& f : dir) {
    std::string const full = path + slash + f.first;
    WriteXMLEntry(xml, path, f.first, full, f.second);
  }
  xml.EndElement(); // Directory
}

void cmCTestGlobalVC::WriteXMLGlobal(cmXMLWriter& xml)
{
  if (!NewRevision.empty()) {
    xml.Element("Revision", NewRevision);
  }
  if (!OldRevision.empty() && OldRevision != NewRevision) {
    xml.Element("PriorRevision", OldRevision);
  }
}

bool cmCTestGlobalVC::WriteXMLUpdates(cmXMLWriter& xml)
{
  bool result = true;
  cmCTestLog(CTest, HANDLER_OUTPUT,
             "   Gathering version information (one . per revision):\n"
             "    "
               << std::flush);
  result = LoadRevisions() && result;
  cmCTestLog(CTest, HANDLER_OUTPUT, std::endl);

  result = LoadModifications() && result;

  WriteXMLGlobal(xml);

  for (auto const& d : Dirs) {
    WriteXMLDirectory(xml, d.first, d.second);
  }

  return result;
}

void cmCTestGlobalVC::SetNewRevision(std::string const& revision)
{
  NewRevision = revision;
}
