/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestGIT.h"

#include <cctype>
#include <cstdio>
#include <cstdlib>
#include <ctime>
#include <utility>
#include <vector>

#include "cmsys/FStream.hxx"
#include "cmsys/Process.h"

#include "cmCTest.h"
#include "cmCTestVC.h"
#include "cmProcessOutput.h"
#include "cmProcessTools.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

static unsigned int cmCTestGITVersion(unsigned int epic, unsigned int major,
                                      unsigned int minor, unsigned int fix)
{
  // 1.6.5.0 maps to 10605000
  return fix + minor * 1000 + major * 100000 + epic * 10000000;
}

cmCTestGIT::cmCTestGIT(cmCTest* ct, std::ostream& log)
  : cmCTestGlobalVC(ct, log)
{
  PriorRev = Unknown;
  CurrentGitVersion = 0;
}

cmCTestGIT::~cmCTestGIT() = default;

class cmCTestGIT::OneLineParser : public cmCTestVC::LineParser
{
public:
  OneLineParser(cmCTestGIT* git, const char* prefix, std::string& l)
    : Line1(l)
  {
    SetLog(&git->Log, prefix);
  }

private:
  std::string& Line1;
  bool ProcessLine() override
  {
    // Only the first line is of interest.
    Line1 = Line;
    return false;
  }
};

std::string cmCTestGIT::GetWorkingRevision()
{
  // Run plumbing "git rev-list" to get work tree revision.
  const char* git = CommandLineTool.c_str();
  const char* git_rev_list[] = { git,    "rev-list", "-n",   "1",
                                 "HEAD", "--",       nullptr };
  std::string rev;
  OneLineParser out(this, "rl-out> ", rev);
  OutputLogger err(Log, "rl-err> ");
  RunChild(git_rev_list, &out, &err);
  return rev;
}

bool cmCTestGIT::NoteOldRevision()
{
  OldRevision = GetWorkingRevision();
  cmCTestLog(CTest, HANDLER_OUTPUT,
             "   Old revision of repository is: " << OldRevision << "\n");
  PriorRev.Rev = OldRevision;
  return true;
}

bool cmCTestGIT::NoteNewRevision()
{
  NewRevision = GetWorkingRevision();
  cmCTestLog(CTest, HANDLER_OUTPUT,
             "   New revision of repository is: " << NewRevision << "\n");
  return true;
}

std::string cmCTestGIT::FindGitDir()
{
  std::string git_dir;

  // Run "git rev-parse --git-dir" to locate the real .git directory.
  const char* git = CommandLineTool.c_str();
  char const* git_rev_parse[] = { git, "rev-parse", "--git-dir", nullptr };
  std::string git_dir_line;
  OneLineParser rev_parse_out(this, "rev-parse-out> ", git_dir_line);
  OutputLogger rev_parse_err(Log, "rev-parse-err> ");
  if (RunChild(git_rev_parse, &rev_parse_out, &rev_parse_err, nullptr,
               cmProcessOutput::UTF8)) {
    git_dir = git_dir_line;
  }
  if (git_dir.empty()) {
    git_dir = ".git";
  }

  // Git reports a relative path only when the .git directory is in
  // the current directory.
  if (git_dir[0] == '.') {
    git_dir = SourceDirectory + "/" + git_dir;
  }
#if defined(_WIN32) && !defined(__CYGWIN__)
  else if (git_dir[0] == '/') {
    // Cygwin Git reports a full path that Cygwin understands, but we
    // are a Windows application.  Run "cygpath" to get Windows path.
    std::string cygpath_exe =
      cmStrCat(cmSystemTools::GetFilenamePath(git), "/cygpath.exe");
    if (cmSystemTools::FileExists(cygpath_exe)) {
      char const* cygpath[] = { cygpath_exe.c_str(), "-w", git_dir.c_str(),
                                0 };
      OneLineParser cygpath_out(this, "cygpath-out> ", git_dir_line);
      OutputLogger cygpath_err(this->Log, "cygpath-err> ");
      if (this->RunChild(cygpath, &cygpath_out, &cygpath_err, nullptr,
                         cmProcessOutput::UTF8)) {
        git_dir = git_dir_line;
      }
    }
  }
#endif
  return git_dir;
}

std::string cmCTestGIT::FindTopDir()
{
  std::string top_dir = SourceDirectory;

  // Run "git rev-parse --show-cdup" to locate the top of the tree.
  const char* git = CommandLineTool.c_str();
  char const* git_rev_parse[] = { git, "rev-parse", "--show-cdup", nullptr };
  std::string cdup;
  OneLineParser rev_parse_out(this, "rev-parse-out> ", cdup);
  OutputLogger rev_parse_err(Log, "rev-parse-err> ");
  if (RunChild(git_rev_parse, &rev_parse_out, &rev_parse_err, nullptr,
               cmProcessOutput::UTF8) &&
      !cdup.empty()) {
    top_dir += "/";
    top_dir += cdup;
    top_dir = cmSystemTools::CollapseFullPath(top_dir);
  }
  return top_dir;
}

bool cmCTestGIT::UpdateByFetchAndReset()
{
  const char* git = CommandLineTool.c_str();

  // Use "git fetch" to get remote commits.
  std::vector<char const*> git_fetch;
  git_fetch.push_back(git);
  git_fetch.push_back("fetch");

  // Add user-specified update options.
  std::string opts = CTest->GetCTestConfiguration("UpdateOptions");
  if (opts.empty()) {
    opts = CTest->GetCTestConfiguration("GITUpdateOptions");
  }
  std::vector<std::string> args = cmSystemTools::ParseArguments(opts);
  for (std::string const& arg : args) {
    git_fetch.push_back(arg.c_str());
  }

  // Sentinel argument.
  git_fetch.push_back(nullptr);

  // Fetch upstream refs.
  OutputLogger fetch_out(Log, "fetch-out> ");
  OutputLogger fetch_err(Log, "fetch-err> ");
  if (!RunUpdateCommand(&git_fetch[0], &fetch_out, &fetch_err)) {
    return false;
  }

  // Identify the merge head that would be used by "git pull".
  std::string sha1;
  {
    std::string fetch_head = FindGitDir() + "/FETCH_HEAD";
    cmsys::ifstream fin(fetch_head.c_str(), std::ios::in | std::ios::binary);
    if (!fin) {
      Log << "Unable to open " << fetch_head << "\n";
      return false;
    }
    std::string line;
    while (sha1.empty() && cmSystemTools::GetLineFromStream(fin, line)) {
      Log << "FETCH_HEAD> " << line << "\n";
      if (line.find("\tnot-for-merge\t") == std::string::npos) {
        std::string::size_type pos = line.find('\t');
        if (pos != std::string::npos) {
          sha1 = std::move(line);
          sha1.resize(pos);
        }
      }
    }
    if (sha1.empty()) {
      Log << "FETCH_HEAD has no upstream branch candidate!\n";
      return false;
    }
  }

  // Reset the local branch to point at that tracked from upstream.
  char const* git_reset[] = { git, "reset", "--hard", sha1.c_str(), nullptr };
  OutputLogger reset_out(Log, "reset-out> ");
  OutputLogger reset_err(Log, "reset-err> ");
  return RunChild(&git_reset[0], &reset_out, &reset_err);
}

bool cmCTestGIT::UpdateByCustom(std::string const& custom)
{
  std::vector<std::string> git_custom_command = cmExpandedList(custom, true);
  std::vector<char const*> git_custom;
  git_custom.reserve(git_custom_command.size() + 1);
  for (std::string const& i : git_custom_command) {
    git_custom.push_back(i.c_str());
  }
  git_custom.push_back(nullptr);

  OutputLogger custom_out(Log, "custom-out> ");
  OutputLogger custom_err(Log, "custom-err> ");
  return RunUpdateCommand(&git_custom[0], &custom_out, &custom_err);
}

bool cmCTestGIT::UpdateInternal()
{
  std::string custom = CTest->GetCTestConfiguration("GITUpdateCustom");
  if (!custom.empty()) {
    return UpdateByCustom(custom);
  }
  return UpdateByFetchAndReset();
}

bool cmCTestGIT::UpdateImpl()
{
  if (!UpdateInternal()) {
    return false;
  }

  std::string top_dir = FindTopDir();
  const char* git = CommandLineTool.c_str();
  const char* recursive = "--recursive";
  const char* sync_recursive = "--recursive";

  // Git < 1.6.5 did not support submodule --recursive
  if (GetGitVersion() < cmCTestGITVersion(1, 6, 5, 0)) {
    recursive = nullptr;
    // No need to require >= 1.6.5 if there are no submodules.
    if (cmSystemTools::FileExists(top_dir + "/.gitmodules")) {
      Log << "Git < 1.6.5 cannot update submodules recursively\n";
    }
  }

  // Git < 1.8.1 did not support sync --recursive
  if (GetGitVersion() < cmCTestGITVersion(1, 8, 1, 0)) {
    sync_recursive = nullptr;
    // No need to require >= 1.8.1 if there are no submodules.
    if (cmSystemTools::FileExists(top_dir + "/.gitmodules")) {
      Log << "Git < 1.8.1 cannot synchronize submodules recursively\n";
    }
  }

  OutputLogger submodule_out(Log, "submodule-out> ");
  OutputLogger submodule_err(Log, "submodule-err> ");

  bool ret;

  std::string init_submodules =
    CTest->GetCTestConfiguration("GITInitSubmodules");
  if (cmIsOn(init_submodules)) {
    char const* git_submodule_init[] = { git, "submodule", "init", nullptr };
    ret = RunChild(git_submodule_init, &submodule_out, &submodule_err,
                   top_dir.c_str());

    if (!ret) {
      return false;
    }
  }

  char const* git_submodule_sync[] = { git, "submodule", "sync",
                                       sync_recursive, nullptr };
  ret = RunChild(git_submodule_sync, &submodule_out, &submodule_err,
                 top_dir.c_str());

  if (!ret) {
    return false;
  }

  char const* git_submodule[] = { git, "submodule", "update", recursive,
                                  nullptr };
  return RunChild(git_submodule, &submodule_out, &submodule_err,
                  top_dir.c_str());
}

unsigned int cmCTestGIT::GetGitVersion()
{
  if (!CurrentGitVersion) {
    const char* git = CommandLineTool.c_str();
    char const* git_version[] = { git, "--version", nullptr };
    std::string version;
    OneLineParser version_out(this, "version-out> ", version);
    OutputLogger version_err(Log, "version-err> ");
    unsigned int v[4] = { 0, 0, 0, 0 };
    if (RunChild(git_version, &version_out, &version_err) &&
        sscanf(version.c_str(), "git version %u.%u.%u.%u", &v[0], &v[1], &v[2],
               &v[3]) >= 3) {
      CurrentGitVersion = cmCTestGITVersion(v[0], v[1], v[2], v[3]);
    }
  }
  return CurrentGitVersion;
}

/* Diff format:

   :src-mode dst-mode src-sha1 dst-sha1 status\0
   src-path\0
   [dst-path\0]

   The format is repeated for every file changed.  The [dst-path\0]
   line appears only for lines with status 'C' or 'R'.  See 'git help
   diff-tree' for details.
*/
class cmCTestGIT::DiffParser : public cmCTestVC::LineParser
{
public:
  DiffParser(cmCTestGIT* git, const char* prefix)
    : LineParser('\0', false)
    , GIT(git)
    , DiffField(DiffFieldNone)
  {
    SetLog(&git->Log, prefix);
  }

  using Change = cmCTestGIT::Change;
  std::vector<Change> Changes;

protected:
  cmCTestGIT* GIT;
  enum DiffFieldType
  {
    DiffFieldNone,
    DiffFieldChange,
    DiffFieldSrc,
    DiffFieldDst
  };
  DiffFieldType DiffField;
  Change CurChange;

  void DiffReset()
  {
    DiffField = DiffFieldNone;
    Changes.clear();
  }

  bool ProcessLine() override
  {
    if (Line[0] == ':') {
      DiffField = DiffFieldChange;
      CurChange = Change();
    }
    if (DiffField == DiffFieldChange) {
      // :src-mode dst-mode src-sha1 dst-sha1 status
      if (Line[0] != ':') {
        DiffField = DiffFieldNone;
        return true;
      }
      const char* src_mode_first = Line.c_str() + 1;
      const char* src_mode_last = ConsumeField(src_mode_first);
      const char* dst_mode_first = ConsumeSpace(src_mode_last);
      const char* dst_mode_last = ConsumeField(dst_mode_first);
      const char* src_sha1_first = ConsumeSpace(dst_mode_last);
      const char* src_sha1_last = ConsumeField(src_sha1_first);
      const char* dst_sha1_first = ConsumeSpace(src_sha1_last);
      const char* dst_sha1_last = ConsumeField(dst_sha1_first);
      const char* status_first = ConsumeSpace(dst_sha1_last);
      const char* status_last = ConsumeField(status_first);
      if (status_first != status_last) {
        CurChange.Action = *status_first;
        DiffField = DiffFieldSrc;
      } else {
        DiffField = DiffFieldNone;
      }
    } else if (DiffField == DiffFieldSrc) {
      // src-path
      if (CurChange.Action == 'C') {
        // Convert copy to addition of destination.
        CurChange.Action = 'A';
        DiffField = DiffFieldDst;
      } else if (CurChange.Action == 'R') {
        // Convert rename to deletion of source and addition of destination.
        CurChange.Action = 'D';
        CurChange.Path = Line;
        Changes.push_back(CurChange);

        CurChange = Change('A');
        DiffField = DiffFieldDst;
      } else {
        CurChange.Path = Line;
        Changes.push_back(CurChange);
        DiffField = DiffFieldNone;
      }
    } else if (DiffField == DiffFieldDst) {
      // dst-path
      CurChange.Path = Line;
      Changes.push_back(CurChange);
      DiffField = DiffFieldNone;
    }
    return true;
  }

  const char* ConsumeSpace(const char* c)
  {
    while (*c && isspace(*c)) {
      ++c;
    }
    return c;
  }
  const char* ConsumeField(const char* c)
  {
    while (*c && !isspace(*c)) {
      ++c;
    }
    return c;
  }
};

/* Commit format:

   commit ...\n
   tree ...\n
   parent ...\n
   author ...\n
   committer ...\n
   \n
       Log message indented by (4) spaces\n
       (even blank lines have the spaces)\n
 [[
   \n
   [Diff format]
 OR
   \0
 ]]

   The header may have more fields.  See 'git help diff-tree'.
*/
class cmCTestGIT::CommitParser : public cmCTestGIT::DiffParser
{
public:
  CommitParser(cmCTestGIT* git, const char* prefix)
    : DiffParser(git, prefix)
    , Section(SectionHeader)
  {
    Separator = SectionSep[Section];
  }

private:
  using Revision = cmCTestGIT::Revision;
  enum SectionType
  {
    SectionHeader,
    SectionBody,
    SectionDiff,
    SectionCount
  };
  static char const SectionSep[SectionCount];
  SectionType Section;
  Revision Rev;

  struct Person
  {
    std::string Name;
    std::string EMail;
    unsigned long Time = 0;
    long TimeZone = 0;
  };

  void ParsePerson(const char* str, Person& person)
  {
    // Person Name <person@domain.com> 1234567890 +0000
    const char* c = str;
    while (*c && isspace(*c)) {
      ++c;
    }

    const char* name_first = c;
    while (*c && *c != '<') {
      ++c;
    }
    const char* name_last = c;
    while (name_last != name_first && isspace(*(name_last - 1))) {
      --name_last;
    }
    person.Name.assign(name_first, name_last - name_first);

    const char* email_first = *c ? ++c : c;
    while (*c && *c != '>') {
      ++c;
    }
    const char* email_last = *c ? c++ : c;
    person.EMail.assign(email_first, email_last - email_first);

    person.Time = strtoul(c, const_cast<char**>(&c), 10);
    person.TimeZone = strtol(c, const_cast<char**>(&c), 10);
  }

  bool ProcessLine() override
  {
    if (Line.empty()) {
      if (Section == SectionBody && LineEnd == '\0') {
        // Skip SectionDiff
        NextSection();
      }
      NextSection();
    } else {
      switch (Section) {
        case SectionHeader:
          DoHeaderLine();
          break;
        case SectionBody:
          DoBodyLine();
          break;
        case SectionDiff:
          DiffParser::ProcessLine();
          break;
        case SectionCount:
          break; // never happens
      }
    }
    return true;
  }

  void NextSection()
  {
    Section = SectionType((Section + 1) % SectionCount);
    Separator = SectionSep[Section];
    if (Section == SectionHeader) {
      GIT->DoRevision(Rev, Changes);
      Rev = Revision();
      DiffReset();
    }
  }

  void DoHeaderLine()
  {
    // Look for header fields that we need.
    if (cmHasLiteralPrefix(Line, "commit ")) {
      Rev.Rev = Line.substr(7);
    } else if (cmHasLiteralPrefix(Line, "author ")) {
      Person author;
      ParsePerson(Line.c_str() + 7, author);
      Rev.Author = author.Name;
      Rev.EMail = author.EMail;
      Rev.Date = FormatDateTime(author);
    } else if (cmHasLiteralPrefix(Line, "committer ")) {
      Person committer;
      ParsePerson(Line.c_str() + 10, committer);
      Rev.Committer = committer.Name;
      Rev.CommitterEMail = committer.EMail;
      Rev.CommitDate = FormatDateTime(committer);
    }
  }

  void DoBodyLine()
  {
    // Commit log lines are indented by 4 spaces.
    if (Line.size() >= 4) {
      Rev.Log += Line.substr(4);
    }
    Rev.Log += "\n";
  }

  std::string FormatDateTime(Person const& person)
  {
    // Convert the time to a human-readable format that is also easy
    // to machine-parse: "CCYY-MM-DD hh:mm:ss".
    time_t seconds = static_cast<time_t>(person.Time);
    struct tm* t = gmtime(&seconds);
    char dt[1024];
    sprintf(dt, "%04d-%02d-%02d %02d:%02d:%02d", t->tm_year + 1900,
            t->tm_mon + 1, t->tm_mday, t->tm_hour, t->tm_min, t->tm_sec);
    std::string out = dt;

    // Add the time-zone field "+zone" or "-zone".
    char tz[32];
    if (person.TimeZone >= 0) {
      sprintf(tz, " +%04ld", person.TimeZone);
    } else {
      sprintf(tz, " -%04ld", -person.TimeZone);
    }
    out += tz;
    return out;
  }
};

char const cmCTestGIT::CommitParser::SectionSep[SectionCount] = { '\n', '\n',
                                                                  '\0' };

bool cmCTestGIT::LoadRevisions()
{
  // Use 'git rev-list ... | git diff-tree ...' to get revisions.
  std::string range = OldRevision + ".." + NewRevision;
  const char* git = CommandLineTool.c_str();
  const char* git_rev_list[] = { git,           "rev-list", "--reverse",
                                 range.c_str(), "--",       nullptr };
  const char* git_diff_tree[] = {
    git,  "diff-tree",    "--stdin",          "--always", "-z",
    "-r", "--pretty=raw", "--encoding=utf-8", nullptr
  };
  Log << cmCTestGIT::ComputeCommandLine(git_rev_list) << " | "
      << cmCTestGIT::ComputeCommandLine(git_diff_tree) << "\n";

  cmsysProcess* cp = cmsysProcess_New();
  cmsysProcess_AddCommand(cp, git_rev_list);
  cmsysProcess_AddCommand(cp, git_diff_tree);
  cmsysProcess_SetWorkingDirectory(cp, SourceDirectory.c_str());

  CommitParser out(this, "dt-out> ");
  OutputLogger err(Log, "dt-err> ");
  cmCTestGIT::RunProcess(cp, &out, &err, cmProcessOutput::UTF8);

  // Send one extra zero-byte to terminate the last record.
  out.Process("", 1);

  cmsysProcess_Delete(cp);
  return true;
}

bool cmCTestGIT::LoadModifications()
{
  const char* git = CommandLineTool.c_str();

  // Use 'git update-index' to refresh the index w.r.t. the work tree.
  const char* git_update_index[] = { git, "update-index", "--refresh",
                                     nullptr };
  OutputLogger ui_out(Log, "ui-out> ");
  OutputLogger ui_err(Log, "ui-err> ");
  RunChild(git_update_index, &ui_out, &ui_err, nullptr, cmProcessOutput::UTF8);

  // Use 'git diff-index' to get modified files.
  const char* git_diff_index[] = { git,    "diff-index", "-z",
                                   "HEAD", "--",         nullptr };
  DiffParser out(this, "di-out> ");
  OutputLogger err(Log, "di-err> ");
  RunChild(git_diff_index, &out, &err, nullptr, cmProcessOutput::UTF8);

  for (Change const& c : out.Changes) {
    DoModification(PathModified, c.Path);
  }
  return true;
}
