/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestCurl.h"

#include <cstdio>
#include <ostream>

#include <cmext/algorithm>

#include "cmCTest.h"
#include "cmCurl.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"

cmCTestCurl::cmCTestCurl(cmCTest* ctest)
{
  CTest = ctest;
  SetProxyType();
  UseHttp10 = false;
  // In windows, this will init the winsock stuff
  ::curl_global_init(CURL_GLOBAL_ALL);
  // default is to verify https
  VerifyPeerOff = false;
  VerifyHostOff = false;
  Quiet = false;
  TimeOutSeconds = 0;
  Curl = curl_easy_init();
}

cmCTestCurl::~cmCTestCurl()
{
  ::curl_easy_cleanup(Curl);
  ::curl_global_cleanup();
}

std::string cmCTestCurl::Escape(std::string const& source)
{
  char* data1 = curl_easy_escape(Curl, source.c_str(), 0);
  std::string ret = data1;
  curl_free(data1);
  return ret;
}

namespace {
size_t curlWriteMemoryCallback(void* ptr, size_t size, size_t nmemb,
                               void* data)
{
  int realsize = static_cast<int>(size * nmemb);
  const char* chPtr = static_cast<char*>(ptr);
  cm::append(*static_cast<std::vector<char>*>(data), chPtr, chPtr + realsize);
  return realsize;
}

size_t curlDebugCallback(CURL* /*unused*/, curl_infotype /*unused*/,
                         char* chPtr, size_t size, void* data)
{
  cm::append(*static_cast<std::vector<char>*>(data), chPtr, chPtr + size);
  return 0;
}
}

void cmCTestCurl::SetCurlOptions(std::vector<std::string> const& args)
{
  for (std::string const& arg : args) {
    if (arg == "CURLOPT_SSL_VERIFYPEER_OFF") {
      VerifyPeerOff = true;
    }
    if (arg == "CURLOPT_SSL_VERIFYHOST_OFF") {
      VerifyHostOff = true;
    }
  }
}

bool cmCTestCurl::InitCurl()
{
  if (!Curl) {
    return false;
  }
  cmCurlSetCAInfo(Curl);
  if (VerifyPeerOff) {
    curl_easy_setopt(Curl, CURLOPT_SSL_VERIFYPEER, 0);
  }
  if (VerifyHostOff) {
    curl_easy_setopt(Curl, CURLOPT_SSL_VERIFYHOST, 0);
  }
  if (!HTTPProxy.empty()) {
    curl_easy_setopt(Curl, CURLOPT_PROXY, HTTPProxy.c_str());
    curl_easy_setopt(Curl, CURLOPT_PROXYTYPE, HTTPProxyType);
    if (!HTTPProxyAuth.empty()) {
      curl_easy_setopt(Curl, CURLOPT_PROXYUSERPWD, HTTPProxyAuth.c_str());
    }
  }
  if (UseHttp10) {
    curl_easy_setopt(Curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0);
  }
  // enable HTTP ERROR parsing
  curl_easy_setopt(Curl, CURLOPT_FAILONERROR, 1);

  // if there is little to no activity for too long stop submitting
  if (TimeOutSeconds) {
    curl_easy_setopt(Curl, CURLOPT_LOW_SPEED_LIMIT, 1);
    curl_easy_setopt(Curl, CURLOPT_LOW_SPEED_TIME, TimeOutSeconds);
  }

  return true;
}

bool cmCTestCurl::UploadFile(std::string const& local_file,
                             std::string const& url, std::string const& fields,
                             std::string& response)
{
  response.clear();
  if (!InitCurl()) {
    cmCTestLog(CTest, ERROR_MESSAGE, "Initialization of curl failed");
    return false;
  }
  /* enable uploading */
  curl_easy_setopt(Curl, CURLOPT_UPLOAD, 1);

  /* HTTP PUT please */
  ::curl_easy_setopt(Curl, CURLOPT_PUT, 1);
  ::curl_easy_setopt(Curl, CURLOPT_VERBOSE, 1);

  FILE* ftpfile = cmsys::SystemTools::Fopen(local_file, "rb");
  if (!ftpfile) {
    cmCTestLog(CTest, ERROR_MESSAGE,
               "Could not open file for upload: " << local_file << "\n");
    return false;
  }
  // set the url
  std::string upload_url = cmStrCat(url, '?', fields);
  ::curl_easy_setopt(Curl, CURLOPT_URL, upload_url.c_str());
  // now specify which file to upload
  ::curl_easy_setopt(Curl, CURLOPT_INFILE, ftpfile);
  unsigned long filelen = cmSystemTools::FileLength(local_file);
  // and give the size of the upload (optional)
  ::curl_easy_setopt(Curl, CURLOPT_INFILESIZE, static_cast<long>(filelen));
  ::curl_easy_setopt(Curl, CURLOPT_WRITEFUNCTION, curlWriteMemoryCallback);
  ::curl_easy_setopt(Curl, CURLOPT_DEBUGFUNCTION, curlDebugCallback);
  // Set Content-Type to satisfy fussy modsecurity rules.
  struct curl_slist* headers =
    ::curl_slist_append(nullptr, "Content-Type: text/xml");
  // Add any additional headers that the user specified.
  for (std::string const& h : HttpHeaders) {
    cmCTestOptionalLog(
      CTest, DEBUG, "   Add HTTP Header: \"" << h << "\"" << std::endl, Quiet);
    headers = ::curl_slist_append(headers, h.c_str());
  }
  ::curl_easy_setopt(Curl, CURLOPT_HTTPHEADER, headers);
  std::vector<char> responseData;
  std::vector<char> debugData;
  ::curl_easy_setopt(Curl, CURLOPT_FILE, &responseData);
  ::curl_easy_setopt(Curl, CURLOPT_DEBUGDATA, &debugData);
  ::curl_easy_setopt(Curl, CURLOPT_FAILONERROR, 1);
  // Now run off and do what you've been told!
  ::curl_easy_perform(Curl);
  ::fclose(ftpfile);
  ::curl_easy_setopt(Curl, CURLOPT_HTTPHEADER, NULL);
  ::curl_slist_free_all(headers);

  if (!responseData.empty()) {
    response = std::string(responseData.begin(), responseData.end());
    cmCTestOptionalLog(CTest, HANDLER_VERBOSE_OUTPUT,
                       "Curl response: [" << response << "]\n", Quiet);
  }
  std::string curlDebug;
  if (!debugData.empty()) {
    curlDebug = std::string(debugData.begin(), debugData.end());
    cmCTestOptionalLog(CTest, DEBUG, "Curl debug: [" << curlDebug << "]\n",
                       Quiet);
  }
  if (response.empty()) {
    cmCTestLog(CTest, ERROR_MESSAGE,
               "No response from server.\n"
                 << curlDebug);
    return false;
  }
  return true;
}

bool cmCTestCurl::HttpRequest(std::string const& url,
                              std::string const& fields, std::string& response)
{
  response.clear();
  cmCTestOptionalLog(CTest, DEBUG,
                     "HttpRequest\n"
                       << "url: " << url << "\n"
                       << "fields " << fields << "\n",
                     Quiet);
  if (!InitCurl()) {
    cmCTestLog(CTest, ERROR_MESSAGE, "Initialization of curl failed");
    return false;
  }
  curl_easy_setopt(Curl, CURLOPT_POST, 1);
  curl_easy_setopt(Curl, CURLOPT_POSTFIELDS, fields.c_str());
  ::curl_easy_setopt(Curl, CURLOPT_URL, url.c_str());
  ::curl_easy_setopt(Curl, CURLOPT_FOLLOWLOCATION, 1);
  // set response options
  ::curl_easy_setopt(Curl, CURLOPT_WRITEFUNCTION, curlWriteMemoryCallback);
  ::curl_easy_setopt(Curl, CURLOPT_DEBUGFUNCTION, curlDebugCallback);
  std::vector<char> responseData;
  std::vector<char> debugData;
  ::curl_easy_setopt(Curl, CURLOPT_FILE, &responseData);
  ::curl_easy_setopt(Curl, CURLOPT_DEBUGDATA, &debugData);
  ::curl_easy_setopt(Curl, CURLOPT_FAILONERROR, 1);

  // Add headers if any were specified.
  struct curl_slist* headers = nullptr;
  if (!HttpHeaders.empty()) {
    for (std::string const& h : HttpHeaders) {
      cmCTestOptionalLog(CTest, DEBUG,
                         "   Add HTTP Header: \"" << h << "\"" << std::endl,
                         Quiet);
      headers = ::curl_slist_append(headers, h.c_str());
    }
  }

  ::curl_easy_setopt(Curl, CURLOPT_HTTPHEADER, headers);
  CURLcode res = ::curl_easy_perform(Curl);
  ::curl_slist_free_all(headers);

  if (!responseData.empty()) {
    response = std::string(responseData.begin(), responseData.end());
    cmCTestOptionalLog(CTest, DEBUG, "Curl response: [" << response << "]\n",
                       Quiet);
  }
  if (!debugData.empty()) {
    std::string curlDebug = std::string(debugData.begin(), debugData.end());
    cmCTestOptionalLog(CTest, DEBUG, "Curl debug: [" << curlDebug << "]\n",
                       Quiet);
  }
  cmCTestOptionalLog(CTest, DEBUG, "Curl res: " << res << "\n", Quiet);
  return (res == 0);
}

void cmCTestCurl::SetProxyType()
{
  HTTPProxy.clear();
  // this is the default
  HTTPProxyType = CURLPROXY_HTTP;
  HTTPProxyAuth.clear();
  if (cmSystemTools::GetEnv("HTTP_PROXY", HTTPProxy)) {
    std::string port;
    if (cmSystemTools::GetEnv("HTTP_PROXY_PORT", port)) {
      HTTPProxy += ":";
      HTTPProxy += port;
    }
    std::string type;
    if (cmSystemTools::GetEnv("HTTP_PROXY_TYPE", type)) {
      // HTTP/SOCKS4/SOCKS5
      if (type == "HTTP") {
        HTTPProxyType = CURLPROXY_HTTP;
      } else if (type == "SOCKS4") {
        HTTPProxyType = CURLPROXY_SOCKS4;
      } else if (type == "SOCKS5") {
        HTTPProxyType = CURLPROXY_SOCKS5;
      }
    }
    cmSystemTools::GetEnv("HTTP_PROXY_USER", HTTPProxyAuth);
    std::string passwd;
    if (cmSystemTools::GetEnv("HTTP_PROXY_PASSWD", passwd)) {
      HTTPProxyAuth += ":";
      HTTPProxyAuth += passwd;
    }
  }
}
