/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCTestBuildCommand.h"

#include <sstream>

#include <cmext/string_view>

#include "cmCTest.h"
#include "cmCTestBuildHandler.h"
#include "cmGlobalGenerator.h"
#include "cmMakefile.h"
#include "cmMessageType.h"
#include "cmProperty.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmake.h"

class cmExecutionStatus;

void cmCTestBuildCommand::BindArguments()
{
  cmCTestHandlerCommand::BindArguments();
  Bind("NUMBER_ERRORS"_s, NumberErrors);
  Bind("NUMBER_WARNINGS"_s, NumberWarnings);
  Bind("TARGET"_s, Target);
  Bind("CONFIGURATION"_s, Configuration);
  Bind("FLAGS"_s, Flags);
  Bind("PROJECT_NAME"_s, ProjectName);
}

cmCTestBuildCommand::~cmCTestBuildCommand() = default;

cmCTestGenericHandler* cmCTestBuildCommand::InitializeHandler()
{
  cmCTestBuildHandler* handler = CTest->GetBuildHandler();
  handler->Initialize();

  Handler = handler;

  cmProp ctestBuildCommand = Makefile->GetDefinition("CTEST_BUILD_COMMAND");
  if (cmNonempty(ctestBuildCommand)) {
    CTest->SetCTestConfiguration("MakeCommand", *ctestBuildCommand, Quiet);
  } else {
    cmProp cmakeGeneratorName =
      Makefile->GetDefinition("CTEST_CMAKE_GENERATOR");

    // Build configuration is determined by: CONFIGURATION argument,
    // or CTEST_BUILD_CONFIGURATION script variable, or
    // CTEST_CONFIGURATION_TYPE script variable, or ctest -C command
    // line argument... in that order.
    //
    cmProp ctestBuildConfiguration =
      Makefile->GetDefinition("CTEST_BUILD_CONFIGURATION");
    const std::string* cmakeBuildConfiguration = !Configuration.empty()
      ? &Configuration
      : (cmNonempty(ctestBuildConfiguration) ? ctestBuildConfiguration
                                             : &CTest->GetConfigType());

    const std::string* cmakeBuildAdditionalFlags =
      !Flags.empty() ? &Flags : Makefile->GetDefinition("CTEST_BUILD_FLAGS");
    const std::string* cmakeBuildTarget = !Target.empty()
      ? &Target
      : Makefile->GetDefinition("CTEST_BUILD_TARGET");

    if (cmNonempty(cmakeGeneratorName)) {
      if (!cmakeBuildConfiguration) {
        static const std::string sRelease = "Release";
        cmakeBuildConfiguration = &sRelease;
      }
      if (GlobalGenerator) {
        if (GlobalGenerator->GetName() != *cmakeGeneratorName) {
          GlobalGenerator.reset();
        }
      }
      if (!GlobalGenerator) {
        GlobalGenerator = Makefile->GetCMakeInstance()->CreateGlobalGenerator(
          *cmakeGeneratorName);
        if (!GlobalGenerator) {
          std::string e = cmStrCat("could not create generator named \"",
                                   *cmakeGeneratorName, '"');
          Makefile->IssueMessage(MessageType::FATAL_ERROR, e);
          cmSystemTools::SetFatalErrorOccured();
          return nullptr;
        }
      }
      if (cmakeBuildConfiguration->empty()) {
        const std::string* config = nullptr;
#ifdef CMAKE_INTDIR
        static const std::string sIntDir = CMAKE_INTDIR;
        config = &sIntDir;
#endif
        if (!config) {
          static const std::string sDebug = "Debug";
          config = &sDebug;
        }
        cmakeBuildConfiguration = config;
      }

      std::string dir = CTest->GetCTestConfiguration("BuildDirectory");
      std::string buildCommand = GlobalGenerator->GenerateCMakeBuildCommand(
        cmakeBuildTarget ? *cmakeBuildTarget : "", *cmakeBuildConfiguration,
        cmakeBuildAdditionalFlags ? *cmakeBuildAdditionalFlags : "",
        Makefile->IgnoreErrorsCMP0061());
      cmCTestOptionalLog(CTest, HANDLER_VERBOSE_OUTPUT,
                         "SetMakeCommand:" << buildCommand << "\n", Quiet);
      CTest->SetCTestConfiguration("MakeCommand", buildCommand, Quiet);
    } else {
      std::ostringstream ostr;
      /* clang-format off */
      ostr << "has no project to build. If this is a "
        "\"built with CMake\" project, verify that CTEST_CMAKE_GENERATOR "
        "is set. Otherwise, set CTEST_BUILD_COMMAND to build the project "
        "with a custom command line.";
      /* clang-format on */
      SetError(ostr.str());
      return nullptr;
    }
  }

  if (cmProp useLaunchers = Makefile->GetDefinition("CTEST_USE_LAUNCHERS")) {
    CTest->SetCTestConfiguration("UseLaunchers", *useLaunchers, Quiet);
  }

  if (cmProp labelsForSubprojects =
        Makefile->GetDefinition("CTEST_LABELS_FOR_SUBPROJECTS")) {
    CTest->SetCTestConfiguration("LabelsForSubprojects", *labelsForSubprojects,
                                 Quiet);
  }

  handler->SetQuiet(Quiet);
  return handler;
}

bool cmCTestBuildCommand::InitialPass(std::vector<std::string> const& args,
                                      cmExecutionStatus& status)
{
  bool ret = cmCTestHandlerCommand::InitialPass(args, status);
  if (!NumberErrors.empty()) {
    Makefile->AddDefinition(NumberErrors,
                            std::to_string(Handler->GetTotalErrors()));
  }
  if (!NumberWarnings.empty()) {
    Makefile->AddDefinition(NumberWarnings,
                            std::to_string(Handler->GetTotalWarnings()));
  }
  return ret;
}
