/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmCPackIFWRepository.h"

#include <cstddef>

#include "cmCPackIFWGenerator.h"
#include "cmGeneratedFileStream.h"
#include "cmSystemTools.h"
#include "cmXMLParser.h"
#include "cmXMLWriter.h"

cmCPackIFWRepository::cmCPackIFWRepository()
  : Update(cmCPackIFWRepository::None)
{
}

bool cmCPackIFWRepository::IsValid() const
{
  bool valid = true;

  switch (Update) {
    case cmCPackIFWRepository::None:
    case cmCPackIFWRepository::Add:
    case cmCPackIFWRepository::Remove:
      valid = !Url.empty();
      break;
    case cmCPackIFWRepository::Replace:
      valid = !OldUrl.empty() && !NewUrl.empty();
      break;
  }

  return valid;
}

bool cmCPackIFWRepository::ConfigureFromOptions()
{
  // Name;
  if (Name.empty()) {
    return false;
  }

  std::string prefix =
    "CPACK_IFW_REPOSITORY_" + cmsys::SystemTools::UpperCase(Name) + "_";

  // Update
  if (IsOn(prefix + "ADD")) {
    Update = cmCPackIFWRepository::Add;
  } else if (IsOn(prefix + "REMOVE")) {
    Update = cmCPackIFWRepository::Remove;
  } else if (IsOn(prefix + "REPLACE")) {
    Update = cmCPackIFWRepository::Replace;
  } else {
    Update = cmCPackIFWRepository::None;
  }

  // Url
  if (const char* url = GetOption(prefix + "URL")) {
    Url = url;
  } else {
    Url.clear();
  }

  // Old url
  if (const char* oldUrl = GetOption(prefix + "OLD_URL")) {
    OldUrl = oldUrl;
  } else {
    OldUrl.clear();
  }

  // New url
  if (const char* newUrl = GetOption(prefix + "NEW_URL")) {
    NewUrl = newUrl;
  } else {
    NewUrl.clear();
  }

  // Enabled
  if (IsOn(prefix + "DISABLED")) {
    Enabled = "0";
  } else {
    Enabled.clear();
  }

  // Username
  if (const char* username = GetOption(prefix + "USERNAME")) {
    Username = username;
  } else {
    Username.clear();
  }

  // Password
  if (const char* password = GetOption(prefix + "PASSWORD")) {
    Password = password;
  } else {
    Password.clear();
  }

  // DisplayName
  if (const char* displayName = GetOption(prefix + "DISPLAY_NAME")) {
    DisplayName = displayName;
  } else {
    DisplayName.clear();
  }

  return IsValid();
}

/** \class cmCPackeIFWUpdatesPatcher
 * \brief Helper class that parses and patch Updates.xml file (QtIFW)
 */
class cmCPackeIFWUpdatesPatcher : public cmXMLParser
{
public:
  cmCPackeIFWUpdatesPatcher(cmCPackIFWRepository* r, cmXMLWriter& x)
    : repository(r)
    , xout(x)
    , patched(false)
  {
  }

  cmCPackIFWRepository* repository;
  cmXMLWriter& xout;
  bool patched;

protected:
  void StartElement(const std::string& name, const char** atts) override
  {
    xout.StartElement(name);
    StartFragment(atts);
  }

  void StartFragment(const char** atts)
  {
    for (size_t i = 0; atts[i]; i += 2) {
      const char* key = atts[i];
      const char* value = atts[i + 1];
      xout.Attribute(key, value);
    }
  }

  void EndElement(const std::string& name) override
  {
    if (name == "Updates" && !patched) {
      repository->WriteRepositoryUpdates(xout);
      patched = true;
    }
    xout.EndElement();
    if (patched) {
      return;
    }
    if (name == "Checksum") {
      repository->WriteRepositoryUpdates(xout);
      patched = true;
    }
  }

  void CharacterDataHandler(const char* data, int length) override
  {
    std::string content(data, data + length);
    if (content.empty() || content == " " || content == "  " ||
        content == "\n") {
      return;
    }
    xout.Content(content);
  }
};

bool cmCPackIFWRepository::PatchUpdatesXml()
{
  // Lazy directory initialization
  if (Directory.empty() && Generator) {
    Directory = Generator->toplevel;
  }

  // Filenames
  std::string updatesXml = Directory + "/repository/Updates.xml";
  std::string updatesPatchXml = Directory + "/repository/UpdatesPatch.xml";

  // Output stream
  cmGeneratedFileStream fout(updatesPatchXml);
  cmXMLWriter xout(fout);

  xout.StartDocument();

  WriteGeneratedByToStrim(xout);

  // Patch
  {
    cmCPackeIFWUpdatesPatcher patcher(this, xout);
    patcher.ParseFile(updatesXml.data());
  }

  xout.EndDocument();

  fout.Close();

  return cmSystemTools::RenameFile(updatesPatchXml, updatesXml);
}

void cmCPackIFWRepository::WriteRepositoryConfig(cmXMLWriter& xout)
{
  xout.StartElement("Repository");

  // Url
  xout.Element("Url", Url);
  // Enabled
  if (!Enabled.empty()) {
    xout.Element("Enabled", Enabled);
  }
  // Username
  if (!Username.empty()) {
    xout.Element("Username", Username);
  }
  // Password
  if (!Password.empty()) {
    xout.Element("Password", Password);
  }
  // DisplayName
  if (!DisplayName.empty()) {
    xout.Element("DisplayName", DisplayName);
  }

  xout.EndElement();
}

void cmCPackIFWRepository::WriteRepositoryUpdate(cmXMLWriter& xout)
{
  xout.StartElement("Repository");

  switch (Update) {
    case cmCPackIFWRepository::None:
      break;
    case cmCPackIFWRepository::Add:
      xout.Attribute("action", "add");
      break;
    case cmCPackIFWRepository::Remove:
      xout.Attribute("action", "remove");
      break;
    case cmCPackIFWRepository::Replace:
      xout.Attribute("action", "replace");
      break;
  }

  // Url
  if (Update == cmCPackIFWRepository::Add ||
      Update == cmCPackIFWRepository::Remove) {
    xout.Attribute("url", Url);
  } else if (Update == cmCPackIFWRepository::Replace) {
    xout.Attribute("oldUrl", OldUrl);
    xout.Attribute("newUrl", NewUrl);
  }
  // Enabled
  if (!Enabled.empty()) {
    xout.Attribute("enabled", Enabled);
  }
  // Username
  if (!Username.empty()) {
    xout.Attribute("username", Username);
  }
  // Password
  if (!Password.empty()) {
    xout.Attribute("password", Password);
  }
  // DisplayName
  if (!DisplayName.empty()) {
    xout.Attribute("displayname", DisplayName);
  }

  xout.EndElement();
}

void cmCPackIFWRepository::WriteRepositoryUpdates(cmXMLWriter& xout)
{
  if (!RepositoryUpdate.empty()) {
    xout.StartElement("RepositoryUpdate");
    for (cmCPackIFWRepository* r : RepositoryUpdate) {
      r->WriteRepositoryUpdate(xout);
    }
    xout.EndElement();
  }
}
