/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLSurfaceUtils.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// local includes
#include "vtkPCLSurfaceUtils.h"

// pcl includes
#include <pcl/common/io.h>

// vtk includes
#include <vtkCellArray.h>
#include <vtkIdTypeArray.h>
#include <vtkInformation.h>
#include <vtkNew.h>
#include <vtkPolyData.h>
#include <vtkSMPTools.h>

// std includes
#include <vector>

//----------------------------------------------------------------------------
// Implementation of the New function
vtkStandardNewMacro(vtkPCLSurfaceUtils);

//----------------------------------------------------------------------------
vtkSmartPointer<vtkPolyData> vtkPCLSurfaceUtils::CreateVTKPolygons(vtkPolyData* pointcloud,
  std::vector<pcl::Vertices>& polygons)
{
  vtkNew<vtkPolyData> output;
  // Convert outputCloud data in vtk format
  output->SetPoints(pointcloud->GetPoints());

  vtkNew<vtkIdTypeArray> offsets;
  offsets->SetNumberOfValues(polygons.size() + 1);
  vtkIdType offset = 0;
  for (vtkIdType idx = 0; idx < polygons.size(); idx++)
  {
    offsets->SetValue(idx, offset);
    offset += polygons[idx].vertices.size();
  }
  offsets->SetValue(polygons.size(), offset);

  vtkNew<vtkIdTypeArray> connectivity;
  connectivity->SetNumberOfValues(offset);
  vtkSMPTools::For(0,
    polygons.size(),
    [&](vtkIdType begin, vtkIdType end)
    {
      for (vtkIdType polygonIdx = begin; polygonIdx < end; ++polygonIdx)
      {
        const pcl::Vertices& poly = polygons[polygonIdx];
        const vtkIdType& offset = offsets->GetValue(polygonIdx);
        std::copy(poly.vertices.begin(), poly.vertices.end(), connectivity->GetPointer(offset));
      }
    });

  // Add polygons to output
  vtkNew<vtkCellArray> polys;
  polys->SetData(offsets, connectivity);
  output->SetPolys(polys);

  return output;
}
