/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLPoisson.h

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLPoisson_h
#define vtkPCLPoisson_h

// vtk includes
#include <vtkPolyDataAlgorithm.h>

// pcl includes
#include <pcl/point_cloud.h>
#include <pcl/point_types.h>

#include "vtkPCLSurfaceModule.h" // for export macro

/**
 * @brief The vtkPCLPoisson class implements a poisson reconstruction algorithm. This filter
 * wraps the Poisson class from the PCL library.
 */
class VTKPCLSURFACE_EXPORT vtkPCLPoisson : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLPoisson* New();
  vtkTypeMacro(vtkPCLPoisson, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  vtkSetMacro(Depth, int);
  vtkGetMacro(Depth, int);

  vtkSetMacro(SolverDivide, int);
  vtkGetMacro(SolverDivide, int);

  vtkSetMacro(IsoDivide, int);
  vtkGetMacro(IsoDivide, int);

  vtkSetMacro(PointWeight, float);
  vtkGetMacro(PointWeight, float);

  vtkSetMacro(Scale, float);
  vtkGetMacro(Scale, float);

  vtkSetMacro(SamplesPerNode, float);
  vtkGetMacro(SamplesPerNode, float);

  vtkSetMacro(Confidence, bool);
  vtkGetMacro(Confidence, bool);

  vtkSetMacro(Manifold, bool);
  vtkGetMacro(Manifold, bool);

  vtkSetMacro(Degree, int);
  vtkGetMacro(Degree, int);

  vtkSetMacro(Radius, float);
  vtkGetMacro(Radius, float);

protected:
  vtkPCLPoisson() = default;
  ~vtkPCLPoisson() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  // Maximum depth of the tree that will be used for surface reconstruction.
  int Depth = 8;
  // Depth at which a block Gauss-Seidel solver is used to solve the Laplacian equation.
  int SolverDivide = 8;
  // Depth at which iso-surface extraction is done (this should be less than or equal to the depth)
  int IsoDivide = 8;
  // Point weight factor for Poisson equation.
  float PointWeight = 4.f;
  // Ratio between the diameter of the cube used for reconstruction and the diameter of the samples'
  // bounding cube.
  float Scale = 1.0f;
  // Minimum number of sample points that should fall within an octree node as the octree
  // construction is adapted to sampling density.
  float SamplesPerNode = 1.f;
  // Confidence flag. Enabling this flag tells the reconstructor to use the size of the normals as
  // confidence information. When the flag is not enabled, all normals are normalized to have
  // unit-length prior to reconstruction.
  bool Confidence = false;
  // Specifies if the output mesh should be a manifold (true) or not (false).
  bool Manifold = false;
  // Degree of the polynomial that will be fit to the local data to reconstruct the local surface.
  int Degree = 2;

  // Radius search parameter for the search of the nearest neighbors used for normal estimation.
  float Radius = 1.0f;

private:
  vtkPCLPoisson(const vtkPCLPoisson&);
  void operator=(const vtkPCLPoisson&);
};

#endif // vtkPCLPoisson_h
