/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLPoisson.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// local includes
#include "vtkPCLPoisson.h"
#include "vtkPCLSurfaceUtils.h"
#include "vtkPCLUtils.h"

// pcl includes
#include <pcl/common/io.h>
#include <pcl/point_types.h>
#include <pcl/surface/poisson.h>

// vtk includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>
#include <vtkSMPTools.h>

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLPoisson);

//-----------------------------------------------------------------------------
void vtkPCLPoisson::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLPoisson::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Convert the input data to PCL format
  pcl::PointCloud<pcl::PointXYZ>::Ptr pointCloud(new pcl::PointCloud<pcl::PointXYZ>);
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);
  // Compute normals
  pcl::PointCloud<pcl::PointNormal>::Ptr pointCloudNormal =
    vtkPCLUtils::EstimateAndMergePCLNormals(pointCloud, this->Radius);

  pcl::PointCloud<pcl::PointNormal>::Ptr outputCloud(new pcl::PointCloud<pcl::PointNormal>);

  // Apply reconstruction
  pcl::Poisson<pcl::PointNormal> filter;
  filter.setDepth(this->Depth);
  filter.setSolverDivide(this->SolverDivide);
  filter.setIsoDivide(this->IsoDivide);
  filter.setPointWeight(this->PointWeight);
  filter.setSamplesPerNode(this->SamplesPerNode);
  filter.setScale(this->Scale);
  filter.setConfidence(this->Confidence);
  filter.setManifold(this->Manifold);
  filter.setDegree(this->Degree);
  filter.setThreads(vtkSMPTools::GetEstimatedNumberOfThreads());

  filter.setInputCloud(pointCloudNormal);
  std::vector<pcl::Vertices> polygons;
  filter.performReconstruction(*outputCloud, polygons);

  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  // Convert the output to a vtkPolyData mesh
  output->ShallowCopy(
    vtkPCLSurfaceUtils::CreateVTKPolygons<pcl::PointNormal>(outputCloud, polygons));

  return 1;
}
