/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLMarchingCubes.h

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLMarchingCubes_h
#define vtkPCLMarchingCubes_h

// vtk includes
#include <vtkPolyDataAlgorithm.h>

// pcl includes
#include <pcl/point_cloud.h>
#include <pcl/point_types.h>

#include "vtkPCLSurfaceModule.h" // for export macro

/**
 * @brief The vtkPCLMarchingCubes class implements a HOPPE and RBF marching cubes algorithm
 * from the PCL library.
 */
class VTKPCLSURFACE_EXPORT vtkPCLMarchingCubes : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLMarchingCubes* New();
  vtkTypeMacro(vtkPCLMarchingCubes, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  enum Algorithm
  {
    HOPPE = 0,
    RBF
  };

  vtkSetMacro(AlgorithmType, Algorithm);
  void SetAlgorithmType(int algorithm)
  {
    this->AlgorithmType = static_cast<Algorithm>(algorithm);
    this->Modified();
  }
  vtkGetMacro(AlgorithmType, Algorithm);

  vtkSetMacro(IsoLevel, float);
  vtkGetMacro(IsoLevel, float);

  vtkSetVector3Macro(GridResolution, int);
  vtkGetVector3Macro(GridResolution, int);

  vtkSetMacro(PercentExtendGrid, float);
  vtkGetMacro(PercentExtendGrid, float);

  vtkSetMacro(DistanceIgnore, float);
  vtkGetMacro(DistanceIgnore, float);

  vtkSetMacro(OffSurfaceDisplacement, float);
  vtkGetMacro(OffSurfaceDisplacement, float);

  vtkSetMacro(Radius, float);
  vtkGetMacro(Radius, float);

protected:
  vtkPCLMarchingCubes() = default;
  ~vtkPCLMarchingCubes() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  //! Algorithm used for the marching cubes reconstruction
  Algorithm AlgorithmType = HOPPE;

  //! Iso level of the surface to be extracted
  float IsoLevel = 0.0f;

  //! Marching cubes grid resolution
  int GridResolution[3] = { 100, 100, 100 };

  //! Defines how much free space should be left inside the grid between the bounding box of the
  //! point cloud and the grid limits
  float PercentExtendGrid = 0.0f;

  //! Distance for ignoring voxels which are far from point cloud
  float DistanceIgnore = 5.0f;

  //! Off surface displacement
  float OffSurfaceDisplacement = 0.0f;

  //! Radius search parameter for the search of the nearest neighbors used for normal estimation.
  float Radius = 1.0f;

private:
  vtkPCLMarchingCubes(const vtkPCLMarchingCubes&);
  void operator=(const vtkPCLMarchingCubes&);
};

#endif // vtkPCLMarchingCubes_h
