/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLMarchingCubes.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// local includes
#include "vtkPCLMarchingCubes.h"
#include "vtkPCLSurfaceUtils.h"
#include "vtkPCLUtils.h"

// pcl includes
#include <pcl/common/io.h>
#include <pcl/point_types.h>
#include <pcl/surface/marching_cubes_hoppe.h>
#include <pcl/surface/marching_cubes_rbf.h>

// vtk includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLMarchingCubes);

//-----------------------------------------------------------------------------
void vtkPCLMarchingCubes::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLMarchingCubes::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Convert the input data to PCL format
  pcl::PointCloud<pcl::PointXYZ>::Ptr pointCloud(new pcl::PointCloud<pcl::PointXYZ>);
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);
  // Compute normals
  pcl::PointCloud<pcl::PointNormal>::Ptr pointCloudNormal =
    vtkPCLUtils::EstimateAndMergePCLNormals(pointCloud, this->Radius);

  pcl::PointCloud<pcl::PointNormal>::Ptr outputCloud(new pcl::PointCloud<pcl::PointNormal>);

  // Apply reconstruction
  pcl::MarchingCubes<pcl::PointNormal>::Ptr surfaceReconstruction;

  switch (this->AlgorithmType)
  {
    case vtkPCLMarchingCubes::HOPPE:
    {
      pcl::MarchingCubesHoppe<pcl::PointNormal>::Ptr hoppeReconstruction(
        new pcl::MarchingCubesHoppe<pcl::PointNormal>);
      hoppeReconstruction->setDistanceIgnore(this->DistanceIgnore);
      surfaceReconstruction = hoppeReconstruction;
      break;
    }

    case vtkPCLMarchingCubes::RBF:
    {
      pcl::MarchingCubesRBF<pcl::PointNormal>::Ptr rbfReconstruction(
        new pcl::MarchingCubesRBF<pcl::PointNormal>);
      rbfReconstruction->setOffSurfaceDisplacement(this->OffSurfaceDisplacement);
      surfaceReconstruction = rbfReconstruction;
      break;
    }
    default:
      vtkErrorMacro("Invalid algorithm type");
      return 0;
  }
  surfaceReconstruction->setGridResolution(
    this->GridResolution[0], this->GridResolution[1], this->GridResolution[2]);
  surfaceReconstruction->setIsoLevel(this->IsoLevel);
  surfaceReconstruction->setPercentageExtendGrid(this->PercentExtendGrid);
  surfaceReconstruction->setInputCloud(pointCloudNormal);
  std::vector<pcl::Vertices> polygons;
  surfaceReconstruction->reconstruct(*outputCloud, polygons);

  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  // Convert the output to a vtkPolyData mesh
  output->ShallowCopy(
    vtkPCLSurfaceUtils::CreateVTKPolygons<pcl::PointNormal>(outputCloud, polygons));

  return 1;
}
