/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLGreedyProjectionTriangulation.h

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLGreedyProjectionTriangulation_h
#define vtkPCLGreedyProjectionTriangulation_h

// vtk includes
#include <vtkPolyDataAlgorithm.h>

// pcl includes
#include <pcl/point_cloud.h>
#include <pcl/point_types.h>

#include "vtkPCLSurfaceModule.h" // for export macro

/**
 * @brief The vtkPCLGreedyProjectionTriangulation class implements a Greedy Projection Triangulation
 * algorithm. The greedy surface triangulation algorithm builds a triangle mesh by projecting local
 * neighborhoods of points onto their normals and connecting unconnected points.
 * It grows the mesh iteratively from "fringe" points, handling unorganized points, multiple scans,
 * and regions with varying densities. Parameters like neighborhood size, edge length, and angular
 * constraints ensure the method adapts to surface smoothness and transitions.
 */
class VTKPCLSURFACE_EXPORT vtkPCLGreedyProjectionTriangulation : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLGreedyProjectionTriangulation* New();
  vtkTypeMacro(vtkPCLGreedyProjectionTriangulation, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  ///@{
  /**
   * Radius used for normal estimation. Defines the neighborhood size
   * for computing surface normals.
   */
  vtkSetMacro(NormalRadius, double);
  vtkGetMacro(NormalRadius, double);
  ///@}

  ///@{
  /**
   * Maximum search radius for finding neighbors. This is effectively
   * the maximum edge length of each triangle in the triangulation.
   */
  vtkSetMacro(SearchRadius, double);
  vtkGetMacro(SearchRadius, double);
  ///@}

  ///@{
  /**
   * Maximum number of nearest neighbors to consider during neighborhood
   * search for triangulation. Typical values are 50-100.
   */
  vtkSetMacro(MaximumNearestNeighbors, unsigned);
  vtkGetMacro(MaximumNearestNeighbors, unsigned);
  ///@}

  ///@{
  /**
   * A scaling factor that specifies the maximum acceptable distance
   * for a neighbor point, relative to the nearest neighbor's distance.
   * Typical values are 2.5-3 (or 1.5 for grids).
   */
  vtkSetMacro(Mu, double);
  vtkGetMacro(Mu, double);
  ///@}

  ///@{
  /**
   * Maximum surface angle for connecting points, used to handle sharp
   * edges and corners. Points whose normals deviate more than this angle
   * are not connected. Typical value is 45 degrees (in radians).
   */
  vtkSetMacro(MaximumSurfaceAngle, double);
  vtkGetMacro(MaximumSurfaceAngle, double);
  ///@}

  ///@{
  /**
   * Minimum angle for each triangle in the triangulation. While this
   * value is not guaranteed, it is used as a guideline. Typical value
   * is 10 degrees (in radians).
   */
  vtkSetMacro(MinimumAngle, double);
  vtkGetMacro(MinimumAngle, double);
  ///@}

  ///@{
  /**
   * Maximum angle for each triangle in the triangulation. This value
   * is guaranteed during triangulation. Typical value is 120 degrees
   * (in radians).
   */
  vtkSetMacro(MaximumAngle, double);
  vtkGetMacro(MaximumAngle, double);
  ///@}

protected:
  vtkPCLGreedyProjectionTriangulation() = default;
  ~vtkPCLGreedyProjectionTriangulation() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

private:
  vtkPCLGreedyProjectionTriangulation(const vtkPCLGreedyProjectionTriangulation&);
  void operator=(const vtkPCLGreedyProjectionTriangulation&);

  double NormalRadius = 0.025;
  double SearchRadius = 0.025;
  unsigned MaximumNearestNeighbors = 100;
  double Mu = 2.5;
  double MaximumSurfaceAngle = M_PI / 4;
  double MinimumAngle = M_PI / 18;
  double MaximumAngle = 2 * M_PI / 3;
};

#endif // vtkPCLGreedyProjectionTriangulation_h
