/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLGreedyProjectionTriangulation.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// local includes
#include "vtkPCLGreedyProjectionTriangulation.h"
#include "vtkPCLSurfaceUtils.h"
#include "vtkPCLUtils.h"

// pcl includes
#include <pcl/common/io.h>
#include <pcl/point_types.h>
#include <pcl/surface/gp3.h>

// vtk includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLGreedyProjectionTriangulation);

//-----------------------------------------------------------------------------
void vtkPCLGreedyProjectionTriangulation::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLGreedyProjectionTriangulation::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input and the output
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  // Convert the input data to PCL format
  pcl::PointCloud<pcl::PointXYZ>::Ptr cloud(new pcl::PointCloud<pcl::PointXYZ>);
  vtkPCLUtils::PolyDataToPointCloud(input, *cloud);
  // Compute normals
  pcl::PointCloud<pcl::PointNormal>::Ptr cloudWithNormals = vtkPCLUtils::EstimateAndMergePCLNormals(
    cloud, this->NormalRadius);

  pcl::PointCloud<pcl::PointNormal>::Ptr outputCloud(new pcl::PointCloud<pcl::PointNormal>);

  // Create search tree
  pcl::search::KdTree<pcl::PointNormal>::Ptr tree(new pcl::search::KdTree<pcl::PointNormal>);
  tree->setInputCloud(cloudWithNormals);

  // Initialize objects
  pcl::GreedyProjectionTriangulation<pcl::PointNormal> greedyProjection;
  pcl::PolygonMesh triangles;

  // Set the maximum distance between connected points (maximum edge length)
  greedyProjection.setSearchRadius(this->SearchRadius);

  // Set typical values for the parameters
  greedyProjection.setMu(this->Mu);
  greedyProjection.setMaximumNearestNeighbors(this->MaximumNearestNeighbors);
  greedyProjection.setMaximumSurfaceAngle(this->MaximumSurfaceAngle);
  greedyProjection.setMinimumAngle(this->MinimumAngle);
  greedyProjection.setMaximumAngle(this->MaximumAngle);
  greedyProjection.setNormalConsistency(false);

  // Get result
  greedyProjection.setInputCloud(cloudWithNormals);
  greedyProjection.setSearchMethod(tree);
  greedyProjection.reconstruct(triangles);

  // Convert the output to a vtkPolyData mesh
  output->DeepCopy(vtkPCLSurfaceUtils::CreateVTKPolygons(input, triangles.polygons));

  return 1;
}
