/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLRegionGrowingRGB.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLRegionGrowingRGB_h
#define vtkPCLRegionGrowingRGB_h

// VTK includes
#include <vtkPolyData.h>
#include <vtkPolyDataAlgorithm.h>
#include <vtkSmartPointer.h>

// PCL includes
#include <pcl/PointIndices.h>

#include "vtkPCLSegmentationModule.h" // For export macro

/**
 * @class vtkPCLRegionGrowingRGB
 * @brief Extracts clusters from a point cloud using the Region Growing algorithm with RGB. Tis
 * algorithm can be very slow. This class wraps PCLRegionGrowing from the PCL library.
 */
class VTKPCLSEGMENTATION_EXPORT vtkPCLRegionGrowingRGB : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLRegionGrowingRGB* New();
  vtkTypeMacro(vtkPCLRegionGrowingRGB, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  vtkSetMacro(MinClusterSize, int);
  vtkGetMacro(MinClusterSize, int);

  vtkSetMacro(MaxClusterSize, int);
  vtkGetMacro(MaxClusterSize, int);

  vtkSetMacro(PointColorThreshold, double);
  vtkGetMacro(PointColorThreshold, double);

  vtkSetMacro(Radius, double);
  vtkGetMacro(Radius, double);

  vtkSetMacro(SmoothnessFlag, bool);
  vtkGetMacro(SmoothnessFlag, bool);

  vtkSetMacro(SmoothnessThreshold, double);
  vtkGetMacro(SmoothnessThreshold, double);

  vtkSetMacro(CurvatureFlag, bool);
  vtkGetMacro(CurvatureFlag, bool);

  vtkSetMacro(CurvatureThreshold, double);
  vtkGetMacro(CurvatureThreshold, double);

  vtkSetMacro(ResidualFlag, bool);
  vtkGetMacro(ResidualFlag, bool);

  vtkSetMacro(ResidualThreshold, double);
  vtkGetMacro(ResidualThreshold, double);

protected:
  vtkPCLRegionGrowingRGB() = default;
  ~vtkPCLRegionGrowingRGB() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  //! Minimum number of points that a cluster needs to contain in order to be considered valid
  int MinClusterSize = 50;

  //! Maximum number of points that a cluster needs to contain in order to be considered valid
  int MaxClusterSize = 1000000;

  //! Radius of the spherical neighborhood used for clustering
  double Radius = 0.5;

  //! Color threshold for the region growing algorithm
  double PointColorThreshold = 6.0;

  //! Flag indicating whether the smoothness constraint is used
  bool SmoothnessFlag = false;
  double SmoothnessThreshold = 7.0 / 180.0 * M_PI;

  //! Flag indicating whether the curvature constraint is used
  bool CurvatureFlag = false;
  double CurvatureThreshold = 1.0;

  //! Flag indicating whether the residual constraint is used
  bool ResidualFlag = false;
  double ResidualThreshold = 0.05;

private:
  vtkPCLRegionGrowingRGB(const vtkPCLRegionGrowingRGB&);
  void operator=(const vtkPCLRegionGrowingRGB&);
};

#endif // vtkPCLRegionGrowingRGB_h
