/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLRegionGrowingRGB.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Local includes
#include "vtkPCLRegionGrowingRGB.h"
#include "vtkPCLSegmentationUtils.h"

// VTK includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

// PCL includes
#include <pcl/segmentation/region_growing_rgb.h>

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLRegionGrowingRGB);

//-----------------------------------------------------------------------------
void vtkPCLRegionGrowingRGB::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLRegionGrowingRGB::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Get the output
  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));

  // Convert input data in pcl format
  pcl::PointCloud<pcl::PointXYZRGB>::Ptr pointCloud(new pcl::PointCloud<pcl::PointXYZRGB>);
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);

  // Set color data if available

  vtkAbstractArray* colorArray = input->GetPointData()->GetAbstractArray("Color");
  vtkDataArray* colors = vtkDataArray::SafeDownCast(colorArray);
  if (!colors || colors->GetNumberOfComponents() != 3)
  {
    vtkErrorMacro("No color data available");
    return 0;
  }

  pointCloud->points.resize(input->GetNumberOfPoints());
  for (int i = 0; i < input->GetNumberOfPoints(); i++)
  {
    pointCloud->points[i].r = colors->GetComponent(i, 0);
    pointCloud->points[i].g = colors->GetComponent(i, 1);
    pointCloud->points[i].b = colors->GetComponent(i, 2);
  }

  // A kd-tree is used to search for the nearest neighbors
  pcl::search::KdTree<pcl::PointXYZRGB>::Ptr tree(new pcl::search::KdTree<pcl::PointXYZRGB>());

  // Compute the normals
  pcl::PointCloud<pcl::Normal>::Ptr cloudNormals(new pcl::PointCloud<pcl::Normal>);
  vtkPCLSegmentationUtils::RetreiveVTKNormalsOrCompute<pcl::PointXYZRGB>(
    input, this->Radius, pointCloud, cloudNormals);

  pcl::RegionGrowingRGB<pcl::PointXYZRGB, pcl::Normal> segmentation;
  segmentation.setSearchMethod(tree);
  segmentation.setMinClusterSize(this->MinClusterSize);
  segmentation.setMaxClusterSize(this->MaxClusterSize);
  segmentation.setPointColorThreshold(this->PointColorThreshold);
  segmentation.setSmoothModeFlag(this->SmoothnessFlag);
  segmentation.setSmoothnessThreshold(this->SmoothnessThreshold);
  segmentation.setCurvatureTestFlag(this->CurvatureFlag);
  segmentation.setCurvatureThreshold(this->CurvatureThreshold);
  segmentation.setResidualTestFlag(this->ResidualFlag);
  segmentation.setResidualThreshold(this->ResidualThreshold);
  segmentation.setInputCloud(pointCloud);
  segmentation.setInputNormals(cloudNormals);
  std::vector<pcl::PointIndices> clusterIndices;
  segmentation.extract(clusterIndices);

  output->ShallowCopy(input);
  vtkPCLSegmentationUtils::SetIndices(output, clusterIndices);

  return 1;
}
