/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLRegionGrowing.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLRegionGrowing_h
#define vtkPCLRegionGrowing_h

// VTK includes
#include <vtkPolyData.h>
#include <vtkPolyDataAlgorithm.h>
#include <vtkSmartPointer.h>

// PCL includes
#include <pcl/PointIndices.h>

#include "vtkPCLSegmentationModule.h" // For export macro

/**
 * @class vtkPCLRegionGrowing
 * @brief Extracts clusters from a point cloud using the Region Growing algorithm with normals.
 * This class wraps PCLRegionGrowing from the PCL library.
 */
class VTKPCLSEGMENTATION_EXPORT vtkPCLRegionGrowing : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLRegionGrowing* New();
  vtkTypeMacro(vtkPCLRegionGrowing, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  vtkSetMacro(MinClusterSize, int);
  vtkGetMacro(MinClusterSize, int);

  vtkSetMacro(MaxClusterSize, int);
  vtkGetMacro(MaxClusterSize, int);

  vtkSetMacro(Radius, double);
  vtkGetMacro(Radius, double);

  vtkSetMacro(SmoothnessFlag, bool);
  vtkGetMacro(SmoothnessFlag, bool);

  vtkSetMacro(SmoothnessThreshold, double);
  vtkGetMacro(SmoothnessThreshold, double);

  vtkSetMacro(CurvatureFlag, bool);
  vtkGetMacro(CurvatureFlag, bool);

  vtkSetMacro(CurvatureThreshold, double);
  vtkGetMacro(CurvatureThreshold, double);

  vtkSetMacro(ResidualFlag, bool);
  vtkGetMacro(ResidualFlag, bool);

  vtkSetMacro(ResidualThreshold, double);
  vtkGetMacro(ResidualThreshold, double);

protected:
  vtkPCLRegionGrowing() = default;
  ~vtkPCLRegionGrowing() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  //! Minimum number of points that a cluster needs to contain in order to be considered valid
  int MinClusterSize = 50;

  //! Maximum number of points that a cluster needs to contain in order to be considered valid
  int MaxClusterSize = 1000000;

  //! Radius of the spherical neighborhood used for clustering
  double Radius = 0.5;

  //! Flag indicating whether the smoothness constraint is used
  bool SmoothnessFlag = false;
  double SmoothnessThreshold = 7.0 / 180.0 * M_PI;

  //! Flag indicating whether the curvature constraint is used
  bool CurvatureFlag = false;
  double CurvatureThreshold = 1.0;

  //! Flag indicating whether the residual constraint is used
  bool ResidualFlag = false;
  double ResidualThreshold = 0.05;

private:
  vtkPCLRegionGrowing(const vtkPCLRegionGrowing&);
  void operator=(const vtkPCLRegionGrowing&);
};

#endif // vtkPCLRegionGrowing_h
