/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLRegionGrowing.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Local includes
#include "vtkPCLRegionGrowing.h"
#include "vtkPCLSegmentationUtils.h"
#include "vtkPCLUtils.h"

// VTK includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

// PCL includes
#include <pcl/segmentation/region_growing.h>

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLRegionGrowing);

//-----------------------------------------------------------------------------
void vtkPCLRegionGrowing::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLRegionGrowing::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Get the output
  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));

  // Convert input data in pcl format
  pcl::PointCloud<pcl::PointXYZ>::Ptr pointCloud(new pcl::PointCloud<pcl::PointXYZ>);
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);

  // A kd-tree is used to search for the nearest neighbors
  pcl::search::KdTree<pcl::PointXYZ>::Ptr tree(new pcl::search::KdTree<pcl::PointXYZ>());

  // Compute the normals
  pcl::PointCloud<pcl::Normal>::Ptr cloudNormals(new pcl::PointCloud<pcl::Normal>);
  vtkPCLSegmentationUtils::RetreiveVTKNormalsOrCompute<pcl::PointXYZ>(
    input, this->Radius, pointCloud, cloudNormals);

  pcl::RegionGrowing<pcl::PointXYZ, pcl::Normal> segmentation;
  segmentation.setSearchMethod(tree);
  segmentation.setMinClusterSize(this->MinClusterSize);
  segmentation.setMaxClusterSize(this->MaxClusterSize);
  segmentation.setSmoothModeFlag(this->SmoothnessFlag);
  segmentation.setSmoothnessThreshold(this->SmoothnessThreshold);
  segmentation.setCurvatureTestFlag(this->CurvatureFlag);
  segmentation.setCurvatureThreshold(this->CurvatureThreshold);
  segmentation.setResidualTestFlag(this->ResidualFlag);
  segmentation.setResidualThreshold(this->ResidualThreshold);
  segmentation.setInputCloud(pointCloud);
  segmentation.setInputNormals(cloudNormals);
  std::vector<pcl::PointIndices> clusterIndices;
  segmentation.extract(clusterIndices);

  output->ShallowCopy(input);
  vtkPCLSegmentationUtils::SetIndices(output, clusterIndices);

  return 1;
}
