/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLEuclideanClusterExtraction.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLEuclideanClusterExtraction_h
#define vtkPCLEuclideanClusterExtraction_h

// VTK includes
#include <vtkPolyData.h>
#include <vtkPolyDataAlgorithm.h>

// PCL includes
#include <pcl/PointIndices.h>

#include "vtkPCLSegmentationModule.h" // For export macro

/**
 * @class vtkPCLEuclideanClusterExtraction
 * @brief Extracts clusters from a point cloud using the Euclidean Cluster Extraction algorithm.
 * This class wraps PCLEuclideanClusterExtraction from the PCL library.
 */
class VTKPCLSEGMENTATION_EXPORT vtkPCLEuclideanClusterExtraction : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLEuclideanClusterExtraction* New();
  vtkTypeMacro(vtkPCLEuclideanClusterExtraction, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  vtkSetMacro(ClusterTolerance, double);
  vtkGetMacro(ClusterTolerance, double);

  vtkSetMacro(MinClusterSize, int);
  vtkGetMacro(MinClusterSize, int);

  vtkSetMacro(MaxClusterSize, int);
  vtkGetMacro(MaxClusterSize, int);

protected:
  vtkPCLEuclideanClusterExtraction() = default;
  ~vtkPCLEuclideanClusterExtraction() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  //! Maximum distance between two points to be considered as in the same cluster
  double ClusterTolerance = 0.2;

  //! Minimum number of points that a cluster needs to contain in order to be considered valid
  int MinClusterSize = 50;

  //! Maximum number of points that a cluster needs to contain in order to be considered valid
  int MaxClusterSize = 1000000;

private:
  vtkPCLEuclideanClusterExtraction(const vtkPCLEuclideanClusterExtraction&);
  void operator=(const vtkPCLEuclideanClusterExtraction&);
};

#endif // vtkPCLEuclideanClusterExtraction_h
