/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLEuclideanClusterExtraction.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Local includes
#include "vtkPCLEuclideanClusterExtraction.h"
#include "vtkPCLSegmentationUtils.h"
#include "vtkPCLUtils.h"

// VTK includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>
#include <vtkSmartPointer.h>

// PCL includes
#include <pcl/segmentation/extract_clusters.h>

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLEuclideanClusterExtraction);

//-----------------------------------------------------------------------------
void vtkPCLEuclideanClusterExtraction::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLEuclideanClusterExtraction::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Get the output
  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));

  // Convert input data in pcl format
  pcl::PointCloud<pcl::PointXYZ>::Ptr pointCloud(new pcl::PointCloud<pcl::PointXYZ>);
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);

  // A kd-tree is used to search for the nearest neighbors
  pcl::search::KdTree<pcl::PointXYZ>::Ptr tree(new pcl::search::KdTree<pcl::PointXYZ>());

  pcl::EuclideanClusterExtraction<pcl::PointXYZ> segmentation;
  segmentation.setSearchMethod(tree);
  segmentation.setClusterTolerance(this->ClusterTolerance);
  segmentation.setMinClusterSize(this->MinClusterSize);
  segmentation.setMaxClusterSize(this->MaxClusterSize);
  segmentation.setInputCloud(pointCloud);
  std::vector<pcl::PointIndices> clusterIndices;
  segmentation.extract(clusterIndices);

  output->ShallowCopy(input);
  vtkPCLSegmentationUtils::SetIndices(output, clusterIndices);

  return 1;
}
