/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLWallFitting.h
  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLWallFitting_h
#define vtkPCLWallFitting_h

// VTK includes
#include <vtkPolyData.h>
#include <vtkPolyDataAlgorithm.h>
#include <vtkSmartPointer.h>
#include <vtkUnsignedIntArray.h>

// Local includes
#include "vtkPCLFittingModel.h"
#include "vtkPCLSampleConsensusModule.h" // For export macro

/**
 * @brief The vtkPCLWallFitting class fits a set of planes to a point cloud using
 * orthogonality / parallelism of floor and walls.  The normal axis of the floor and ceilling plane
 * must be specified with the NormalAxis property. The ThresholdAngle property specifies the
 * threshold angle used for the perpendicular plane (in degrees).
 */
class VTKPCLSAMPLECONSENSUS_EXPORT vtkPCLWallFitting : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLWallFitting* New();
  vtkTypeMacro(vtkPCLWallFitting, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  enum PlaneType
  {
    Floor = 0, // Floor and Ceilling
    Wall
  };

  vtkGetMacro(SampleConsensusType, vtkPCLFittingModel::SampleConsensus);
  vtkSetMacro(SampleConsensusType, vtkPCLFittingModel::SampleConsensus);
  void SetSampleConsensus(int type)
  {
    this->SetSampleConsensusType(static_cast<vtkPCLFittingModel::SampleConsensus>(type));
  }

  vtkGetMacro(DistanceThreshold, double);
  vtkSetMacro(DistanceThreshold, double);

  vtkGetMacro(Probability, double);
  vtkSetMacro(Probability, double);

  vtkGetMacro(MaxIterations, int);
  vtkSetMacro(MaxIterations, int);

  vtkGetMacro(ThresholdAngle, double);
  vtkSetMacro(ThresholdAngle, double);

  vtkGetMacro(ConstraintsBetweenWalls, bool);
  vtkSetMacro(ConstraintsBetweenWalls, bool);

  vtkGetVector3Macro(NormalAxis, double);
  vtkSetVector3Macro(NormalAxis, double);

  vtkGetMacro(NumberOfFloorCeillingPlanes, int);
  vtkSetMacro(NumberOfFloorCeillingPlanes, int);

  vtkGetMacro(NumberOfWallPlanes, int);
  vtkSetMacro(NumberOfWallPlanes, int);

  vtkGetMacro(ProjectInliers, bool);
  vtkSetMacro(ProjectInliers, bool);

protected:
  vtkPCLWallFitting() = default;
  ~vtkPCLWallFitting() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  /**
   * @brief InitFittingModel Initialize vtkPCLFittingModel with the input polydata
   */
  vtkSmartPointer<vtkPCLFittingModel> InitFittingModel(vtkPolyData* poly);

  /**
   * @brief MergeInstanceDataArrays Use information provided by PCLFittingModel to assign an id to
   * each point according to the model it belongs to
   */
  void MergeInstanceDataArrays(vtkUnsignedIntArray* instanceArray,
    vtkUnsignedIntArray* newInstanceArray);

  /**
   * @brief FittingPlanesWithConstraints Fit instanceCount planes following the constraints function
   */
  void FittingPlanesWithConstraints(unsigned int instanceCount,
    std::function<bool(const Eigen::VectorXf&)> constraintsFunction,
    vtkPolyData* poly,
    vtkUnsignedIntArray* objectArray,
    PlaneType planeType);

  /**
   * @brief ModelVectorAngle Compute the angle between the normal of the model and a given vector
   */
  double ModelVectorAngle(const Eigen::VectorXf& model, Eigen::Vector3d vector);

  /**
   * @brief AngularConstraints Test if the normal of the model is close to the normal axis
   */
  bool AngularConstraints(const Eigen::VectorXf& model,
    Eigen::Vector3d vector,
    double constraintAngle);

  /**
   * @brief HorizontalConstraints Test if the normal of the model if close to the normal axis to
   * detect the floor and ceilling
   */
  bool HorizontalConstraints(const Eigen::VectorXf& model);

  /**
   * @brief VerticalConstraints Test if the normal of the model is orthogonal to the normal axis to
   * detect the walls
   */
  bool VerticalConstraints(const Eigen::VectorXf& model);

  /**
   * @brief WallConstraints Test VerticalConstraints and if the walls are parallel or orthogonal to
   * each other
   */
  bool WallConstraints(const Eigen::VectorXf& model);

  //! Define the sample consensus type used to approximate the model (RandomSampleConsensus by
  //! default)
  vtkPCLFittingModel::SampleConsensus SampleConsensusType = vtkPCLFittingModel::RANSAC;

  //! Maximum distance from point to model, to consider the point part of the model
  double DistanceThreshold = 0.1; // (in meters)

  //! Desired probability of choosing at least one sample free from outliers.
  double Probability = 0.99;

  //! Maximum number of iterations.
  int MaxIterations = 1000;

  //! Threshold angle used for accepting a plane model
  double ThresholdAngle = 10.0; // (in degrees)

  //! Flag to indicate if the constraints between walls should be used (orthogonality and
  //! parallelism)
  bool ConstraintsBetweenWalls = false;

  //! Normal axis used for the perpendicular plane
  double NormalAxis[3] = { 0.0, 0.0, 1.0 };

  //! Number of planes to fit for the floor and ceiling
  int NumberOfFloorCeillingPlanes = 2;
  //! Number of planes to fit for the walls
  int NumberOfWallPlanes = 4;

  //! Specify if the points should be projected on the planes
  bool ProjectInliers = false;

  //! Vector containing the ids of the outliers
  std::vector<int> OutlierIds;

  //! Counter for the number of plane instances currently found
  unsigned int InstanceCount = 0;

  //! Normal axis of the first wall plane found
  Eigen::Vector3d NormalWallAxis;
  //! Flag to indicate if the first wall plane has been found
  bool FirstWall = true;

private:
  vtkPCLWallFitting(const vtkPCLWallFitting&);
  void operator=(const vtkPCLWallFitting&);
};

#endif // vtkPCLWallFitting_h
