/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLSampleConsensusUtils.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkPCLSampleConsensusUtils.h"
#include "vtkPCLUtils.h"

#include <vtkCylinderSource.h>
#include <vtkPlaneSource.h>
#include <vtkPolyData.h>
#include <vtkTransform.h>
#include <vtkTransformPolyDataFilter.h>

#include <pcl/common/common.h>
#include <pcl/common/io.h>
#include <pcl/common/transforms.h>

namespace
{
constexpr int CYLINDER_RESOLUTION = 30;
}

//----------------------------------------------------------------------------
// Implementation of the New function
vtkStandardNewMacro(vtkPCLSampleConsensusUtils);

//-----------------------------------------------------------------------------
void vtkPCLSampleConsensusUtils::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

// ------------------------------------------------------------------------------
void vtkPCLSampleConsensusUtils::CreateVTKPlaneFromPCL(const pcl::PointCloud<pcl::PointXYZ>& cloud,
  const std::vector<int>& indices,
  vtkPolyData* output)
{
  Eigen::Vector3d centroid;
  Eigen::Matrix3d eigenVectors;
  Eigen::Vector3d eigenValues;
  vtkPCLUtils::ComputeMeanAndPCA<pcl::PointXYZ, double>(
    cloud, indices, centroid, eigenVectors, eigenValues);

  // Fill the plane axes
  Eigen::Vector3d normal = eigenVectors.col(0);
  Eigen::Vector3d widthDir = eigenVectors.col(2);
  Eigen::Vector3d heightDir = eigenVectors.col(1);

  Eigen::Isometry3d isometry = Eigen::Isometry3d::Identity();
  isometry.linear().col(0) = widthDir;
  isometry.linear().col(1) = heightDir;
  isometry.linear().col(2) = normal;
  isometry.translation() = centroid;

  // Transform the pointcloud to compute the width ,the height and the frame origin
  pcl::IndicesPtr pclIndices(new std::vector<int>(indices.begin(), indices.end()));
  pcl::PointCloud<pcl::PointXYZ> projectedPcl;
  pcl::transformPointCloud(cloud, *pclIndices, projectedPcl, isometry.inverse().cast<float>());

  // Get min and max points in the plane frame
  Eigen::Vector4f Pmin, Pmax;
  pcl::getMinMax3D(projectedPcl, Pmin, Pmax);

  double width = std::abs((Pmax - Pmin).x());
  double height = std::abs((Pmax - Pmin).y());

  Eigen::Vector3d denoisedPmin = { Pmin.x(), Pmin.y(), 0. };
  Eigen::Vector3d PminWorld = isometry * denoisedPmin;

  Eigen::Vector3d point1, point2;
  point1 = PminWorld + width * widthDir;
  point2 = PminWorld + height * heightDir;

  // Create a plane source
  vtkNew<vtkPlaneSource> planeSource;
  planeSource->SetOrigin(PminWorld[0], PminWorld[1], PminWorld[2]);
  planeSource->SetPoint1(point1[0], point1[1], point1[2]);
  planeSource->SetPoint2(point2[0], point2[1], point2[2]);
  planeSource->Update();
  output->ShallowCopy(planeSource->GetOutput());
}

// ------------------------------------------------------------------------------
void vtkPCLSampleConsensusUtils::CreateVTKCylinderFromPCL(
  const pcl::PointCloud<pcl::PointXYZ>& cloud,
  const std::vector<int>& indices,
  Eigen::Vector3d pointOnAxis,
  Eigen::Vector3d axisDirection,
  double radius,
  vtkPolyData* output)
{
  axisDirection.normalize();

  double minProjection = std::numeric_limits<double>::max();
  double maxProjection = std::numeric_limits<double>::lowest();

  for (const auto& idx : indices)
  {
    const pcl::PointXYZ& point = cloud.at(idx);
    Eigen::Vector3d pointVec(point.x, point.y, point.z);

    // Compute the projection of the point onto the cylinder axis
    double projection = (pointVec - pointOnAxis).dot(axisDirection);

    minProjection = std::min(minProjection, projection);
    maxProjection = std::max(maxProjection, projection);
  }
  double cylinderHeight = maxProjection - minProjection;

  Eigen::Vector3d cylinderStart = pointOnAxis + minProjection * axisDirection;
  Eigen::Vector3d cylinderEnd = pointOnAxis + maxProjection * axisDirection;
  Eigen::Vector3d cylinderCenter = (cylinderStart + cylinderEnd) / 2.0;

  vtkSmartPointer<vtkCylinderSource> cylinderSource = vtkSmartPointer<vtkCylinderSource>::New();
  cylinderSource->SetRadius(radius);
  cylinderSource->SetHeight(cylinderHeight);
  cylinderSource->SetResolution(::CYLINDER_RESOLUTION);

  // Set the center and orientation of the cylinder using a transformation
  vtkSmartPointer<vtkTransform> transform = vtkSmartPointer<vtkTransform>::New();
  transform->Translate(cylinderCenter.x(), cylinderCenter.y(), cylinderCenter.z());

  // Compute the rotation to align the default cylinder orientation with the axisDirection
  Eigen::Vector3d defaultAxis = Eigen::Vector3d::UnitY();
  Eigen::Vector3d rotationAxis = defaultAxis.cross(axisDirection);

  if (rotationAxis.norm() > 1e-6)
  {
    rotationAxis.normalize();

    double rotationAngle = vtkMath::DegreesFromRadians(std::acos(defaultAxis.dot(axisDirection)));
    transform->RotateWXYZ(rotationAngle, rotationAxis.x(), rotationAxis.y(), rotationAxis.z());
  }
  vtkSmartPointer<vtkTransformPolyDataFilter> transformFilter =
    vtkSmartPointer<vtkTransformPolyDataFilter>::New();
  transformFilter->SetInputConnection(cylinderSource->GetOutputPort());
  transformFilter->SetTransform(transform);
  transformFilter->Update();

  output->ShallowCopy(transformFilter->GetOutput());
}
