/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLPlaneAlignment.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLPlaneAlignment_h
#define vtkPCLPlaneAlignment_h

// VTK includes
#include <vtkPolyData.h>
#include <vtkPolyDataAlgorithm.h>
#include <vtkSmartPointer.h>

// Eigen includes
#include <Eigen/Dense>

#include "vtkPCLSampleConsensusModule.h" // For export macro

/**
 * @class vtkPCLPlaneAlignment
 * @brief This class try to fit a plane to a point cloud and use this plane to align the floor to
 * the XY plane. The plane is fitted using the RANSAC algorithm. By default, the main plane of the
 * pointcloud is fitted. If the UseNormalAxis is set to true, the plane will be fitted using the
 * NormalAxis and ThresholdAngle. If the AlignToOrigin is set to true, the plane will be centered
 * on the origin of the z axis. The n-th plane can be used to align the floor to the XY plane with
 * PlaneNumber. The approximate distance can be used to select the plane to align the floor with
 * more control.
 */
class VTKPCLSAMPLECONSENSUS_EXPORT vtkPCLPlaneAlignment : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLPlaneAlignment* New();
  vtkTypeMacro(vtkPCLPlaneAlignment, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  vtkGetMacro(DistanceThreshold, double);
  vtkSetMacro(DistanceThreshold, double);

  vtkGetMacro(Probability, double);
  vtkSetMacro(Probability, double);

  vtkGetMacro(MaxIterations, int);
  vtkSetMacro(MaxIterations, int);

  vtkGetMacro(Flip, bool);
  vtkSetMacro(Flip, bool);

  vtkGetMacro(UseNormalAxis, bool);
  vtkSetMacro(UseNormalAxis, bool);

  vtkGetVector3Macro(NormalAxis, double);
  vtkSetVector3Macro(NormalAxis, double);

  vtkGetMacro(ThresholdAngle, double);
  vtkSetMacro(ThresholdAngle, double);

  vtkGetMacro(TranslateToOrigin, bool);
  vtkSetMacro(TranslateToOrigin, bool);

  vtkGetMacro(PlaneNumber, int);
  vtkSetMacro(PlaneNumber, int);

  vtkGetMacro(UseApproximateDistance, bool);
  vtkSetMacro(UseApproximateDistance, bool);

  vtkGetMacro(ApproximateDistance, double);
  vtkSetMacro(ApproximateDistance, double);

  vtkGetMacro(TemporalAveraging, bool);
  void SetTemporalAveraging(bool value);

  vtkGetMacro(MaxTemporalAngleChange, double);
  vtkSetMacro(MaxTemporalAngleChange, double);

  vtkGetMacro(PrevEstimationWeight, double);
  vtkSetMacro(PrevEstimationWeight, double);

  vtkGetMacro(RefineCoefficients, bool);
  vtkSetMacro(RefineCoefficients, bool);

protected:
  // constructor / destructor
  vtkPCLPlaneAlignment() = default;
  ~vtkPCLPlaneAlignment() = default;

  // Request data
  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  /**
   * @brief RefineRansac Refine the plane coefficients
   */
  void RefineRansac(vtkPolyData* poly, Eigen::VectorXf& plane);

  //! Maximum distance from point to model, to consider the point part of the model
  double DistanceThreshold;

  //! Desired probability of choosing at least one sample free from outliers.
  double Probability;

  //! Maximum number of iterations.
  int MaxIterations;

  //! Specify if the plane should be flipped
  bool Flip = false;

  //! Specify if an approximate normal axis for the plane to be fitted should be used
  bool UseNormalAxis = false;

  //! Normal axis used for the perpendicular plane
  double NormalAxis[3] = { 0.0, 0.0, 1.0 };
  //! Threshold angle used for the perpendicular plane
  double ThresholdAngle = 10.0; // (in degrees)

  //! Specify if the plane should be aligned to the origin
  bool TranslateToOrigin = false;

  //! Specify if the approximate distance should be used, otherwise the PlaneNumber will be used
  bool UseApproximateDistance = false;
  //! Approximate signed distance from the origin to the plane
  double ApproximateDistance = 0.;
  //! Amount of plane candidates used when signed distance is used as a criteria for finding the
  //! best plane
  int DefaultPlaneNumber = 7;
  //! Align the pointcloud with the PlaneNumber-th plane found UseApproximateDistance is set to
  //! false
  int PlaneNumber = 1;

  //! Plane coefficients
  Eigen::VectorXf PrevPlaneCoeffs = Eigen::VectorXf::Zero(4);

  //! Flag to refine the plane coefficients using PCA
  bool RefineCoefficients = true;

  //! Flag to enable temporal averaging
  bool TemporalAveraging = false;
  //! Maximum angle change between two consecutive planes
  double MaxTemporalAngleChange = 20.0;
  //! Temporal averaging weight
  double PrevEstimationWeight = 0.9;

private:
  // copy operators
  vtkPCLPlaneAlignment(const vtkPCLPlaneAlignment&);
  void operator=(const vtkPCLPlaneAlignment&);
};

#endif // vtkPCLPlaneAlignment_h
