/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLFittingModel.h

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLFittingModel_h
#define vtkPCLFittingModel_h

// vtk includes
#include <vtkDataObjectAlgorithm.h>
#include <vtkImplicitFunction.h>
#include <vtkMath.h>
#include <vtkPolyData.h>
#include <vtkSmartPointer.h>
#include <vtkTable.h>
#include <vtkUnsignedIntArray.h>

#include <pcl/point_cloud.h>
#include <pcl/point_types.h>
#include <pcl/sample_consensus/ransac.h>

#include "vtkPCLSampleConsensusModule.h" // For export macro

/**
 * @brief The vtkPCLFittingModel class fits a model following a chosen sample consensus algorithm
 * (see SampleConsensus) within a point cloud using the pcl library. The model can be a circle, a
 * cylinder, a cone, a sphere, a line, a plane, a perpendicular plane or a normal plane (see Model).
 * The inliers of the model can be projected on the model. An "instance" array is added to the
 * output to indicate which points are in the model and which are not. The output can be the input
 * point cloud or the inliers of the model or the outliers of the model (see OutputMode).
 */
class VTKPCLSAMPLECONSENSUS_EXPORT vtkPCLFittingModel : public vtkDataObjectAlgorithm
{
public:
  static vtkPCLFittingModel* New();
  vtkTypeMacro(vtkPCLFittingModel, vtkDataObjectAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  enum SampleConsensus
  {
    LMEDS = 0,
    MLESAC,
    MSAC,
    PROSAC,
    RANSAC,
    RMSAC,
    RRANSAC
  };

  enum Model
  {
    Circle2D = 0,
    Circle3D,
    Cone,
    Cylinder,
    Sphere,
    Line,
    ParallelLine,
    Plane,
    PerpendicularPlane,
    NormalPlane
  };

  // AllData : output all the points of the input point cloud in a single output
  // Inliers : output all the inliers of the model in a single output
  // Outliers : output all the outliers of the model in a single output
  enum OutputMode
  {
    AllData = 0,
    Inliers,
    Outliers,
    InsidePoints,
    OutsidePoints,
    NoData
  };

  vtkGetMacro(SampleConsensusType, int);
  vtkSetMacro(SampleConsensusType, int);

  vtkGetMacro(DistanceThreshold, double);
  vtkSetMacro(DistanceThreshold, double);

  vtkGetMacro(Probability, double);
  vtkSetMacro(Probability, double);

  vtkGetMacro(MaxIterations, int);
  vtkSetMacro(MaxIterations, int);

  vtkGetMacro(ModelType, int);
  vtkSetMacro(ModelType, int);

  vtkGetMacro(IsSamplesMaxDistUsed, bool);
  vtkSetMacro(IsSamplesMaxDistUsed, bool);

  vtkGetMacro(SamplesMaxDist, double);
  vtkSetMacro(SamplesMaxDist, double);

  vtkSetMacro(OutputMode, int);
  vtkGetMacro(OutputMode, int);

  vtkSetMacro(ShowMeshes, bool);
  vtkGetMacro(ShowMeshes, bool);

  vtkGetMacro(MinRadiusModel, double);
  vtkSetMacro(MinRadiusModel, double);

  vtkGetMacro(MaxRadiusModelUsed, bool);
  vtkSetMacro(MaxRadiusModelUsed, bool);

  vtkGetMacro(MaxRadiusModel, double);
  vtkSetMacro(MaxRadiusModel, double);

  vtkGetMacro(SearchRadius, double);
  vtkSetMacro(SearchRadius, double);

  vtkGetVector3Macro(Axis, double);
  vtkSetVector3Macro(Axis, double);

  vtkGetMacro(ThresholdAngle, double);
  vtkSetMacro(ThresholdAngle, double);

  vtkGetMacro(ProjectInliers, bool);
  vtkSetMacro(ProjectInliers, bool);

  vtkGetMacro(InstanceCount, int);
  vtkSetMacro(InstanceCount, int);

  vtkGetMacro(FractionPreTest, double);
  vtkSetMacro(FractionPreTest, double);

  vtkGetMacro(EMIterations, int);
  vtkSetMacro(EMIterations, int);

  void SetIndices(std::vector<int>& indices) { this->Indices = indices; }
  std::vector<int>& GetIndices() { return this->Indices; }

  void SetModelConstraintsFunction(std::function<bool(const Eigen::VectorXf&)> function)
  {
    this->ModelConstraintsFunction = function;
  }

  const std::vector<Eigen::VectorXf>& GetModelCoefficients() const
  {
    return this->ModelCoefficients;
  }

protected:
  vtkPCLFittingModel();
  ~vtkPCLFittingModel() = default;

  int FillInputPortInformation(int port, vtkInformation* info) override;
  int RequestDataObject(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;
  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  /**
   * @brief CreateSampleConsensus Create the sample consensus object to approximate the model
   */
  pcl::SampleConsensus<pcl::PointXYZ>::Ptr CreateSampleConsensus(
    pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr Model);

  /**
   * @brief CreateModel Create the model to approximate
   */
  pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr CreateModel(
    pcl::PointCloud<pcl::PointXYZ>::Ptr pointCloud,
    pcl::PointCloud<pcl::Normal>::Ptr normals);

  /**
   * @brief SetPolyDataOutput Set the output according to the output mode
   */
  void SetPolyDataOutput(vtkPolyData* input, vtkUnsignedIntArray* objectArray);

  /**
   * @brief Return a vector of VTK implicit function with fitted models
   */
  std::vector<vtkSmartPointer<vtkImplicitFunction>> CreateImplicitFunctions();

  /**
   * @brief SetModelCoefficientsOutput Set the model coefficients output
   */
  void SetModelCoefficientsOutput(vtkTable* table);

  /**
   * @brief SetModelMeshOutput Initiate mesh models, return true if sucessful.
   */
  bool SetModelMeshOutput(const pcl::PointCloud<pcl::PointXYZ>& cloud,
    const std::vector<int>& indices,
    const Eigen::VectorXf coeff,
    vtkPolyData* output);

  //! Define the sample consensus type used to approximate the model (RandomSampleConsensus by
  //! default)
  int SampleConsensusType = 4;

  //! Maximum distance from point to model, to consider the point part of the model
  double DistanceThreshold;

  //! Desired probability of choosing at least one sample free from outliers.
  double Probability;

  //! Maximum number of iterations.
  int MaxIterations;

  //! Model to approximate, plane by default
  int ModelType = 7;

  //! Specify if the sample maximum distance should be used
  bool IsSamplesMaxDistUsed = false;
  //! Specify the maximum distance allowed when drawing random samples
  double SamplesMaxDist = vtkMath::Inf(); // (in meters)

  //! Output mode
  int OutputMode = 0;

  //! Show meshes
  bool ShowMeshes = false;

  //! Specify the minimum allowable radius limits for the model
  double MinRadiusModel = 0.0; // (in meters)
  //! Specify if the maximum radius should be used
  bool MaxRadiusModelUsed = false;
  //! Specify the maximum allowable radius limits for the model
  double MaxRadiusModel = vtkMath::Inf(); // (in meters)

  //! Specify the radius within which the surrounding points used to estimate the normal are
  //! selected
  double SearchRadius = 1.0; // (in meters)

  //! Axis used as normal for the perpendicular plane and as direction for the parallel line
  double Axis[3] = { 0.0, 0.0, 1.0 };
  //! Threshold angle used for the perpendicular plane
  double ThresholdAngle = 10.0; // (in degrees)

  //! Specify if the points should be projected on the model
  bool ProjectInliers = false;

  //! Number of instances of the model to find
  int InstanceCount = 1;

  //! Number of points to use for pre-testing (used for RRANSAC and RMSAC)
  double FractionPreTest = 0.1; // (in percentage)

  //! Number of EM iterations (used for MLESAC)
  int EMIterations = 10;

  //! Indices of the points to use
  std::vector<int> Indices;

  //! Specify a model constraints function to use in order to validate the model
  std::function<bool(const Eigen::VectorXf&)> ModelConstraintsFunction;

  //! Vector to save the coefficients of the models found
  std::vector<Eigen::VectorXf> ModelCoefficients;

private:
  vtkPCLFittingModel(const vtkPCLFittingModel&);
  void operator=(const vtkPCLFittingModel&);
};

#endif // vtkPCLFittingModel_h
