/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLFittingModel.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// local includes
#include "vtkPCLFittingModel.h"
#include "vtkPCLSampleConsensusUtils.h"
#include "vtkPCLUtils.h"

// vtk includes
#include <vtkCellData.h>
#include <vtkConvertToPointCloud.h>
#include <vtkCylinder.h>
#include <vtkExtractSelection.h>
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkMultiBlockDataSet.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>
#include <vtkSMPTools.h>
#include <vtkSelection.h>
#include <vtkSelectionNode.h>
#include <vtkSmartPointer.h>
#include <vtkSphere.h>
#include <vtkSphereSource.h>
#include <vtkStringArray.h>
#include <vtkType.h>
#include <vtkUnsignedIntArray.h>

// pcl includes
#include <pcl/point_types.h>
#include <pcl/sample_consensus/lmeds.h>
#include <pcl/sample_consensus/mlesac.h>
#include <pcl/sample_consensus/msac.h>
#include <pcl/sample_consensus/prosac.h>
#include <pcl/sample_consensus/ransac.h>
#include <pcl/sample_consensus/rmsac.h>
#include <pcl/sample_consensus/rransac.h>
#include <pcl/sample_consensus/sac.h>
#include <pcl/sample_consensus/sac_model_circle.h>
#include <pcl/sample_consensus/sac_model_circle3d.h>
#include <pcl/sample_consensus/sac_model_cone.h>
#include <pcl/sample_consensus/sac_model_cylinder.h>
#include <pcl/sample_consensus/sac_model_line.h>
#include <pcl/sample_consensus/sac_model_normal_plane.h>
#include <pcl/sample_consensus/sac_model_parallel_line.h>
#include <pcl/sample_consensus/sac_model_perpendicular_plane.h>
#include <pcl/sample_consensus/sac_model_plane.h>
#include <pcl/sample_consensus/sac_model_sphere.h>

// std includes
#include <memory>

namespace
{

// clang-format off
const std::vector<const char*> CIRCLE2D_COEFF = { "center_x", "center_y", "radius" };
const std::vector<const char*> CIRCLE3D_COEFF = { "center_x", "center_y", "center_z", "radius", "normal_x", "normal_y", "normal_z" };
const std::vector<const char*> CONE_COEFF = { "apex_x", "apex_y", "apex_z", "axis_direction_x", "axis_direction_y", "axis_direction_z", "opening_angle" };
const std::vector<const char*> CYLINDER_COEFF = { "point_on_axis_x", "point_on_axis_y", "point_on_axis_z", "axis_direction_x", "axis_direction_y", "axis_direction_z", "radius" };
const std::vector<const char*> SPHERE_COEFF = { "center_x", "center_y", "center_z", "radius" };
const std::vector<const char*> LINE_COEFF = { "point_on_line_x", "point_on_line_y", "point_on_line_z", "line_direction_x", "line_direction_y", "line_direction_z" };
const std::vector<const char*> PLANE_COEFF = { "normal_x", "normal_y", "normal_z", "d" };
// clang-format on

const std::map<vtkPCLFittingModel::Model, std::vector<const char*>> COEFF_NAMES = {
  { vtkPCLFittingModel::Circle2D, ::CIRCLE2D_COEFF },
  { vtkPCLFittingModel::Circle3D, ::CIRCLE3D_COEFF },
  { vtkPCLFittingModel::Cone, ::CONE_COEFF },
  { vtkPCLFittingModel::Cylinder, ::CYLINDER_COEFF },
  { vtkPCLFittingModel::Sphere, ::SPHERE_COEFF },
  { vtkPCLFittingModel::Line, ::LINE_COEFF },
  { vtkPCLFittingModel::ParallelLine, ::LINE_COEFF },
  { vtkPCLFittingModel::Plane, ::PLANE_COEFF },
  { vtkPCLFittingModel::PerpendicularPlane, ::PLANE_COEFF },
  { vtkPCLFittingModel::NormalPlane, ::PLANE_COEFF }
};
const std::vector<const char*> MODEL_NAMES = { "Circle2D",
  "Circle3D",
  "Cone",
  "Cylinder",
  "Sphere",
  "Line",
  "ParallelLine",
  "Plane",
  "PerpendicularPlane",
  "NormalPlane" };

constexpr int SPHERE_RESOLUTION = 30;
}

// Implementation of the New function
vtkStandardNewMacro(vtkPCLFittingModel);

vtkPCLFittingModel::vtkPCLFittingModel()
{
  this->SetNumberOfOutputPorts(2);
}

//-----------------------------------------------------------------------------
void vtkPCLFittingModel::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

int vtkPCLFittingModel::FillInputPortInformation(int port, vtkInformation* info)
{
  if (port == 0)
  {
    info->Set(vtkDataObject::DATA_TYPE_NAME(), "vtkPolyData");
    return 1;
  }
  return 0;
}

//----------------------------------------------------------------------------
int vtkPCLFittingModel::RequestDataObject(vtkInformation*,
  vtkInformationVector**,
  vtkInformationVector* outputVector)
{
  int outputType = this->ShowMeshes ? VTK_MULTIBLOCK_DATA_SET : VTK_POLY_DATA;
  vtkDataObjectAlgorithm::SetOutputDataObject(
    outputType, outputVector->GetInformationObject(0), true);
  vtkDataObjectAlgorithm::SetOutputDataObject(
    VTK_TABLE, outputVector->GetInformationObject(1), true);
  return 1;
}

//-----------------------------------------------------------------------------
int vtkPCLFittingModel::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));

  this->ModelCoefficients.clear();

  // Convert input data in pcl format
  auto pointCloud = std::make_shared<pcl::PointCloud<pcl::PointXYZ>>();
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);

  pcl::PointCloud<pcl::Normal>::Ptr cloudNormals;
  // Compute normals for the models that need them
  if (this->ModelType == vtkPCLFittingModel::NormalPlane ||
    this->ModelType == vtkPCLFittingModel::Cone || this->ModelType == vtkPCLFittingModel::Cylinder)
  {
    cloudNormals = std::make_shared<pcl::PointCloud<pcl::Normal>>();
    vtkPCLUtils::EstimatePCLNormals<pcl::PointXYZ>(pointCloud, this->SearchRadius, cloudNormals);
  }

  // Create a vtkArray to store the instance number of the model to which each point belongs (0
  // corresponds to outliers)
  vtkSmartPointer<vtkUnsignedIntArray> objectArray = vtkSmartPointer<vtkUnsignedIntArray>::New();
  objectArray->Allocate(input->GetNumberOfPoints());
  objectArray->SetName("instance");
  objectArray->SetNumberOfTuples(input->GetNumberOfPoints());
  objectArray->Fill(0);

  std::vector<int> outliers;

  unsigned int meshesIndex = 0;
  // Only used if this->ShowMeshes is true
  vtkMultiBlockDataSet* meshesCollection =
    vtkMultiBlockDataSet::GetData(outputVector->GetInformationObject(0));
  for (int i = 0; i < this->InstanceCount; i++)
  {
    // Inliers's index according to the model and threshold
    std::vector<int> inliers;

    // Instantiate the model
    pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr SampleConsensusModel =
      this->CreateModel(pointCloud, cloudNormals);

    // Set the radius limits if needed
    double maxRadiusLimit = this->MaxRadiusModelUsed ? this->MaxRadiusModel : vtkMath::Inf();
    SampleConsensusModel->setRadiusLimits(this->MinRadiusModel, maxRadiusLimit);

    // Set the sample maximum distance if needed
    if (this->IsSamplesMaxDistUsed)
    {
      auto tree = std::make_shared<pcl::search::KdTree<pcl::PointXYZ>>();
      tree->setInputCloud(pointCloud);
      SampleConsensusModel->setSamplesMaxDist(this->SamplesMaxDist, tree);
    }

    // Add a model constraints function if needed
    if (this->ModelConstraintsFunction != nullptr)
    {
      SampleConsensusModel->setModelConstraints(this->ModelConstraintsFunction);
    }

    // Use the indices if multiple instances of the model are searched to remove points already used
    // by a model
    if (this->InstanceCount > 1 && i > 0)
    {
      SampleConsensusModel->setIndices(outliers);
    }
    // Use the indices if provided by the user
    else if (!this->Indices.empty())
    {
      SampleConsensusModel->setIndices(this->Indices);
    }

    // Instanciate and launch the sample consensus
    pcl::SampleConsensus<pcl::PointXYZ>::Ptr sampleConsensus =
      this->CreateSampleConsensus(SampleConsensusModel);
    sampleConsensus->setDistanceThreshold(this->DistanceThreshold);
    sampleConsensus->setProbability(this->Probability);
    sampleConsensus->setMaxIterations(this->MaxIterations);
    sampleConsensus->setNumberOfThreads(vtkSMPTools::GetEstimatedNumberOfThreads());
    sampleConsensus->computeModel();
    sampleConsensus->getInliers(inliers); // get inlier list
    vtkDebugMacro("Inliers size:" << inliers.size());

    outliers.clear();

    // populate the new array
    vtkSMPTools::For(0,
      inliers.size(),
      [&](vtkIdType begin, vtkIdType end)
      {
        for (vtkIdType k = begin; k < end; ++k)
        {
          objectArray->SetValue(inliers[k], i + 1);
        }
      });

    // If indices are provided, the new outliers of the model are calculated from Indices, otherwise
    // the outliers are calculated from the input
    int indicesSize = !this->Indices.empty() ? this->Indices.size() : input->GetNumberOfPoints();
    for (int k = 0; k < indicesSize; ++k)
    {
      int index = !this->Indices.empty() ? this->Indices[k] : k;
      if (objectArray->GetValue(index) == 0)
      {
        outliers.push_back(index);
      }
    }

    // Save the model coefficients if points belong to the model
    if (!inliers.empty())
    {
      // Get Model coefficients
      Eigen::VectorXf modelCoefficients;
      sampleConsensus->getModelCoefficients(modelCoefficients);
      this->ModelCoefficients.push_back(modelCoefficients);
      // Project inliers on the model if requested
      if (this->ProjectInliers && this->ModelType != vtkPCLFittingModel::Sphere)
      {
        SampleConsensusModel->projectPoints(inliers, modelCoefficients, *pointCloud);
      }

      if (this->ShowMeshes)
      {
        // Add to the collection current detected inliers as VTK source
        vtkNew<vtkPolyData> source;
        if (this->SetModelMeshOutput(*pointCloud, inliers, modelCoefficients, source))
        {
          std::string blockName("Mesh-" + std::to_string(i));
          meshesCollection->SetBlock(meshesIndex, source);
          meshesCollection->GetMetaData(meshesIndex)
            ->Set(vtkCompositeDataSet::NAME(), blockName.c_str());
          meshesIndex += 1;
        }
      }
    }
  }

  vtkPolyData* output;
  if (this->ShowMeshes)
  {
    output = input->NewInstance();
  }
  else
  {
    output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  }
  output->ShallowCopy(input);
  vtkTable* coefficientsOutput = vtkTable::GetData(outputVector->GetInformationObject(1));

  // Add the array
  output->GetPointData()->AddArray(objectArray);
  // Set the active scalars to the instance array
  output->GetPointData()->SetActiveScalars("instance");

  // Convert the projected points to vtkPoints if requested
  if (this->ProjectInliers && this->ModelType != vtkPCLFittingModel::Sphere)
  {
    vtkSmartPointer<vtkPolyData> projected = vtkSmartPointer<vtkPolyData>::New();
    vtkPCLUtils::PointCloudToPolyData(*pointCloud, projected);
    output->SetPoints(projected->GetPoints());
  }

  // Modify the output to select only the inliers, outliers, inside or outside if requested
  this->SetPolyDataOutput(output, objectArray);

  if (this->ShowMeshes)
  {
    if (output->GetNumberOfPoints())
    {
      meshesCollection->SetBlock(meshesIndex, output);
      meshesCollection->GetMetaData(meshesIndex)->Set(vtkCompositeDataSet::NAME(), "Point Cloud");
    }
    else
    {
      output->Delete();
    }
  }
  // Write the model coefficients in a vtkTable
  this->SetModelCoefficientsOutput(coefficientsOutput);

  return 1;
}

//-----------------------------------------------------------------------------
pcl::SampleConsensus<pcl::PointXYZ>::Ptr vtkPCLFittingModel::CreateSampleConsensus(
  pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr Model)
{
  pcl::SampleConsensus<pcl::PointXYZ>::Ptr sampleConsensus;

  switch (this->SampleConsensusType)
  {
    case vtkPCLFittingModel::LMEDS:
      sampleConsensus = std::make_shared<pcl::LeastMedianSquares<pcl::PointXYZ>>(Model);
      break;

    case vtkPCLFittingModel::MLESAC:
    {
      auto MLESAC = std::make_shared<pcl::MaximumLikelihoodSampleConsensus<pcl::PointXYZ>>(Model);
      MLESAC->setEMIterations(this->EMIterations);
      sampleConsensus = pcl::SampleConsensus<pcl::PointXYZ>::Ptr(MLESAC);
      break;
    }

    case vtkPCLFittingModel::MSAC:
      sampleConsensus = std::make_shared<pcl::MEstimatorSampleConsensus<pcl::PointXYZ>>(Model);
      break;

    case vtkPCLFittingModel::PROSAC:
      sampleConsensus = std::make_shared<pcl::ProgressiveSampleConsensus<pcl::PointXYZ>>(Model);
      break;

    case vtkPCLFittingModel::RMSAC:
    {
      auto RMSAC = std::make_shared<pcl::RandomizedMEstimatorSampleConsensus<pcl::PointXYZ>>(Model);
      RMSAC->setFractionNrPretest(this->FractionPreTest);
      sampleConsensus = pcl::SampleConsensus<pcl::PointXYZ>::Ptr(RMSAC);
      break;
    }

    case vtkPCLFittingModel::RRANSAC:
    {
      auto RRANSAC = std::make_shared<pcl::RandomizedRandomSampleConsensus<pcl::PointXYZ>>(Model);
      RRANSAC->setFractionNrPretest(this->FractionPreTest);
      sampleConsensus = pcl::SampleConsensus<pcl::PointXYZ>::Ptr(RRANSAC);
      break;
    }

    case vtkPCLFittingModel::RANSAC:
    default:
      sampleConsensus = std::make_shared<pcl::RandomSampleConsensus<pcl::PointXYZ>>(Model);
      break;
  }

  return sampleConsensus;
}

//-----------------------------------------------------------------------------
pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr vtkPCLFittingModel::CreateModel(
  pcl::PointCloud<pcl::PointXYZ>::Ptr pointCloud,
  pcl::PointCloud<pcl::Normal>::Ptr cloudNormals)
{
  pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr scModel;

  switch (this->ModelType)
  {
    case vtkPCLFittingModel::Circle2D:
      scModel = std::make_shared<pcl::SampleConsensusModelCircle2D<pcl::PointXYZ>>(pointCloud);
      break;

    case vtkPCLFittingModel::Circle3D:
      scModel = std::make_shared<pcl::SampleConsensusModelCircle3D<pcl::PointXYZ>>(pointCloud);
      break;

    case vtkPCLFittingModel::Cone:
    {
      auto scConeModel =
        std::make_shared<pcl::SampleConsensusModelCone<pcl::PointXYZ, pcl::Normal>>(pointCloud);
      scConeModel->setInputNormals(cloudNormals);
      scModel = pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr(scConeModel);
      break;
    }

    case vtkPCLFittingModel::Cylinder:
    {
      auto scCylinderModel =
        std::make_shared<pcl::SampleConsensusModelCylinder<pcl::PointXYZ, pcl::Normal>>(pointCloud);
      scCylinderModel->setInputNormals(cloudNormals);
      scModel = pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr(scCylinderModel);
      break;
    }

    case vtkPCLFittingModel::Sphere:
      scModel = std::make_shared<pcl::SampleConsensusModelSphere<pcl::PointXYZ>>(pointCloud);
      break;

    case vtkPCLFittingModel::Line:
      scModel = std::make_shared<pcl::SampleConsensusModelLine<pcl::PointXYZ>>(pointCloud);
      break;

    case vtkPCLFittingModel::ParallelLine:
    {
      auto scParallelLineModel =
        std::make_shared<pcl::SampleConsensusModelParallelLine<pcl::PointXYZ>>(pointCloud);
      scParallelLineModel->setAxis(Eigen::Vector3f(this->Axis[0], this->Axis[1], this->Axis[2]));
      scParallelLineModel->setEpsAngle(vtkMath::RadiansFromDegrees(this->ThresholdAngle));
      scModel = pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr(scParallelLineModel);
      break;
    }

    case vtkPCLFittingModel::Plane:
      scModel = std::make_shared<pcl::SampleConsensusModelPlane<pcl::PointXYZ>>(pointCloud);
      break;

    case vtkPCLFittingModel::PerpendicularPlane:
    {
      auto scPerpendicularPlaneModel =
        std::make_shared<pcl::SampleConsensusModelPerpendicularPlane<pcl::PointXYZ>>(pointCloud);
      scPerpendicularPlaneModel->setAxis(
        Eigen::Vector3f(this->Axis[0], this->Axis[1], this->Axis[2]));
      scPerpendicularPlaneModel->setEpsAngle(vtkMath::RadiansFromDegrees(this->ThresholdAngle));
      scModel = pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr(scPerpendicularPlaneModel);
      break;
    }

    case vtkPCLFittingModel::NormalPlane:
    {
      auto scNormalPlaneModel =
        std::make_shared<pcl::SampleConsensusModelNormalPlane<pcl::PointXYZ, pcl::Normal>>(
          pointCloud);
      scNormalPlaneModel->setInputNormals(cloudNormals);
      scModel = pcl::SampleConsensusModel<pcl::PointXYZ>::Ptr(scNormalPlaneModel);
      break;
    }
    default:
      vtkErrorMacro("No model : " << this->ModelType);
      break;
  }

  return scModel;
}

//-----------------------------------------------------------------------------
void vtkPCLFittingModel::SetPolyDataOutput(vtkPolyData* poly, vtkUnsignedIntArray* objectArray)
{
  if (this->OutputMode == vtkPCLFittingModel::AllData)
  {
    return;
  }
  if (this->OutputMode == vtkPCLFittingModel::NoData)
  {
    poly->ShallowCopy(vtkPolyData::New());
    return;
  }

  // Use vtkExtractPointSelection to select the inliers or the outliers using a vtkSelection
  vtkSmartPointer<vtkExtractSelection> extractInliers = vtkSmartPointer<vtkExtractSelection>::New();
  extractInliers->SetInputData(0, poly);
  vtkSmartPointer<vtkSelectionNode> selectionNode = vtkSmartPointer<vtkSelectionNode>::New();
  selectionNode->SetFieldType(vtkSelectionNode::POINT);
  selectionNode->SetContentType(vtkSelectionNode::INDICES);
  vtkSmartPointer<vtkSelection> selection = vtkSmartPointer<vtkSelection>::New();
  vtkSmartPointer<vtkIdTypeArray> selectedIds = vtkSmartPointer<vtkIdTypeArray>::New();

  switch (this->OutputMode)
  {
    case vtkPCLFittingModel::Inliers:
      for (int k = 0; k < poly->GetNumberOfPoints(); ++k)
      {
        if (objectArray->GetValue(k) != 0)
        {
          selectedIds->InsertNextValue(k);
        }
      }
      break;

    case vtkPCLFittingModel::Outliers:
      for (int k = 0; k < poly->GetNumberOfPoints(); ++k)
      {
        if (objectArray->GetValue(k) == 0)
        {
          selectedIds->InsertNextValue(k);
        }
      }
      break;

    case vtkPCLFittingModel::InsidePoints:
    case vtkPCLFittingModel::OutsidePoints:
    {
      std::vector<vtkSmartPointer<vtkImplicitFunction>> functions = this->CreateImplicitFunctions();
      // Model not implemented default to AllData output
      if (functions.empty())
        return;

      const bool isOutputModeInside = this->OutputMode == vtkPCLFittingModel::InsidePoints;
      for (int k = 0; k < poly->GetNumberOfPoints(); ++k)
      {
        double point[3];
        poly->GetPoint(k, point);
        bool isInside = false;
        for (auto& function : functions)
        {
          if (function->FunctionValue(point) <= 0)
          {
            isInside = true;
            break;
          }
        }
        if (isInside == isOutputModeInside)
        {
          selectedIds->InsertNextValue(k);
        }
      }
      break;
    }

    default:
      break;
  }

  // If no inliers or outliers, return an empty polydata
  if (selectedIds->GetNumberOfTuples() == 0)
  {
    vtkWarningMacro("No inliers or outliers found, the output is empty");
    poly->ShallowCopy(vtkPolyData::New());
    return;
  }

  selectionNode->SetSelectionList(selectedIds);
  selection->AddNode(selectionNode);
  extractInliers->SetInputData(1, selection);
  extractInliers->Update();

  // Convert to polydata pointcloud
  vtkNew<vtkConvertToPointCloud> convertToPointCloud;
  convertToPointCloud->SetInputData(extractInliers->GetOutput());
  // Generate one vertex cell by point to be coherent with LidarView's pointcloud
  convertToPointCloud->SetCellGenerationMode(vtkConvertToPointCloud::VERTEX_CELLS);
  convertToPointCloud->Update();
  poly->ShallowCopy(convertToPointCloud->GetOutput());
}

//-----------------------------------------------------------------------------
std::vector<vtkSmartPointer<vtkImplicitFunction>> vtkPCLFittingModel::CreateImplicitFunctions()
{
  std::vector<vtkSmartPointer<vtkImplicitFunction>> functions;
  for (const auto& coeffs : this->ModelCoefficients)
  {
    // Cone not implemented as vtk implicit function does not have same parameters
    switch (this->ModelType)
    {
      case vtkPCLFittingModel::Sphere:
      {
        auto sphere = vtkSmartPointer<vtkSphere>::New();
        sphere->SetCenter(coeffs[0], coeffs[1], coeffs[2]);
        sphere->SetRadius(coeffs[3] + this->DistanceThreshold);
        functions.emplace_back(sphere);
        break;
      }

      case vtkPCLFittingModel::Cylinder:
      {
        auto cylinder = vtkSmartPointer<vtkCylinder>::New();
        cylinder->SetCenter(coeffs[0], coeffs[1], coeffs[2]);
        cylinder->SetAxis(coeffs[3], coeffs[4], coeffs[5]);
        cylinder->SetRadius(coeffs[6] + this->DistanceThreshold);
        functions.emplace_back(cylinder);
        break;
      }
      default:
        break;
    }
  }

  return std::move(functions);
}

//-----------------------------------------------------------------------------
void vtkPCLFittingModel::SetModelCoefficientsOutput(vtkTable* output)
{
  // Write the model descriptions in a vtkTable
  vtkNew<vtkTable> table;
  vtkNew<vtkStringArray> col1;
  col1->SetName("model_type");
  table->AddColumn(col1);

  if (this->ModelType >= ::MODEL_NAMES.size())
  {
    vtkErrorMacro("Model type name is not associated.");
    return;
  }

  const auto& names = ::COEFF_NAMES.at(static_cast<vtkPCLFittingModel::Model>(this->ModelType));
  if (names.size() != this->ModelCoefficients[0].size())
  {
    vtkErrorMacro("Mismatch between number of coefficients and number of names associated.");
    return;
  }

  for (const auto& name : names)
  {
    vtkNew<vtkFloatArray> col;
    col->SetName(name);
    table->AddColumn(col);
  }

  table->SetNumberOfRows(this->ModelCoefficients.size());
  for (unsigned int i = 0; i < this->ModelCoefficients.size(); i++)
  {
    table->SetValue(i, 0, ::MODEL_NAMES[this->ModelType]);
    for (unsigned int j = 0; j < this->ModelCoefficients[i].size(); j++)
    {
      table->SetValue(i, j + 1, this->ModelCoefficients[i][j]);
    }
  }
  output->ShallowCopy(table);
}

//-----------------------------------------------------------------------------
bool vtkPCLFittingModel::SetModelMeshOutput(const pcl::PointCloud<pcl::PointXYZ>& cloud,
  const std::vector<int>& indices,
  const Eigen::VectorXf coeff,
  vtkPolyData* output)
{
  switch (this->ModelType)
  {
    case vtkPCLFittingModel::Cylinder:
    {
      Eigen::Vector3d pointOnAxis(coeff[0], coeff[1], coeff[2]);
      Eigen::Vector3d axisDirection(coeff[3], coeff[4], coeff[5]);
      double radius = coeff[6];
      vtkPCLSampleConsensusUtils::CreateVTKCylinderFromPCL(
        cloud, indices, pointOnAxis, axisDirection, radius, output);
      return true;
    }

    case vtkPCLFittingModel::Sphere:
    {
      vtkNew<vtkSphereSource> sphereSource;
      sphereSource->SetCenter(coeff[0], coeff[1], coeff[2]);
      sphereSource->SetRadius(coeff[3]);
      sphereSource->SetThetaResolution(::SPHERE_RESOLUTION);
      sphereSource->SetPhiResolution(::SPHERE_RESOLUTION);
      sphereSource->Update();
      output->ShallowCopy(sphereSource->GetOutput());
      return true;
    }

    case vtkPCLFittingModel::Plane:
    case vtkPCLFittingModel::PerpendicularPlane:
    case vtkPCLFittingModel::NormalPlane:
    {
      vtkPCLSampleConsensusUtils::CreateVTKPlaneFromPCL(cloud, indices, output);
      return true;
    }

    // Not yet implemented
    default:
      return false;
  }
}
