/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLVoxelGrid.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Local includes
#include "vtkPCLVoxelGrid.h"
#include "vtkPCLFiltersUtils.h"
#include "vtkPCLUtils.h"

// PCL includes
#include <pcl/filters/uniform_sampling.h>

// VTK includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLVoxelGrid);

//-----------------------------------------------------------------------------
void vtkPCLVoxelGrid::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLVoxelGrid::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  if (this->LeafSize <= 0)
  {
    vtkErrorMacro("LeafSize must be positive");
    return 0;
  }
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Convert input data in pcl format
  pcl::PointCloud<Point>::Ptr pointCloud(new pcl::PointCloud<Point>);
  vtkPCLUtils::PolyDataToPointCloud(input, *pointCloud);
  // Convert the input to a PCL point cloud
  pcl::PointCloud<Point>::Ptr outputCloud(new pcl::PointCloud<Point>);

  // Apply the PCL voxel grid filter
  // NOTE : UniformSampling is preferred over VoxelGrid because of getRemovedIndices() function
  pcl::UniformSampling<Point> filter(true);
  filter.setInputCloud(pointCloud);
  filter.setRadiusSearch(this->LeafSize);
  filter.filter(*outputCloud);

  // Remove filtered points from the input
  const std::vector<int>* removedIndices = filter.getRemovedIndices().get();

  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  output->ShallowCopy(vtkPCLFiltersUtils::RemoveIndices(input, removedIndices));

  return 1;
}
