/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLOutlierRemoval.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLOutlierRemoval_h
#define vtkPCLOutlierRemoval_h

// VTK includes
#include <vtkPolyDataAlgorithm.h>

// PCL includes
#include <pcl/point_cloud.h>
#include <pcl/point_types.h>

#include "vtkPCLFiltersModule.h" // for export macro

/**
 * @brief The vtkPCLOutlierRemoval class filters outlier points in a point cloud with. A statistical
 * or radius based approach can be used to determine if a point is an outlier. This filter wraps the
 * OutlierRemoval classes from the PCL library.
 */
class VTKPCLFILTERS_EXPORT vtkPCLOutlierRemoval : public vtkPolyDataAlgorithm
{
public:
  static vtkPCLOutlierRemoval* New();
  vtkTypeMacro(vtkPCLOutlierRemoval, vtkPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  using Point = pcl::PointXYZ;

  enum AlgorithmType
  {
    RadiusRemoval = 0,
    StatisticalRemoval
  };

  vtkGetMacro(Algorithm, AlgorithmType);
  vtkSetMacro(Algorithm, AlgorithmType);
  void SetAlgorithm(int algorithm)
  {
    this->Algorithm = static_cast<AlgorithmType>(algorithm);
    this->Modified();
  }

  vtkGetMacro(MinNeighbors, int);
  vtkSetMacro(MinNeighbors, int);

  vtkGetMacro(Radius, double);
  vtkSetMacro(Radius, double);

  vtkGetMacro(MeanK, int);
  vtkSetMacro(MeanK, int);

  vtkGetMacro(StddevMulThresh, double);
  vtkSetMacro(StddevMulThresh, double);

protected:
  vtkPCLOutlierRemoval() = default;
  ~vtkPCLOutlierRemoval() = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

private:
  vtkPCLOutlierRemoval(const vtkPCLOutlierRemoval&);
  void operator=(const vtkPCLOutlierRemoval&);

  //! Algorithm to use for outlier removal.
  AlgorithmType Algorithm = RadiusRemoval;

  //! Number of neighbors that need to be present in order to be classified as an inlier. Used by
  //! RadiusRemoval.
  int MinNeighbors = 1;

  //! Radius of the sphere that will determine which points are neighbors. Used by RadiusRemoval.
  double Radius = 1.0;

  //! Number of nearest neighbors to use for mean distance estimation. Used by StatisticalRemoval.
  int MeanK = 8;

  //! Standard deviation multiplier for the distance threshold calculation. Used by
  //! StatisticalRemoval.
  double StddevMulThresh = 1.0;
};

#endif // vtkPCLOutlierRemoval_h
