/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLOutlierRemoval.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Local includes
#include "vtkPCLOutlierRemoval.h"
#include "vtkPCLFiltersUtils.h"
#include "vtkPCLUtils.h"

// PCL includes
#include <pcl/filters/radius_outlier_removal.h>
#include <pcl/filters/statistical_outlier_removal.h>

// VTK includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLOutlierRemoval);

//-----------------------------------------------------------------------------
void vtkPCLOutlierRemoval::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLOutlierRemoval::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Convert input data in pcl format
  pcl::PointCloud<Point>::Ptr inputCloud(new pcl::PointCloud<Point>);
  vtkPCLUtils::PolyDataToPointCloud(input, *inputCloud);
  pcl::PointCloud<Point>::Ptr outputCloud(new pcl::PointCloud<Point>);

  pcl::FilterIndices<Point>::Ptr filter;
  switch (this->Algorithm)
  {
    case vtkPCLOutlierRemoval::RadiusRemoval:
    {
      pcl::RadiusOutlierRemoval<Point>::Ptr radiusFilter(
        new pcl::RadiusOutlierRemoval<Point>(true));
      radiusFilter->setRadiusSearch(this->Radius);
      radiusFilter->setMinNeighborsInRadius(this->MinNeighbors);
      filter = radiusFilter;
      break;
    }

    case vtkPCLOutlierRemoval::StatisticalRemoval:
    {
      pcl::StatisticalOutlierRemoval<Point>::Ptr statisticalFilter(
        new pcl::StatisticalOutlierRemoval<Point>(true));
      statisticalFilter->setMeanK(this->MeanK);
      statisticalFilter->setStddevMulThresh(this->StddevMulThresh);
      filter = statisticalFilter;
      break;
    }

    default:
      vtkErrorMacro("Invalid algorithm");
      return 0;
  }

  // Apply the filter
  filter->setInputCloud(inputCloud);
  filter->filter(*outputCloud);

  // Remove filtered points from the input
  const std::vector<int>* removedIndices = filter->getRemovedIndices().get();

  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  output->ShallowCopy(vtkPCLFiltersUtils::RemoveIndices(input, removedIndices));

  return 1;
}
