/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLNormalSpaceSampling.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// local includes
#include "vtkPCLNormalSpaceSampling.h"
#include "vtkPCLFiltersUtils.h"
#include "vtkPCLUtils.h"

// pcl includes
#include <pcl/features/normal_3d.h>
#include <pcl/filters/normal_space.h>

// vtk includes
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPolyData.h>

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLNormalSpaceSampling);

//-----------------------------------------------------------------------------
void vtkPCLNormalSpaceSampling::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//-----------------------------------------------------------------------------
int vtkPCLNormalSpaceSampling::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector,
  vtkInformationVector* outputVector)
{
  // Get the input
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]->GetInformationObject(0));
  // Convert input data in pcl format
  pcl::PointCloud<Point>::Ptr inputCloud(new pcl::PointCloud<Point>);
  vtkPCLUtils::PolyDataToPointCloud(input, *inputCloud);
  // Convert the input to a PCL point cloud
  pcl::PointCloud<Point>::Ptr outputCloud(new pcl::PointCloud<Point>);

  // Compute the normals
  pcl::PointCloud<pcl::Normal>::Ptr cloudNormals(new pcl::PointCloud<pcl::Normal>);
  vtkPCLUtils::EstimatePCLNormals<Point>(inputCloud, this->Radius, cloudNormals);

  // Apply the PCL Normal Space Sampling filter
  pcl::NormalSpaceSampling<Point, pcl::Normal> filter(true);
  filter.setInputCloud(inputCloud);
  filter.setNormals(cloudNormals);
  filter.setSample(this->Sample);
  filter.setSeed(this->Seed);
  filter.setBins(this->Bins[0], this->Bins[1], this->Bins[2]);
  filter.filter(*outputCloud);

  const std::vector<int>* removedIndices = filter.getRemovedIndices().get();

  vtkPolyData* output = vtkPolyData::GetData(outputVector->GetInformationObject(0));
  output->ShallowCopy(vtkPCLFiltersUtils::RemoveIndices(input, removedIndices));

  return 1;
}
