/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLFiltersUtils.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Local includes
#include "vtkPCLFiltersUtils.h"

// VTK includes
#include <vtkExtractSelection.h>
#include <vtkIdList.h>
#include <vtkInformation.h>
#include <vtkNew.h>
#include <vtkPolyData.h>
#include <vtkSMPTools.h>
#include <vtkSelection.h>
#include <vtkSelectionNode.h>

// STD includes
#include <numeric>

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPCLFiltersUtils);

//----------------------------------------------------------------------------
vtkSmartPointer<vtkPolyData> vtkPCLFiltersUtils::RemoveIndices(vtkPolyData* input,
  const std::vector<int>* removedIndices)
{
  // Use ExtractSelection to remove the points that were removed by the filter
  vtkNew<vtkExtractSelection> extractSelection;
  extractSelection->SetInputData(0, input);
  vtkNew<vtkSelectionNode> selectionNode;
  selectionNode->SetFieldType(vtkSelectionNode::POINT);
  // Inverse the selection to remove the points that were removed by the filter
  selectionNode->GetProperties()->Set(vtkSelectionNode::INVERSE(), 1);
  selectionNode->SetContentType(vtkSelectionNode::INDICES);
  vtkNew<vtkSelection> selection;
  vtkNew<vtkIdTypeArray> selectedIds;
  for (const int idx : *removedIndices)
  {
    selectedIds->InsertNextValue(idx);
  }

  selectionNode->SetSelectionList(selectedIds);
  selection->AddNode(selectionNode);
  extractSelection->SetInputData(1, selection);
  extractSelection->Update();

  vtkSmartPointer<vtkPolyData> output = vtkSmartPointer<vtkPolyData>::New();
  output->ShallowCopy(extractSelection->GetOutput());
  // vtkSelection does not provide one cell per point, so cells are added manually
  vtkNew<vtkCellArray> cells;
  cells->SetNumberOfCells(output->GetNumberOfPoints());

  vtkNew<vtkIdList> ids;
  ids->SetNumberOfIds(output->GetNumberOfPoints());
  std::iota(ids->begin(), ids->end(), 0);

  cells->InsertNextCell(ids);
  output->SetVerts(cells);

  return output;
}
