/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLUtils.txx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLUtils_txx
#define vtkPCLUtils_txx

#include <pcl/features/normal_3d_omp.h>
#include <pcl/point_types.h>
#include <pcl/search/kdtree.h>

#include <vtkSMPTools.h>

#include "vtkPCLUtils.h"

// ------------------------------------------------------------------------------
template <typename PointT, typename Scalar>
void vtkPCLUtils::ComputeMeanAndPCA(const pcl::PointCloud<PointT>& cloud,
  const std::vector<int>& indices,
  Eigen::Matrix<Scalar, 3, 1>& centroid,
  Eigen::Matrix<Scalar, 3, 3>& eigenVectors,
  Eigen::Matrix<Scalar, 3, 1>& eigenValues)
{
  // Compute mean and normalized covariance matrix
  EIGEN_ALIGN16 Eigen::Matrix<Scalar, 3, 3> covarianceMatrix;
  EIGEN_ALIGN16 Eigen::Matrix<Scalar, 4, 1> xyzCentroid;
  pcl::computeMeanAndCovarianceMatrix(cloud, indices, covarianceMatrix, xyzCentroid);
  centroid = xyzCentroid.template head<3>();

  // Compute eigen values and corresponding eigen vectors
  pcl::eigen33(covarianceMatrix, eigenVectors, eigenValues);
}

// ------------------------------------------------------------------------------
template <typename PointT>
void vtkPCLUtils::EstimatePCLNormals(typename pcl::PointCloud<PointT>::Ptr inputCloud,
  double searchRadius,
  pcl::PointCloud<pcl::Normal>::Ptr outputNormals)
{
  using KdTreeInPtr = typename pcl::search::KdTree<PointT>::Ptr;

  // A kd-tree is used to search for the nearest neighbors
  KdTreeInPtr tree(new pcl::search::KdTree<PointT>());

  pcl::NormalEstimationOMP<PointT, pcl::Normal> normalEstimation;
  normalEstimation.setInputCloud(inputCloud);
  normalEstimation.setSearchMethod(tree);
  normalEstimation.setRadiusSearch(searchRadius);
  normalEstimation.setNumberOfThreads(vtkSMPTools::GetEstimatedNumberOfThreads());
  normalEstimation.compute(*outputNormals);
}

#endif
