/*=========================================================================

  Program:   PCL Plugin
  Module:    vtkPCLUtils.h

  Copyright (c) Kitware Inc.
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef vtkPCLUtils_h
#define vtkPCLUtils_h

#include <vtkObject.h>

#include <pcl/common/common.h>
#include <pcl/point_types.h>

#include "vtkPCLCommonModule.h" // For export macro

class vtkPolyData;
class vtkStructuredGrid;

/**
 * @brief Collection of static convenient functions for PCL filters wrapped in VTK.
 */
class VTKPCLCOMMON_EXPORT vtkPCLUtils : public vtkObject
{
  template <typename PointT>
  using PointTPtr = typename pcl::PointCloud<PointT>::Ptr;

public:
  static vtkPCLUtils* New();

  vtkTypeMacro(vtkPCLUtils, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  ///@{
  /**
   * @brief Conversion function from VTK data structure to PCL
   */
  template <typename PointT>
  static void PolyDataToPointCloud(vtkPolyData* const polydata, pcl::PointCloud<PointT>& cloud);
  template <typename PointT>
  static void StructuredGridToPointCloud(vtkStructuredGrid* const grid,
    pcl::PointCloud<PointT>& cloud);
  ///@}

  ///@{
  /**
   * @brief Conversion function from PCL data structure to VTK
   */
  template <typename PointT>
  static void PointCloudToPolyData(const pcl::PointCloud<PointT>& cloud, vtkPolyData* const pdata);
  template <typename PointT>
  static void PointCloudToStructuredGrid(const pcl::PointCloud<PointT>& cloud,
    vtkStructuredGrid* const grid);
  ///@}

  /**
   * @brief Get the centroid and the eigen vectors / eigen values
   * of a pointcloud
   */
  template <typename PointT, typename Scalar>
  static void ComputeMeanAndPCA(const pcl::PointCloud<PointT>& cloud,
    const std::vector<int>& indices,
    Eigen::Matrix<Scalar, 3, 1>& centroid,
    Eigen::Matrix<Scalar, 3, 3>& eigenVectors,
    Eigen::Matrix<Scalar, 3, 1>& eigenValues);

  /**
   * @brief Estimate normal of input point cloud using PCL NormalEstimation
   */
  template <typename PointT>
  static void EstimatePCLNormals(typename pcl::PointCloud<PointT>::Ptr inputCloud,
    double searchRadius,
    pcl::PointCloud<pcl::Normal>::Ptr outputNormals);

  /**
   * @brief Estimate normal with vtkPCLUtils::EstimatePCLNormals and merge them
   * in a pcl::PointNormal.
   */
  static pcl::PointCloud<pcl::PointNormal>::Ptr EstimateAndMergePCLNormals(
    pcl::PointCloud<pcl::PointXYZ>::Ptr inputCloud,
    double searchRadius);

protected:
  vtkPCLUtils() = default;
  ~vtkPCLUtils() = default;

private:
  vtkPCLUtils(const vtkPCLUtils&);
  void operator=(const vtkPCLUtils&);
};

#include "vtkPCLConversion.txx" // for template implementations
#include "vtkPCLUtils.txx"      // for template implementations

#endif // vtkPCLUtils_h
