/*=========================================================================

  Program:   LidarView
  Module:    vtkLivoxPosePacketInterpreter.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkLivoxPosePacketInterpreter.h"

#include "InterpreterHelper.h"
#include "vtk_livox_sdk2.h"

#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkSmartPointer.h>

namespace livox_sdk2 = livox::sdk2;

namespace
{
typedef livox_sdk2::LivoxLidarEthernetPacket LivoxSDK2Packet;

typedef union
{
  uint8_t stamp_bytes[8];
  int64_t stamp;
} LdsStamp;

uint64_t GetEthPacketTimestamp(const uint8_t* time_stamp, uint8_t size)
{
  LdsStamp time;
  memcpy(time.stamp_bytes, time_stamp, size);
  return time.stamp;
}

const uint64_t kRatioOfMsToNs = 1000000;
}

//-----------------------------------------------------------------------------
class vtkLivoxPosePacketInterpreter::vtkInternals
{
public:
  vtkSmartPointer<vtkFloatArray> GyroX;
  vtkSmartPointer<vtkFloatArray> GyroY;
  vtkSmartPointer<vtkFloatArray> GyroZ;
  vtkSmartPointer<vtkFloatArray> AccelX;
  vtkSmartPointer<vtkFloatArray> AccelY;
  vtkSmartPointer<vtkFloatArray> AccelZ;
  vtkSmartPointer<vtkDoubleArray> Timestamp;
};

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkLivoxPosePacketInterpreter)

//-----------------------------------------------------------------------------
vtkLivoxPosePacketInterpreter::vtkLivoxPosePacketInterpreter()
  : Internals(new vtkLivoxPosePacketInterpreter::vtkInternals())

{
  this->ResetCurrentData();
}

//----------------------------------------------------------------------------
void vtkLivoxPosePacketInterpreter::InitArrays()
{
  auto& internals = this->Internals;

  AddArrayToTableIfNotNull(internals->GyroX, this->RawInformation, "gyro_x");
  AddArrayToTableIfNotNull(internals->GyroY, this->RawInformation, "gyro_y");
  AddArrayToTableIfNotNull(internals->GyroZ, this->RawInformation, "gyro_z");
  AddArrayToTableIfNotNull(internals->AccelX, this->RawInformation, "accel_x");
  AddArrayToTableIfNotNull(internals->AccelY, this->RawInformation, "accel_y");
  AddArrayToTableIfNotNull(internals->AccelZ, this->RawInformation, "accel_z");
  AddArrayToTableIfNotNull(internals->Timestamp, this->RawInformation, "timestamp");
}

//----------------------------------------------------------------------------
bool vtkLivoxPosePacketInterpreter::IsValidPacket(unsigned char const* packetData,
  unsigned int dataLength)
{
  const LivoxSDK2Packet* dataPacket = reinterpret_cast<const LivoxSDK2Packet*>(packetData);

  if (!dataPacket)
  {
    return false;
  }

  if (dataPacket->length != dataLength)
  {
    return false;
  }
  return dataPacket->data_type == livox_sdk2::kLivoxLidarImuData;
}

//----------------------------------------------------------------------------
void vtkLivoxPosePacketInterpreter::ProcessPacket(unsigned char const* packetData,
  unsigned int vtkNotUsed(dataLength))
{
  const LivoxSDK2Packet* pkt = reinterpret_cast<const LivoxSDK2Packet*>(packetData);

  const auto* data = reinterpret_cast<const livox_sdk2::LivoxLidarImuRawPoint*>(pkt->data);
  auto& internals = this->Internals;
  for (uint32_t i = 0; i < pkt->dot_num; i++)
  {
    uint64_t ptsTimestamp = GetEthPacketTimestamp(pkt->timestamp, 8) + i * pkt->time_interval;
    internals->GyroX->InsertNextValue(data[i].gyro_x);
    internals->GyroY->InsertNextValue(data[i].gyro_y);
    internals->GyroZ->InsertNextValue(data[i].gyro_z);
    internals->AccelX->InsertNextValue(data[i].acc_x);
    internals->AccelY->InsertNextValue(data[i].acc_y);
    internals->AccelZ->InsertNextValue(data[i].acc_z);
    internals->Timestamp->InsertNextValue(ptsTimestamp);
  }
}

void vtkLivoxPosePacketInterpreter::FillInterpolatorFromPose() {}
