/*=========================================================================

  Program:   LidarView
  Module:    vtkLivoxPacketInterpreter.cxx

  Copyright (c) Kitware Inc.
  All rights reserved.
  See LICENSE or http://www.apache.org/licenses/LICENSE-2.0 for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkLivoxPacketInterpreter.h"

#include "InterpreterHelper.h"
#include "vtk_livox_sdk2.h"

#include <vtkDoubleArray.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkTransform.h>

#include <vtkUnsignedShortArray.h>

#include <cmath>

namespace livox_sdk2 = livox::sdk2;

namespace
{
typedef livox_sdk2::LivoxLidarEthernetPacket LivoxSDK2Packet;

constexpr const char* MID360_TAG_NAMES[3] = { "tag_glue", "tag_particles", "tag_other" };
constexpr const char* HAP_TAG_NAMES[3] = { "spatial_confidence",
  "intensity_confidence",
  "return_number" };

typedef union
{
  uint8_t stamp_bytes[8];
  int64_t stamp;
} LdsStamp;

//-----------------------------------------------------------------------------
uint64_t GetEthPacketTimestamp(const uint8_t* time_stamp, uint8_t size)
{
  LdsStamp time;
  memcpy(time.stamp_bytes, time_stamp, size);
  return time.stamp;
}

const uint64_t kRatioOfMsToNs = 1000000U;

//-----------------------------------------------------------------------------
std::string GetLidarModelName(livox_sdk2::LivoxLidarDeviceType model)
{
  switch (model)
  {
    case livox_sdk2::kLivoxLidarTypeMid360:
      return "Mid-360";
    case livox_sdk2::kLivoxLidarTypeIndustrialHAP:
      return "HAP";
    default:
      return "Unsupported Model";
  }
}
}

//-----------------------------------------------------------------------------
class vtkLivoxPacketInterpreter::vtkInternals
{
public:
  //-----------------------------------------------------------------------------
  bool DoSplitPacket(const LivoxSDK2Packet* pkt, uint64_t interval)
  {
    uint64_t timestamp = GetEthPacketTimestamp(pkt->timestamp, 8) / kRatioOfMsToNs;
    if (this->IsFirstTime || this->LastTimeStamp > timestamp ||
      timestamp - this->LastTimeStamp > interval * 2U)
    {
      this->IsFirstTime = false;
      this->LastTimeStamp = timestamp;
      return false;
    }

    if (timestamp - this->LastTimeStamp < interval)
    {
      return false;
    }
    this->LastTimeStamp = timestamp;
    return true;
  }

  //-----------------------------------------------------------------------------
  template <typename T>
  void FillPoints(double* pos, const T& point, double timestamp, uint8_t laserId)
  {
    if (std::isnan(pos[0]) || std::isnan(pos[1]) || std::isnan(pos[2]))
    {
      return;
    }

    double distance =
      std::sqrt(std::pow(0 - pos[0], 2) + std::pow(0 - pos[1], 2) + std::pow(0 - pos[2], 2));

    // Skip wrong points
    if (distance <= 0.1 || distance > 500.0)
    {
      return;
    }

    this->Points->InsertNextPoint(pos);
    InsertNextValueIfNotNull(this->PointsX, pos[0]);
    InsertNextValueIfNotNull(this->PointsY, pos[1]);
    InsertNextValueIfNotNull(this->PointsZ, pos[2]);
    InsertNextValueIfNotNull(this->Intensity, point.reflectivity);
    InsertNextValueIfNotNull(this->LaserId, laserId);
    InsertNextValueIfNotNull(this->Timestamp, timestamp);
    InsertNextValueIfNotNull(this->Distance, distance);
    InsertNextValueIfNotNull(this->FirstTag, !!(point.tag & 0x00) + !!(point.tag & 0x01));
    InsertNextValueIfNotNull(this->SecondTag, !!(point.tag & 0x02) + !!(point.tag & 0x03));
    InsertNextValueIfNotNull(this->ThirdTag, !!(point.tag & 0x04) + !!(point.tag & 0x05));
  }

  //-----------------------------------------------------------------------------
  void FillHighRawPoints(const LivoxSDK2Packet* pkt)
  {
    const auto* points =
      reinterpret_cast<const livox_sdk2::LivoxLidarCartesianHighRawPoint*>(pkt->data);

    for (uint32_t i = 0; i < pkt->dot_num; i++)
    {
      // import unit mm
      double pos[3] = { static_cast<double>(points[i].x) / 1000,
        static_cast<double>(points[i].y) / 1000,
        static_cast<double>(points[i].z) / 1000 };
      uint64_t ptsTimestamp = GetEthPacketTimestamp(pkt->timestamp, 8) + i * pkt->time_interval;
      uint8_t laserId = this->GetLaserId(i);
      this->FillPoints(pos, points[i], ptsTimestamp, laserId);
    }
  }

  //-----------------------------------------------------------------------------
  void FillLowRawPoints(const LivoxSDK2Packet* pkt)
  {
    const auto* points =
      reinterpret_cast<const livox_sdk2::LivoxLidarCartesianLowRawPoint*>(pkt->data);

    for (uint32_t i = 0; i < pkt->dot_num; i++)
    {
      // import unit cm
      double pos[3] = { static_cast<double>(points[i].x) / 100,
        static_cast<double>(points[i].y) / 100,
        static_cast<double>(points[i].z) / 100 };
      uint64_t ptsTimestamp = GetEthPacketTimestamp(pkt->timestamp, 8) + i * pkt->time_interval;
      uint8_t laserId = this->GetLaserId(i);
      this->FillPoints(pos, points[i], ptsTimestamp, laserId);
    }
  }

  //-----------------------------------------------------------------------------
  void FillSpherePoints(const LivoxSDK2Packet* pkt)
  {
    const auto* points = reinterpret_cast<const livox_sdk2::LivoxLidarSpherPoint*>(pkt->data);

    for (uint32_t i = 0; i < pkt->dot_num; i++)
    {
      double rho = static_cast<double>(points[i].depth) / 1000;  // import unit mm
      double theta = static_cast<double>(points[i].theta) / 100; // import unit 0.01 degree
      double phi = static_cast<double>(points[i].phi) / 100;     // import unit 0.01 degree

      double pos[3];
      pos[0] = rho * std::sin(phi) * std::cos(theta);
      pos[1] = rho * std::sin(phi) * std::sin(theta);
      pos[2] = rho * std::cos(phi);
      uint64_t ptsTimestamp = GetEthPacketTimestamp(pkt->timestamp, 8) + i * pkt->time_interval;
      uint8_t laserId = this->GetLaserId(i);
      this->FillPoints(pos, points[i], ptsTimestamp, laserId);
    }
  }

  //-----------------------------------------------------------------------------
  const char* GetTagNames(uint8_t tagIdx)
  {
    if (tagIdx >= 3)
    {
      return "";
    }
    switch (this->LidarModel)
    {
      case livox_sdk2::kLivoxLidarTypeHAP:
      case livox_sdk2::kLivoxLidarTypeIndustrialHAP:
        return ::HAP_TAG_NAMES[tagIdx];

      case livox_sdk2::kLivoxLidarTypeMid360:
      default:
        return ::MID360_TAG_NAMES[tagIdx];
    }
  }

private:
  //-----------------------------------------------------------------------------
  uint8_t GetLaserId(uint32_t pointIdx)
  {
    switch (this->LidarModel)
    {
      case livox_sdk2::kLivoxLidarTypeHAP:
      case livox_sdk2::kLivoxLidarTypeIndustrialHAP:
        return pointIdx % 6;

      case livox_sdk2::kLivoxLidarTypeMid360:
      default:
        return pointIdx % 4;
    }
  }

public:
  bool IsNewFrame = true;
  bool IsFirstTime = true;
  uint64_t LastTimeStamp;

  livox_sdk2::LivoxLidarDeviceType LidarModel = livox_sdk2::kLivoxLidarTypeMid360;

  vtkSmartPointer<vtkPoints> Points;
  vtkSmartPointer<vtkDoubleArray> PointsX;
  vtkSmartPointer<vtkDoubleArray> PointsY;
  vtkSmartPointer<vtkDoubleArray> PointsZ;
  vtkSmartPointer<vtkUnsignedCharArray> Intensity;
  vtkSmartPointer<vtkUnsignedCharArray> LaserId;
  vtkSmartPointer<vtkDoubleArray> Distance;
  vtkSmartPointer<vtkDoubleArray> Timestamp;
  vtkSmartPointer<vtkUnsignedShortArray> FirstTag;
  vtkSmartPointer<vtkUnsignedShortArray> SecondTag;
  vtkSmartPointer<vtkUnsignedShortArray> ThirdTag;
};

//-----------------------------------------------------------------------------
vtkStandardNewMacro(vtkLivoxPacketInterpreter)

//-----------------------------------------------------------------------------
vtkLivoxPacketInterpreter::vtkLivoxPacketInterpreter()
  : Internals(new vtkLivoxPacketInterpreter::vtkInternals())
{
  this->ResetCurrentFrame();
  this->SetSensorVendor("Livox");
  this->SetSensorModelName(GetLidarModelName(this->Internals->LidarModel));
}

//-----------------------------------------------------------------------------
bool vtkLivoxPacketInterpreter::IsLidarPacket(unsigned char const* data, unsigned int dataLength)
{
  const LivoxSDK2Packet* dataPacket = reinterpret_cast<const LivoxSDK2Packet*>(data);

  if (!dataPacket)
  {
    return false;
  }

  if (dataPacket->length != dataLength)
  {
    return false;
  }
  if (dataPacket->data_type == livox_sdk2::kLivoxLidarImuData)
  {
    return false;
  }
  return true;
}

//-----------------------------------------------------------------------------
vtkSmartPointer<vtkPolyData> vtkLivoxPacketInterpreter::CreateNewEmptyFrame(vtkIdType nbrOfPoints,
  vtkIdType prereservedNbrOfPoints)
{
  auto& internals = this->Internals;
  const int defaultPrereservedNbrOfPointsPerFrame = 60000;
  // prereserve for 50% points more than actually received in previous frame
  prereservedNbrOfPoints =
    std::max(static_cast<int>(prereservedNbrOfPoints * 1.5), defaultPrereservedNbrOfPointsPerFrame);

  vtkSmartPointer<vtkPolyData> polyData = vtkSmartPointer<vtkPolyData>::New();

  vtkNew<vtkPoints> points;
  points->SetDataTypeToFloat();
  points->Allocate(prereservedNbrOfPoints);
  if (nbrOfPoints > 0)
  {
    points->SetNumberOfPoints(nbrOfPoints);
  }
  points->GetData()->SetName("Points");

  polyData->SetPoints(points.GetPointer());

  internals->Points = points.GetPointer();
  // clang-format off
  InitArrayForPolyData(true, internals->PointsX, "X", nbrOfPoints, prereservedNbrOfPoints, polyData, this->EnableAdvancedArrays);
  InitArrayForPolyData(true, internals->PointsY, "Y", nbrOfPoints, prereservedNbrOfPoints, polyData, this->EnableAdvancedArrays);
  InitArrayForPolyData(true, internals->PointsZ, "Z", nbrOfPoints, prereservedNbrOfPoints, polyData, this->EnableAdvancedArrays);
  InitArrayForPolyData(false, internals->Intensity, "intensity", nbrOfPoints, prereservedNbrOfPoints, polyData);
  InitArrayForPolyData(false, internals->Timestamp, "timestamp", nbrOfPoints, prereservedNbrOfPoints, polyData);
  InitArrayForPolyData(false, internals->LaserId, "laser_id", nbrOfPoints, prereservedNbrOfPoints, polyData);
  InitArrayForPolyData(false, internals->Distance, "distance_m", nbrOfPoints, prereservedNbrOfPoints, polyData);
  InitArrayForPolyData(true, internals->FirstTag, internals->GetTagNames(0), nbrOfPoints, prereservedNbrOfPoints, polyData, this->EnableAdvancedArrays);
  InitArrayForPolyData(true, internals->SecondTag, internals->GetTagNames(1), nbrOfPoints, prereservedNbrOfPoints, polyData, this->EnableAdvancedArrays);
  InitArrayForPolyData(true, internals->ThirdTag, internals->GetTagNames(2), nbrOfPoints, prereservedNbrOfPoints, polyData, this->EnableAdvancedArrays);
  // clang-format on

  // Set the default array to display in the application
  polyData->GetPointData()->SetActiveScalars("intensity");
  return polyData;
}

//-----------------------------------------------------------------------------
bool vtkLivoxPacketInterpreter::PreProcessPacket(unsigned char const* data,
  unsigned int vtkNotUsed(dataLength),
  fpos_t filePosition,
  double packetNetworkTime,
  std::vector<FrameInformation>* frameCatalog)
{
  const LivoxSDK2Packet* pkt = reinterpret_cast<const LivoxSDK2Packet*>(data);

  auto& internals = this->Internals;
  if (internals->IsNewFrame)
  {
    uint64_t timestamp = GetEthPacketTimestamp(pkt->timestamp, 8) / kRatioOfMsToNs;

    internals->IsNewFrame = false;
    this->ParserMetaData.FilePosition = filePosition;
    this->ParserMetaData.FirstPacketDataTime = timestamp / 1000.;
    this->ParserMetaData.FirstPacketNetworkTime = packetNetworkTime;
  }

  if (internals->DoSplitPacket(pkt, this->PublishInterval) && frameCatalog)
  {
    internals->IsNewFrame = true;
    frameCatalog->push_back(this->ParserMetaData);
  }
  return internals->IsNewFrame;
}

//-----------------------------------------------------------------------------
void vtkLivoxPacketInterpreter::ProcessPacket(unsigned char const* data,
  unsigned int vtkNotUsed(dataLength))
{
  // Update the transforms here and then call internal transform
  // if (this->SensorTransform)
  // {
  //   this->SensorTransform->Update();
  // }

  auto& internals = this->Internals;
  const LivoxSDK2Packet* pkt = reinterpret_cast<const LivoxSDK2Packet*>(data);

  switch (pkt->data_type)
  {
    case livox_sdk2::kLivoxLidarCartesianCoordinateHighData:
      internals->FillHighRawPoints(pkt);
      break;

    case livox_sdk2::kLivoxLidarCartesianCoordinateLowData:
      internals->FillLowRawPoints(pkt);
      break;

    case livox_sdk2::kLivoxLidarSphericalCoordinateData:
      internals->FillSpherePoints(pkt);
      break;

    default:
      vtkWarningMacro(<< "Could not handle this data type.");
      break;
  }

  // bool splitPacket = pkt->udp_cnt % 150 == 0;
  if (internals->DoSplitPacket(pkt, this->PublishInterval))
  {
    this->SplitFrame();
  }
}

//-----------------------------------------------------------------------------
void vtkLivoxPacketInterpreter::SetLidarModel(int model)
{
  if (static_cast<int>(this->Internals->LidarModel) == model)
  {
    return;
  }

  auto deviceType = static_cast<livox_sdk2::LivoxLidarDeviceType>(model);
  if (deviceType == livox_sdk2::kLivoxLidarTypeMid360 ||
    deviceType == livox_sdk2::kLivoxLidarTypeHAP ||
    deviceType == livox_sdk2::kLivoxLidarTypeIndustrialHAP)
  {
    this->Internals->LidarModel = deviceType;
    this->SetSensorModelName(GetLidarModelName(deviceType));
    this->Modified();
  }
}

//-----------------------------------------------------------------------------
int vtkLivoxPacketInterpreter::GetLidarModel()
{
  return static_cast<int>(this->Internals->LidarModel);
}

//-----------------------------------------------------------------------------
void vtkLivoxPacketInterpreter::LoadCalibration()
{
  this->IsCalibrated = true;
  this->Internals->IsFirstTime = true;
}

//----------------------------------------------------------------------------
void vtkLivoxPacketInterpreter::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}
