include(lidarview-version)
include(paraview-version)

# Set Packaging Absolute Origin/Destination paths
set(library_paths   "${superbuild_install_location}/${LV_INSTALL_LIBRARY_DIR}")
set(executable_path "${superbuild_install_location}/${LV_INSTALL_RUNTIME_DIR}")
set(share_path      "${superbuild_install_location}/${LV_INSTALL_RESOURCE_DIR}")

set(lidarview_executables
  "${SOFTWARE_NAME}")
if (python3_enabled)
  list(APPEND lidarview_executables
    pvpython)
endif ()

# lidarview is only build shared, see paraview.bundle.common if we need to support static build
set(python_modules
  paraview
  vtk
  vtkmodules
  lidarview
  lidarviewcore
  LidarPlugin) # Comes from VTK or `pythonmpi4py`

macro (check_for_python_module project module)
  if (${project}_built_by_superbuild)
    list(APPEND python_modules
      "${module}")
  endif ()
endmacro ()

check_for_python_module(numpy numpy)

# Get LV & PV Plugins
function (lidarview_detect_plugins plugins_dir output)
    set(plugins_list)
    file(GLOB subpaths LIST_DIRECTORIES TRUE RELATIVE ${plugins_dir} ${plugins_dir}/*)
    foreach(subpath ${subpaths})
      if(IS_DIRECTORY ${plugins_dir}/${subpath})
        LIST(APPEND plugins_list ${subpath})
      endif()
    endforeach()
    set( ${output} ${plugins_list} PARENT_SCOPE)
endfunction ()

# Get Lidarview Plugins Name from directories 
set(lidarview_plugins)
lidarview_detect_plugins( "${superbuild_install_location}/${LV_PLUGIN_INSTALL_SUBDIRECTORY}" lidarview_plugins)
if(NOT lidarview_plugins)
  message(FATAL_ERROR "lidarview_plugins is empty")
endif()
list(REMOVE_ITEM lidarview_plugins "PythonQtPlugin") #WIP while it is still a static library, we have to discard it

# Get Paraview Plugins Name from directories 
set(paraview_plugins)
lidarview_detect_plugins( "${superbuild_install_location}/${LV_INSTALL_PV_PLUGIN_SUBDIR}" paraview_plugins)

# Set include/exclude regexes # can be used for optional packaging things or fixing system linking bugs
# Discarding anything from `/build/lvsb/pvsb/superbuild/` fixes fixup-bundle mistakes
set(include_regexes)
set(exclude_regexes ".*/build/lvsb/pvsb/superbuild/.*")

# Set the license files.
set(qt_license_file "${CMAKE_CURRENT_LIST_DIR}/../pvsb/projects/files/Qt5.LICENSE.LGPLv3")

function (lidarview_install_license project)
  if (EXISTS "${superbuild_install_location}/share/licenses/${project}")
    install(
      DIRECTORY   "${superbuild_install_location}/share/licenses/${project}"
      DESTINATION "${lidarview_license_path}"
      COMPONENT   superbuild)
  else ()
    message(FATAL_ERROR "${superbuild_install_location}/share/licenses/${project} does not exist, aborting.")
  endif ()
endfunction ()

function (lidarview_install_all_licenses)
  set(license_projects "${enabled_projects}")

  foreach (project IN LISTS license_projects)
    if (NOT ${project}_built_by_superbuild)
      list(REMOVE_ITEM license_projects ${project})
    endif ()
  endforeach ()

  # paraview install itself in ParaView directory
  if (paraview IN_LIST license_projects)
    list(REMOVE_ITEM license_projects paraview)
    list(APPEND license_projects ParaView)
  endif ()

  foreach (project IN LISTS license_projects)
    lidarview_install_license("${project}")
  endforeach ()

  # When packaging system qt, install the license manually
  if (qt5_plugin_paths)
    install(
      FILES       "${qt_license_file}"
      DESTINATION "${lidarview_license_path}/qt5"
      COMPONENT   superbuild)
  endif ()
endfunction ()

function (lidarview_install_extra_data)
  lidarview_install_all_licenses()
endfunction ()

# Ship Qt5 plugins
if (Qt5_DIR)
  if (WIN32)
  list(APPEND library_paths
    "${Qt5_DIR}/../../../bin")
  elseif (APPLE)
    list(APPEND library_paths
    "${Qt5_DIR}/../..")
  elseif (UNIX)
    list(APPEND library_paths
    "${Qt5_DIR}/../..")
  endif ()
endif ()

if (qt5_enabled)
  include(qt5.functions)

  set(qt5_plugin_prefix)
  if (NOT WIN32)
    set(qt5_plugin_prefix "lib")
  endif ()

  # Add SVG support, so ParaView can use SVG icons
  set(qt5_plugins
    iconengines/${qt5_plugin_prefix}qsvgicon
    imageformats/${qt5_plugin_prefix}qsvg
    imageformats/${qt5_plugin_prefix}qjpeg
    sqldrivers/${qt5_plugin_prefix}qsqlite)

  if (WIN32)
    list(APPEND qt5_plugins
      platforms/qwindows)

    if (NOT qt5_version VERSION_LESS "5.10")
      list(APPEND qt5_plugins
        styles/qwindowsvistastyle)
    endif ()
  elseif (APPLE)
    list(APPEND qt5_plugins
      platforms/libqcocoa
      printsupport/libcocoaprintersupport)

    if (NOT qt5_version VERSION_LESS "5.10")
      list(APPEND qt5_plugins
        styles/libqmacstyle)
    endif ()
  elseif (UNIX)
    list(APPEND qt5_plugins
      platforms/libqxcb
      platforminputcontexts/libcomposeplatforminputcontextplugin
      xcbglintegrations/libqxcb-glx-integration)
  endif ()

  superbuild_get_qt5_plugin_install_paths(qt5_plugin_paths ${qt5_plugins})
else ()
  set(qt5_plugin_paths)
endif ()
