/* Distributed under the Apache License, Version 2.0.
   See accompanying NOTICE file for details.*/

#include "LBM.h"
#include <cmath>
#include <limits>
#include <iostream>
#include "dirent.h"
#include "FileUtils.h"

double percentTolerance(double expected, double calculated, double epsilon = 1E-20)
{
  // Check for 'invalid' numbers first
  //if (Double.isNaN(expected) || Double.isNaN(calculated) || Double.isInfinite(expected) || Double.isInfinite(calculated))
  //  Log.warn("While finding percent tolerance from values 'expected' = " + expected + " and 'calculated' =" + calculated +
  //    ", invalid values (NaN or Infinity) were found.  Unexpected results may occur.");
  // Handle special cases
  if (expected == 0.0 && calculated == 0.0)
    return 0.0;
  else if (expected == 0.0 || calculated == 0.0)
  {
    if (std::fabs(expected + calculated) < epsilon)
      return 0.0;
    else
    {
      if (expected == 0.0)
        return std::numeric_limits<double>::infinity();
      else if (expected < 0.0)
        return -100.0;
      else
        return 100.0;
    }
  }
  else
    return std::fabs(calculated - expected) / expected * 100.0;
}

double percentDifference(double expected, double calculated, double epsilon = 1E-20)
{
  // Check for 'invalid' numbers first
  //if (Double.isNaN(expected) || Double.isNaN(calculated) || Double.isInfinite(expected) || Double.isInfinite(calculated))
  //  Log.warn("While finding percent difference from values 'expected' = " + expected + " and 'calculated' =" + calculated +
  //    ", invalid values (NaN or Infinity) were found.  Unexpected results may occur.");
  // Handle special cases
  if (expected == 0.0 && calculated == 0.0)
    return 0.0;
  else if (expected == 0.0 || calculated == 0.0)
  {
    if (std::fabs(expected + calculated) < epsilon)
      return 0.0;
    else
      return 200.0;
  }
  else
  {
    double difference = (calculated - expected);
    double average = (calculated + expected) / 2.0;

    if (average == 0.0)
      return std::numeric_limits<double>::infinity();

    return std::fabs(difference / average) * 100.0;
  }
}

bool LBM::CompareDataSet(LBM const& baseline, LBM const& computed, bool vtr_data_only)
{
  double diff;
  bool same = true;
  double percrent_difference = 2.0;

  ////////////////////
  // COMPARE INPUTS //
  ////////////////////

  // Compare the Points
  if (baseline.in.dimensions[0] != computed.in.dimensions[0])
  {
    std::cout << "Number of X dimension points does not match\n";
    same = false;
  }
  if (baseline.in.dimensions[1] != computed.in.dimensions[1])
  {
    std::cout << "Number of Y dimension points does not match\n";
    same = false;
  }
  if (baseline.in.dimensions[2] != computed.in.dimensions[2])
  {
    std::cout << "Number of Z dimension points does not match\n";
    same = false;
  }
  if (!vtr_data_only)
  {
    // TODO Percent compare?
    if (baseline.in.grid_spacing != computed.in.grid_spacing)
    {
      std::cout << "Grid spacing does not match\n";
      same = false;
    }
    if (baseline.in.labels.size() != computed.in.labels.size())
    {
      std::cout << "Input label size does not match\n";
      same = false;
    }
    else
    {
      // Check input labels
      for (unsigned int i = 0; i < baseline.in.labels.size(); i++)
      {
        if (baseline.in.labels[i] != computed.in.labels[i])
        {
          std::cout << "in.labels[" << i << "] does not match.\n";
          same = false;
        }
      }
    }
  }

  /////////////////////
  // COMPARE OUTPUTS //
  /////////////////////

  // Cell Counts
  if (baseline.out.num_cells != computed.out.num_cells)
  {
    std::cout << "Number of cells does not match\n";
    same = false;
  }
  if (baseline.out.num_wall_cells != computed.out.num_wall_cells)
  {
    std::cout << "Number of wall cells does not match\n";
    same = false;
  }
  if (baseline.out.num_inflow_cells != computed.out.num_inflow_cells)
  {
    std::cout << "Number of inflow cells does not match\n";
    same = false;
  }
  if (baseline.out.num_outflow_cells != computed.out.num_outflow_cells)
  {
    std::cout << "Number of outflow cells does not match\n";
    same = false;
  }
  if (baseline.out.num_inactive_cells != computed.out.num_inactive_cells)
  {
    std::cout << "Number of inactive cells does not match\n";
    same = false;
  }
  if (baseline.out.num_interior_cells != computed.out.num_interior_cells)
  {
    std::cout << "Number of interior cells does not match\n";
    same = false;
  }
  if (baseline.out.num_unknown_cells != computed.out.num_unknown_cells)
  {
    std::cout << "Number of unknown cells does not match\n";
    same = false;
  }

  // Scalar Outputs
  if (!vtr_data_only)
  {
    diff = percentDifference(baseline.out.reference_velocity,
      computed.out.reference_velocity);
    if (diff > percrent_difference)
    {
      std::cout << "out.reference_velocity does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.inlet_velocity,
      computed.out.inlet_velocity);
    if (diff > percrent_difference)
    {
      std::cout << "out.inlet_velocity does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.inlet_area,
      computed.out.inlet_area);
    if (diff > percrent_difference)
    {
      std::cout << "out.inlet_area does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.outlet_velocity,
      computed.out.outlet_velocity);
    if (diff > percrent_difference)
    {
      std::cout << "out.outlet_velocity does not match, the error is "<<diff<<"%\n";
      same = false;
    }

    diff = percentDifference(baseline.out.outlet_area,
      computed.out.outlet_area);
    if (diff > percrent_difference)
    {
      std::cout << "out.outlet_area does not match, the error is "<<diff<<"%\n";
      same = false;
    }

    diff = percentDifference(baseline.out.outlet_diameter,
      computed.out.outlet_diameter);
    if (diff > percrent_difference)
    {
      std::cout << "out.outlet_diameter does not match, the error is "<<diff<<"%\n";
      same = false;
    }

    diff = percentDifference(baseline.out.average_area,
      computed.out.average_area);
    if (diff > percrent_difference)
    {
      std::cout << "out.average_area does not match, the error is "<<diff<<"%\n";
      same = false;
    }

    diff = percentDifference(baseline.out.mass_flow,
      computed.out.mass_flow);
    if (diff > percrent_difference)
    {
      std::cout << "out.mass_flow does not match, the error is "<<diff<<"%\n";
      same = false;
    }

    diff = percentDifference(baseline.out.pressure_drop,
      computed.out.pressure_drop);
    if (diff > percrent_difference)
    {
      std::cout << "out.pressure_drop does not match, the error is "<<diff<<"%\n";
      same = false;
    }

    diff = percentDifference(baseline.out.volumetric_flow_rate,
      computed.out.volumetric_flow_rate);
    if (diff > percrent_difference)
    {
      std::cout << "out.volumetric_flow_rate does not match, the error is "<<diff<<"%\n";
      same = false;
    }
  }

  // Array Lengths
  if (baseline.out.Q_num_cell_values != computed.out.Q_num_cell_values)
  {
    std::cout << "Number of Q cell valuess does not match\n";
    same = false;
  }
  if (baseline.out.T_num_cell_values != computed.out.T_num_cell_values)
  {
    std::cout << "Number of T cell values does not match\n";
    same = false;
  }
  if (baseline.out.num_boundary_values != computed.out.num_boundary_values)
  {
    std::cout << "Number of boundary_values does not match\n";
    same = false;
  }
  if (baseline.out.nPDF != computed.out.nPDF)
  {
    std::cout << "Number of nPDF does not match\n";
    same = false;
  }

  int num_cells = baseline.out.num_cells;
  // Arrays
  for (int i = 0; i < num_cells; i++)
  {
    if (baseline.out.labels[i] != computed.out.labels[i])
    {
      std::cout << "out.labels[" << i << "] does not match\n";
      same = false;
    }
    diff = percentDifference(baseline.out.pressure_flows_and_stress[i],
                              computed.out.pressure_flows_and_stress[i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.pressure[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.pressure_flows_and_stress[num_cells+i],
                              computed.out.pressure_flows_and_stress[num_cells+i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.X_flow[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.pressure_flows_and_stress[num_cells*2+i],
                              computed.out.pressure_flows_and_stress[num_cells*2+i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.Y_flow[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
      
    diff = percentDifference(baseline.out.pressure_flows_and_stress[num_cells*3+i],
                              computed.out.pressure_flows_and_stress[num_cells*3+i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.Z_flow[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.pressure_flows_and_stress[num_cells*4+i],
                              computed.out.pressure_flows_and_stress[num_cells*4+i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.XY_stress[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.pressure_flows_and_stress[num_cells*5+i],
                              computed.out.pressure_flows_and_stress[num_cells*5+i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.XZ_stress[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.pressure_flows_and_stress[num_cells*6+i],
                              computed.out.pressure_flows_and_stress[num_cells*6+i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.YZ_stress[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
    diff = percentDifference(baseline.out.temperature[i],
                              computed.out.temperature[i]);
    if (diff > percrent_difference)
    {
      std::cout << "out.temperature[" << i << "] does not match, the error is "<<diff<<"%\n";
      same = false;
    }
  }
  if (!vtr_data_only)
  {
    for (int i = 0; i < baseline.out.nPDF; i++)
    {
      diff = percentDifference(baseline.out.fPDF[i],
                               computed.out.fPDF[i]);
      if (diff > percrent_difference)
      {
        std::cout << "out.fPDF[" << i << "] does not match, the error is "<<diff<<"%\n";
        same = false;
      }
      diff = percentDifference(baseline.out.fnPDF[i],
                               computed.out.fnPDF[i]);
      if (diff > percrent_difference)
      {
        std::cout << "out.fnPDF[" << i << "] does not match, the error is "<<diff<<"%\n";
        same = false;
      }
    }
    for (int i = 0; i < baseline.out.num_boundary_values; i++)
    {
      diff = percentDifference(baseline.out.boundary_conditions[i],
                               computed.out.boundary_conditions[i]);
      if (diff > percrent_difference)
      {
        std::cout << "out.boundary_conditions[" << i << "] does not match, the error is "<<diff<<"%\n";
        same = false;
      }
    }
  }
  std::cout << "Finished comparison\n";
  return same;
}
