/* Distributed under the Apache License, Version 2.0.
   See accompanying NOTICE file for details.*/

#include "LBMVTK.h"

#include <vtkDataSet.h>
#include <vtkCellData.h>
#include <vtkCellTypes.h>
#include <vtkDataSetReader.h>
#include <vtkFieldData.h>
#include <vtkFloatArray.h>
#include <vtkIntArray.h>
#include <vtkPointData.h>
#include <vtkSmartPointer.h>
#include <vtkXMLReader.h>
#include <vtkXMLRectilinearGridReader.h>
#include <vtkXMLRectilinearGridWriter.h>
#include <vtksys/SystemTools.hxx>

vtkSmartPointer<vtkRectilinearGrid> LBMVTK::ToVTR(LBM const& lbm)
{
  // Create the point lattice
  vtkSmartPointer<vtkRectilinearGrid> grid =
    vtkSmartPointer<vtkRectilinearGrid>::New();

  grid->SetDimensions(lbm.in.dimensions[0] + 1, lbm.in.dimensions[1] + 1, lbm.in.dimensions[2] + 1);
  vtkNew<vtkFloatArray> xCoords;
  xCoords->SetName("X");
  for (int i = 0; i < lbm.in.dimensions[0] + 1; i++)
  {
    xCoords->InsertNextValue(i * lbm.in.grid_spacing);
  }
  vtkNew<vtkFloatArray> yCoords;
  yCoords->SetName("Y");
  for (int i = 0; i < lbm.in.dimensions[1] + 1; i++)
  {
    yCoords->InsertNextValue(i * lbm.in.grid_spacing);
  }
  vtkNew<vtkFloatArray> zCoords;
  zCoords->SetName("Z");
  for (int i = 0; i < lbm.in.dimensions[2] + 1; i++)
  {
    zCoords->InsertNextValue(i * lbm.in.grid_spacing);
  }
  grid->SetXCoordinates(xCoords);
  grid->SetYCoordinates(yCoords);
  grid->SetZCoordinates(zCoords);

  if (!lbm.in.source_labels.empty())
  {
    vtkNew<vtkIntArray> sbl;
    sbl->SetName(SOURCE_LABELS);
    for (int i : lbm.in.source_labels)
    {
      sbl->InsertNextValue(i);
    }
    grid->GetCellData()->AddArray(sbl);
  }

  if (lbm.out.labels != nullptr)
  {
    vtkNew<vtkIntArray> bl;
    bl->SetName(BOUNDARY_LABELS);
    for (int i = 0; i < lbm.out.num_cells; i++)
    {
      bl->InsertNextValue(lbm.out.labels[i]);
    }
    grid->GetCellData()->AddArray(bl);
  }

  if (lbm.out.pressure_flows_and_stress != nullptr)
  {
    vtkNew<vtkFloatArray>p;
    p->SetName(PRESSURE);
    grid->GetCellData()->AddArray(p);
    vtkNew<vtkFloatArray>vx;
    vx->SetName(X_FLOW);
    grid->GetCellData()->AddArray(vx);
    vtkNew<vtkFloatArray>vy;
    vy->SetName(Y_FLOW);
    grid->GetCellData()->AddArray(vy);
    vtkNew<vtkFloatArray>vz;
    vz->SetName(Z_FLOW);
    grid->GetCellData()->AddArray(vz);


    vtkNew<vtkFloatArray>Sxy;
    vtkNew<vtkFloatArray>Sxz;
    vtkNew<vtkFloatArray>Syz;
    vtkNew<vtkFloatArray>Sxx;
    vtkNew<vtkFloatArray>Syy;
    vtkNew<vtkFloatArray>Szz;
    if (lbm.cfg.expanded_results)
    {
      Sxy->SetName(XY_STRESS);
      grid->GetCellData()->AddArray(Sxy);
      Sxz->SetName(XZ_STRESS);
      grid->GetCellData()->AddArray(Sxz);
      Syz->SetName(YZ_STRESS);
      grid->GetCellData()->AddArray(Syz);
      Sxx->SetName(XX_STRESS);
      grid->GetCellData()->AddArray(Sxx);
      Syy->SetName(YY_STRESS);
      grid->GetCellData()->AddArray(Syy);
      Szz->SetName(ZZ_STRESS);
      grid->GetCellData()->AddArray(Szz);
    }

    for (int i = 0; i < lbm.out.num_cells; i++)
    {
      p->InsertNextValue(lbm.out.pressure_flows_and_stress[i]);
      vx->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells+i]);
      vy->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells*2+i]);
      vz->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells*3+i]);
      if (lbm.cfg.expanded_results)
      {
        Sxy->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells * 4 + i]);
        Sxz->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells * 5 + i]);
        Syz->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells * 6 + i]);
        Sxx->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells * 7 + i]);
        Syy->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells * 8 + i]);
        Szz->InsertNextValue(lbm.out.pressure_flows_and_stress[lbm.out.num_cells * 9 + i]);
      }
    }
  }

    if (lbm.out.wall_shear_stress != nullptr)
    {
        vtkNew<vtkFloatArray> stress;
        stress->SetName(STRESS);
        grid->GetCellData()->AddArray(stress);

        vtkNew<vtkFloatArray> legacy_stress;
        if (lbm.cfg.expanded_results)
        {
          legacy_stress->SetName(LEGACY_STRESS);
          grid->GetCellData()->AddArray(legacy_stress);
        }

        for (int i = 0; i < lbm.out.num_cells; i++)
        {
          if(lbm.cfg.expanded_results)
            legacy_stress->InsertNextValue(lbm.out.wall_shear_stress[i]);
          stress->InsertNextValue(lbm.out.wall_shear_stress[lbm.out.num_cells+i]);
        }
    }

  if (lbm.out.temperature != nullptr)
  {
    vtkNew<vtkFloatArray> t;
    t->SetName(TEMPERATURE);
    for (int i = 0; i < lbm.out.num_cells; i++)
    {
      t->InsertNextValue(lbm.out.temperature[i]);
    }
    grid->GetCellData()->AddArray(t);
  }

  return grid;
}
bool LBMVTK::WriteVTR(std::string const& filename, LBM const& lbm)
{
  vtkSmartPointer<vtkRectilinearGrid> grid = ToVTR(lbm);
  // Write file
  vtkSmartPointer<vtkXMLRectilinearGridWriter> writer =
  vtkSmartPointer<vtkXMLRectilinearGridWriter>::New();
  writer->SetFileName(filename.c_str());
  writer->SetInputData(grid);
  return writer->Write() == 1;
}

bool LBMVTK::ReadVTR(std::string const& filename, LBM& lbm)
{
  vtkSmartPointer<vtkXMLRectilinearGridReader> reader =
    vtkSmartPointer<vtkXMLRectilinearGridReader>::New();
  reader->SetFileName(filename.c_str());
  reader->Update();
  reader->GetOutput()->Register(reader);
  vtkSmartPointer<vtkRectilinearGrid> vtr = reader->GetOutput();

  return FromVTR(vtr, lbm);
}

bool LBMVTK::FromVTR(vtkSmartPointer<vtkRectilinearGrid> vtr, LBM& lbm)
{
  lbm.in.labels.clear(); // Not stored in the VTR
  lbm.in.dimensions[0] = vtr->GetDimensions()[0]-1;
  lbm.in.dimensions[1] = vtr->GetDimensions()[1]-1;
  lbm.in.dimensions[2] = vtr->GetDimensions()[2]-1;
  lbm.in.grid_spacing = 0; // Cannot assume same spacing in all dimensions from VTR

  lbm.out.Allocate(lbm.in.dimensions);
  int num_cells = lbm.out.num_cells;

  // Now check for cell data
  vtkCellData* cd = vtr->GetCellData();
  if (cd)
  {
    for (int a=0; a<cd->GetNumberOfArrays(); a++)
    {
      std::string name = cd->GetArrayName(a);
      vtkDataArray* array = cd->GetArray(name.c_str());
      if (name.compare(BOUNDARY_LABELS) == 0 || name.compare("iBC") == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.labels[i] = (int)(*array->GetTuple(i));
      }
      if (name.compare(PRESSURE) == 0 || name.compare("p") == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(X_FLOW) == 0 || name.compare("u") == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells+i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(Y_FLOW) == 0 || name.compare("v") == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells*2+i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(Z_FLOW) == 0 || name.compare("w") == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells*3+i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(TEMPERATURE) == 0 || name.compare("t") == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.temperature[i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(STRESS) == 0 )
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.wall_shear_stress[num_cells+i] = (float)(*array->GetTuple(i));
      }

      // Expanded Results (may not be available, but that is ok!)
      if (name.compare(LEGACY_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.wall_shear_stress[i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(XY_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells*4+i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(XZ_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells*5+i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(YZ_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells*6+i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(XX_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells * 4 + i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(YY_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells * 5 + i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(ZZ_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells * 6 + i] = (float)(*array->GetTuple(i));
      }
      if (name.compare(ZZ_STRESS) == 0)
      {
        for (int i = 0; i < array->GetSize(); i++)
          lbm.out.pressure_flows_and_stress[num_cells * 6 + i] = (float)(*array->GetTuple(i));
      }
    }
  }
  return lbm.out.CountCellTypes();
}
