/* Distributed under the Apache License, Version 2.0.
   See accompanying NOTICE file for details.*/

#pragma once

#include <map>
#include <string>
#include <vector>

#ifdef LBM_EXPORTS
  #if defined (__clang__)
    #define LBM_DECL
  #elif defined(__gnu_linux__)
    #define LBM_DECL __attribute__ ((visibility ("default")))
  #else
    #ifdef LBM_EXPORTS
      #define LBM_DECL __declspec(dllexport)
      #define LBM_EXP
   #else
      #define LBM_DECL __declspec(dllimport)
      #define LBM_EXP extern
    #endif
  #endif
#else
  #define LBM_DECL
  #define LBM_EXP
#endif


extern "C"
LBM_DECL bool cuda_device_check(int use_device = -1);

class LBM_DECL LBM
{
  friend class LBMVTK;
public:
  LBM();
  ~LBM();

  enum class BoundaryTypes {
    Inactive = -1,
    Wall = 0,
    Inlet = 100,
    Outlet = 200
  };

  enum class BoundaryCondition { Pressure = 0, Flow = 1 };
  enum class DownsampleType { Fraction = 0, Length = 1 };

  struct LBM_DECL Configuration
  {
    // You can provide any values of any unit
    // As long as they are all consistent
    // ie.e All lengths are in m, time in s
    //   - density in kg_per_m3
    //   - viscosity in m2_per_s
    //   - speed_of_sound in m_per_s
    //   - grid_spacing in m

    float density;
    float fluid_temperature;
    float reference_pressure;
    float speed_of_sound;
    float viscosity;
    float thermal_coefficient;
    float wall_temperature;
    BoundaryCondition inlet_boundary_condition;
    float inlet_value;
    BoundaryCondition outlet_boundary_condition;
    float outlet_value;
    DownsampleType downsample_type = DownsampleType::Fraction;
    double downsample_to_length = 0;
    double downsample_fraction = 0;
    bool expanded_results = false;
    bool setup_only = false;

    void ToString(std::ostream& str) const;
  };

  struct LBM_DECL Input
  {
    Input();
    ~Input() = default;

    float origin[3];
    int dimensions[3];
    float grid_spacing;
    // You have two options for providing labels to LBM
    // 1. A vector of LBM labels
    std::vector<BoundaryTypes> labels;
    // 2. Your own vector of labels, with a mapping to LBM
    //   - This is provided to store your labels with LBM
    //   - Source labels can be modified by some LBM preprocessors
    std::vector<int> source_labels;
    std::map<int, BoundaryTypes> source_to_lbm_label_map;
  };

  class LBM_DECL Output
  {
    friend class LBM;
    friend class LBMVTK;
  public:
    Output();
    ~Output();
    bool Allocate(int dimensions[3]);
    void Deallocate();
    bool CountCellTypes();

    int GetNumCells() const { return num_cells; }
    int GetNumWallCells() const { return num_wall_cells; }
    int GetNumInflowCells() const { return num_inflow_cells; }
    int GetNumOutflowCells() const { return num_outflow_cells; }
    int GetNumInactiveCells() const { return num_inactive_cells; }
    int GetNumInteriorCells() const { return num_interior_cells; }
    int GetNumUnknownCells() const { return num_unknown_cells; }

    float GetReferenceVelocity() const { return reference_velocity; }
    float GetInletVelocity() const { return inlet_velocity; }
    float GetOutletVelocity() const { return outlet_velocity; }
    float GetInletArea() const { return inlet_area; }
    float GetOutletArea() const { return outlet_area; }
    float GetOutletDiameter() const { return outlet_diameter; }
    float GetAverageArea() const { return average_area; }
    float GetMassFlow() const { return mass_flow; }
    float GetPressureDrop() const { return pressure_drop; }
    float GetVolumetricFlowRate() const { return volumetric_flow_rate; }

    const int*   GetLabels() const          { return labels; }
    const float* GetPressure() const        { return pressure_flows_and_stress; }
    const float* GetXFlow() const           { return &pressure_flows_and_stress[num_cells]; }
    const float* GetYFlow() const           { return &pressure_flows_and_stress[num_cells*2]; }
    const float* GetZFlow() const           { return &pressure_flows_and_stress[num_cells*3]; }
    const float* GetTemperature() const     { return temperature; }
    const float* GetWallShearStress() const { return &wall_shear_stress[num_cells]; }
    // Expanded results, may not be available
    const float* GetLegacyStress() const    { return wall_shear_stress; }
    const float* GetXYStress() const        { return &pressure_flows_and_stress[num_cells*4]; }
    const float* GetXZStress() const        { return &pressure_flows_and_stress[num_cells*5]; }
    const float* GetYZStress() const        { return &pressure_flows_and_stress[num_cells*6]; }
    const float* GetXXStress() const        { return &pressure_flows_and_stress[num_cells*7]; }
    const float* GetYYStress() const        { return &pressure_flows_and_stress[num_cells*8]; }
    const float* GetZZStress() const        { return &pressure_flows_and_stress[num_cells*9]; }

  protected:
    // Specify nr. of continuum components (p,ux,uy,uz) and viscous stress components (Sigma_xy, Sigma_xz, Sigma_yz)
    const int nQ = 4;
    const int nS = 6;
    // Specify nr. of spatial labels (exterior, interior, wall, inlet and outlet)
    const int nBound = 5;

    int num_cells = 0;
    // Count up label types
    int num_wall_cells = 0;
    int num_inflow_cells = 0;
    int num_outflow_cells = 0;
    int num_inactive_cells = 0;
    int num_interior_cells = 0;
    int num_unknown_cells = 0;

    float  reference_velocity = 0;
    float  inlet_velocity = 0;
    float  inlet_area = 0;
    float  outlet_velocity = 0;
    float  outlet_area = 0;
    float  outlet_diameter = 0;
    float  average_area = 0;
    float  mass_flow = 0;
    float  pressure_drop = 0;
    float  volumetric_flow_rate = 0;

    // Array Lengths
    int Q_num_cell_values = 0;   // Per q array passed into microLBM
    int T_num_cell_values = 0;   // Per q array passed into microLBM
    int num_boundary_values = 0;
    int nPDF = 0;

    int*   labels=nullptr;
    float* pressure_flows_and_stress=nullptr;
    float* wall_shear_stress=nullptr;
    float* temperature=nullptr;
    float* fPDF=nullptr;//host memory for PDFs
    float* fnPDF=nullptr;//host memory for non-equilibrium PDFs
    float* boundary_conditions=nullptr;
  };

  bool Run();
  Output const& GetOut() const { return out; }

  // Verification Methods
  static bool CompareDataSet(LBM const& baseline, LBM const& computed, bool vtr_data_only=false);

  // Support legacy input decks
  static bool LoadLegacyScenario(std::string const& config_file,
                                 std::string const& voxel_file,
                                 std::string const& grid_file,
                                 LBM& lbm);

  public:
    Configuration cfg;
    Input         in;
  protected:
    Output        out;
};

inline std::ostream& operator<< (std::ostream& out, const LBM::Configuration& cfg)
{
  cfg.ToString(out);
  return out;
}
